/*
   File: dcg_error.c
   Defines routines for diagnostic messages

   Copyright (C) 2008 Marc Seutter

   This library is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published
   by the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with this library.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: dcg_error.c,v 1.9 2008/06/28 13:41:17 marcs Exp $"
*/

/* standard includes */
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

/* local includes */
#include "dcg.h"
#include "dcg_error.h"

int debug;
int verbose;
int full_verbose;

static FILE* error_file;
static char *tool_name;
static int error_count;
static void (*my_onexit)();
void init_error (FILE *f, char *tname, void (*onexit)())
	{ verbose = 0;
	  full_verbose = 0;
	  error_file = f;
	  error_count = 0;
	  tool_name = tname;
	  my_onexit = onexit;
	};

#define MAXERRLEN 1024
void panic (char *format, ...)
	{ char buf[MAXERRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);
	  fprintf (error_file, "%s\n", buf);
	  fprintf (error_file, "tool %s aborted\n", tool_name);
	  if (my_onexit != NULL) my_onexit ();
	  exit (4);
	};

void error (char *format, ...)
	{ char buf[MAXERRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);
	  fprintf (error_file, "%s\n", buf);
	  fflush (error_file);
	  error_count ++;
	};

void wlog (char *format, ...)
	{ char buf[MAXERRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);
	  fprintf (error_file, "%s\n", buf);
	  fflush (error_file);
	};

void warning (char *format, ...)
	{ char buf[MAXERRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);
	  if (verbose)
	     { fprintf (error_file, "%s\n", buf);
	       fflush (error_file);
	     };
	};

void hint (char *format, ...)
	{ char buf[MAXERRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);
	  if (full_verbose)
	     { fprintf (error_file, "%s\n", buf);
	       fflush (error_file);
	     };
	};

void eprint (char *format, ...)
	{ char buf[MAXERRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);
	  fprintf (error_file, "%s", buf);
	  fflush (error_file);
	};

void panic_if_errors ()
	{ if (!error_count) return;
	  panic ("can not continue due to errors");
	};

void bad_tag (int tag, char *routine)
	{ panic ("bad tag %d occurred in routine %s of tool %s",
		 tag, routine, tool_name);
	};

void internal_error (char *routine)
	{ panic ("internal consistency check failed in routine %s of tool %s",
		 routine, tool_name);
	};
