/*
   File: dcg_alloc.c
   Provides basic allocation routines and garbage collection
   while lessening the overhead caused by malloc and free

   Copyright (C) 2008 Marc Seutter

   This library is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published
   by the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with this library.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: dcg_alloc.c,v 1.12 2008/06/28 13:41:16 marcs Exp $"
*/

/* include config.h if autoconfigured */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/* standard includes */
#include <stdio.h>
#include <string.h>

/* Check whether to include malloc or stdlib.h */
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#else
#include <stdlib.h>
#endif

/* local includes */
#include "dcg_error.h"
#include "dcg_alloc.h"

/*
   Nearly all objects allocated by dcg will have the following layout:

      int	ref_count;		ref count < 0 denotes constant
      int	obj_len;		object length in bytes
      char	object[obj_len];	actual object

   Since we plan to do our own memory management (and garbage collection),
   we will maintain free lists for object sizes that are small.
   Our own object overhead is 8 bytes (on Intel or Sparc). Since we do not
   want the overhead caused by malloc (also 8 bytes minimum), we allocate
   the memory for small objects from malloced chunks of 64KB, maintained as
   a simple linked list. For large objects we directly call malloc and free.
*/

/*
   Do a controlled malloc
*/
#define ctl_free(ptr) free (ptr)
static char *ctl_malloc (int size)
	{ char *new = (char *) malloc ((unsigned) size);
	  if (new == NULL) panic ("could not malloc %d bytes", size);
#ifdef DEBUG
	  if (debug) wlog ("Allocated %d bytes at %p", size, new);
#endif
	  return (new);
	};

/*
   Allocate a new chunk and add to the list of chunks
*/
#define ADM_SIZE (2 * sizeof (int))
#define CHUNK_SIZE 65536
static char *current_chunk;
static char *heap_ptr;
static void allocate_chunk ()
	{ char *new_chunk;
	  int offset;

	  /* Allocate chunk and add to linked list of chunks */
	  new_chunk = ctl_malloc (CHUNK_SIZE);
	  * (char **) new_chunk = current_chunk;
	  current_chunk = new_chunk;

	  /* Set up new heap pointer */
	  offset = (sizeof (char *) + ADM_SIZE + 7) & ~7;
	  heap_ptr = new_chunk + offset;

	  /* Report if requested */
#ifdef DEBUG
	  if (debug) wlog ("Allocated chunk from %p to %p, with heap ptr %p",
		   	   new_chunk, new_chunk + CHUNK_SIZE, heap_ptr);
#endif
	};

/*
   Verify that the argument points to a location in one of our chunks
*/
static void verify_managed_address (char *optr)
	{ char *chunk_ptr = * ((char **) current_chunk);
	  unsigned long oint = (unsigned long) optr;
	  if (((unsigned long) current_chunk < oint) &&
	       (oint < (unsigned long) heap_ptr)) return;
	  while (chunk_ptr != NULL)
	     { if (((unsigned long) chunk_ptr < oint) &&
		   (oint < (unsigned long) (chunk_ptr + CHUNK_SIZE))) return;
	       chunk_ptr = * ((char **) chunk_ptr);
	     };
	  error ("found non managed address %p", (void *) optr);
	};

/*
   Allocate a new piece of memory from a current or new chunk
   Register the allocation for debugging purposes

   By choosing ANCHORS_SIZE as 128 we maintain free lists
   for objects of upto 1KB size
*/
#define ANCHORS_SIZE 128
static char *anchors[ANCHORS_SIZE];
static int requested[ANCHORS_SIZE];
static int returned[ANCHORS_SIZE];
static int allocated[ANCHORS_SIZE];
static int huge_allocs, huge_frees;
static char *allocate_from_chunk (int offset)
	{ char *new;
	  int diff;
	  diff = (int) (heap_ptr - current_chunk);
	  if (diff + offset > CHUNK_SIZE)
	     { /* what we need does not fit in current chunk */
	       int rem_offset = CHUNK_SIZE - diff;	/* Note always multiple of 8 */
	       int rem_objlen = rem_offset - ADM_SIZE;	/* also a multiple of 8 */

	       /* if what remains is large enough, add to free lists */
	       if (rem_objlen >= 8)
	          { /* add the remainder to the free lists */
		    int idx = (rem_objlen >> 3) - 1;
		    ((int *) heap_ptr)[-1] = rem_objlen;
		    ((int *) heap_ptr)[-2] = 0;		/* Mark as a free object */
		    * (char **) heap_ptr = anchors[idx];
		    anchors[idx] = heap_ptr;
		    allocated[idx]++;			/* Note as allocation */
		    heap_ptr = heap_ptr + rem_offset;	/* for managed address check */
		  };
	       allocate_chunk ();
	     };
	  new = heap_ptr;
	  heap_ptr = heap_ptr + offset;
	  ((int *) new)[-1] = offset - ADM_SIZE;
	  ((int *) new)[-2] = 0;
	  return (new);
	};

/*
   dcg_malloc will allocate a fresh object either from the free list
   pool or allocate a new one from the current chunk
*/
char *dcg_malloc (int size)
	{ int offset, objlen, idx;
	  char *new, *ptr;

	  /* check argument */
	  if (size <= 0) panic ("trying to allocate %d bytes", size);

	  /* make sure object length is in units of 8 bytes */
	  objlen = (size + 7) & ~7;
	  offset = objlen + ADM_SIZE;

	  /* calculate free list index and allocate the object */
	  idx = (objlen >> 3) - 1;
	  if (idx > ANCHORS_SIZE - 1)			/* if large object */
	     { new = ctl_malloc (offset) + 8;		/* then use libc malloc */
	       huge_allocs++;
	     }
	  else
	     { requested[idx]++;			/* Note request */
	       if ((new = anchors[idx]) != NULL)	/* if present on free lists */
		  anchors[idx] = * (char **) new;	/* then reuse object */
	       else
		  { new = allocate_from_chunk (offset);	/* else allocate new piece */
		    allocated[idx]++;			/* Note allocation */
		  };
	     };

	  /* clear the object, fill in object length and initial refcount */
	  for (ptr = new; ptr < new + objlen; ptr++) *ptr = 0;
	  ((int *) new)[-1] = objlen;
	  ((int *) new)[-2] = 1;
	  return (new);
	};

/*
   dcg_attach increments the reference count of the argument
*/
char *dcg_attach (char *ptr)
	{ int *ref_ptr;
	  if (ptr == NULL)				/* check argument */
	     panic ("trying to attach non existing object");
	  ref_ptr = (int *) (ptr - ADM_SIZE);		/* locate reference count */
	  if (*ref_ptr < 0) return (ptr);		/* constant, return pointer */
	  *ref_ptr = *ref_ptr + 1;			/* increment ref count */
	  return (ptr);					/* return pointer */
	};

/*
   dcg_detach decrements the reference count of the indirect
   argument while nulling the pointer the argument is pointing to.
   If the reference count drops to 0, the object is recycled
*/
void dcg_detach (char **ptr)
	{ char *optr = *ptr;			/* pick up ptr to object */
	  int *ref_ptr;
	  int idx;

	  /* lose ref to object */
	  if (optr == NULL) return;		/* uninitialized object */
	  *ptr = NULL;				/* lose reference */

	  /* check reference count */
	  ref_ptr = (int *) (optr - ADM_SIZE);	/* locate ref count */
	  if (*ref_ptr < 0) return;		/* constant, done with */
	  if (*ref_ptr == 0)
	     panic ("Freeing a free object");
	  *ref_ptr = *ref_ptr - 1;		/* decrement ref count */
	  if (*ref_ptr) return;			/* if refcount > 0, done */

	  /* the object may be freed */
	  idx = (ref_ptr[1] >> 3) - 1;
	  if (idx > ANCHORS_SIZE - 1)		/* if large object */
	     {
#ifdef DEBUG
	       if (debug) wlog ("Freeing %d bytes at %p", ref_ptr[1] + 8, optr - 8);
#endif
	       ctl_free (optr - 8);		/* then use libc free */
	       huge_frees++;			/* note free */
	     }
	  else
	     {
#ifdef DEBUG
	       if (debug) verify_managed_address (optr);
#endif
	       * (char **) optr = anchors[idx];	/* else insert into free list */
	       anchors[idx] = optr;
	       returned[idx]++;			/* note returned to free list */
	     };
	};

/*
   dcg_predetach tries to decrement the reference count of the indirect
   argument while nulling the pointer the argument is pointing to.
   If the reference count would drop to 0, it returns the pointer
   the object, signifying that its subfields should be detached
   before detaching the actual object. It returns NULL otherwise.
*/
char *dcg_predetach (char **ptr)
	{ char *optr = *ptr;			/* pick up ptr to object */
	  int *ref_ptr;

	  /* check reference count */
	  if (optr == NULL) return (NULL);	/* uninitialized object */
	  *ptr = NULL;				/* lose reference */

	  /* ref count < 0 */
	  ref_ptr = (int *) (optr - ADM_SIZE);	/* locate ref count */
	  if (*ref_ptr < 0) return (NULL);	/* is it a constant, done */

	  /* ref count > 1 */
	  if (*ref_ptr > 1)			/* detach is safe */
	     { *ref_ptr = *ref_ptr - 1;		/* decrement ref count */
	       return (NULL);			/* also done */
	     };

	  /* signal the caller to get rid of the subfield(s) before detaching */
	  return (optr);
	};

/*
   dcg_cknonshared verifies that its argument is a valid (non constant) object
   and that it is not shared (i.e. its refcount equals 1). dcg_cknonshared is
   intended to be called by list modification routines.
*/
void dcg_cknonshared (char *ptr)
	{ int refcount;
	  if (ptr == NULL)
	     panic ("Illegal modification of non existing object");
	  refcount = ((int *) ptr)[-2];
	  if (refcount <= 0)
	     panic ("Illegal modification of constant object");
	  if (refcount > 1)
	     panic ("Illegal modification of shared object");
	};

/*
   dcg_mkconstant sets the reference count of an existing object to -1
   i.e. it sets the life time of the object to infinite
*/
void dcg_mkconstant (char *ptr)
	{ if (ptr == NULL)
	     panic ("Illegal constantification of non existing object");
#ifdef DEBUG
	  if (debug) verify_managed_address (ptr);
#endif
	  ((int *) ptr)[-2] = -1;
	  /* optional, push ptr to object on a local string list */
	};

/*
   dcg_calloc allocates an array of things
*/
char *dcg_calloc (int nr, int size)
	{ return (dcg_malloc (nr * size));	/* as you might expect */
	};

/*
   dcg_realloc tries to realloc an array of things
*/
void dcg_realloc (char **ptr, int size)
	{ char *optr = *ptr;			/* pick up ptr to object */
	  char *nptr, *sptr, *dptr;
	  int refc, osize;

	  /* check arguments */
	  if (optr == NULL) panic ("trying to reallocate non existing object");
	  if (size <= 0) panic ("trying to reallocate %d bytes", size);

	  /* check reference count */
	  refc = ((int *) optr)[-2];
	  if (refc != 1) panic ("trying to reallocate a constant or shared object");

	  /* if requested size smaller than resident size, done with */
	  osize = ((int *) optr)[-1];
	  if (size < osize) return;
	  
	  /* Allocate and copy */
	  nptr = dcg_malloc (size);
	  for (sptr = optr, dptr = nptr; sptr < optr + osize; ) *dptr++ = *sptr++;

	  /* detach old, set pointer to new */
	  dcg_detach (ptr);
	  *ptr = nptr;
	};

void dcg_recalloc (char **ptr, int nr, int size)
	{ dcg_realloc (ptr, nr * size);		/* as you might expect */
	};

void init_alloc ()
	{ int ix;
	  current_chunk = NULL;
	  huge_allocs = 0;
	  huge_frees = 0;
	  for (ix = 0; ix < ANCHORS_SIZE; ix++)
	     { anchors[ix] = NULL;
	       allocated[ix] = 0;
	       requested[ix] = 0;
	       returned[ix] = 0;
	     };
	  allocate_chunk ();
	};

/*
   first routines for msck: memory system consistency check
*/
static int free_count[ANCHORS_SIZE];
static void check_free_lists ()
	{ int ix;
	  for (ix = 0; ix < ANCHORS_SIZE; ix++)
	     { int objlen = 0;
	       int count = 0;
	       char *ptr = anchors[ix];
	       while (ptr != NULL)
		  { int refcount = ((int *) ptr)[-2];
		    int len = ((int *) ptr)[-1];
		    if (refcount)
		       error ("found free object of length %d with refcount %d", len, refcount);
		    if ((len >> 3) - 1 != ix)
		       error ("found free object of length %d on queue %d", len, ix);
		    if (!objlen) objlen = len;
		    else if (len != objlen)
		       error ("found objects of size %d and size %d on queue %d", len, objlen, ix);
		    count++;
		    verify_managed_address (ptr);
		    ptr = *((char **) ptr);
		  };
	       free_count[ix] = count;
	     };
	};

#define rep_string \
"queue %d: allocs = %d, %d on free list, requests = %d, frees = %d"
static void make_report ()
	{ int ix;
	  wlog ("Memory report:");
	  for (ix = 0; ix < ANCHORS_SIZE; ix++)
	     { int alloced = allocated[ix];
	       if (alloced)
		  wlog (rep_string, ix, alloced, free_count[ix], requested[ix], returned[ix]);
	     };
	  wlog ("%d huge allocs, %d huge frees", huge_allocs, huge_frees);
	};

void report_alloc ()
	{ check_free_lists ();
	  make_report ();
	};
