/*
   File: rts_alloc.c
   Provides basic allocation routines and garbage collection
*/

/* global includes */
#include <stdio.h>

#if defined (__FreeBSD__)
#include <stdlib.h>
#else
#include <malloc.h>
#endif

/* local includes */
#include <rts_error.h>
#include <rts_alloc.h>

/*
   Mini heap objects (such as TEXTs) will be have the following layout:

      short ref_count;		ref count < 0 denotes constant
      short obj_len;		object length in bytes
      char object[obj_len];	actual object, padded to word alignment
   
   Since we plan to do our own memory management (and garbage collection),
   we will maintain free lists for object sizes less than than 256 bytes.
   Our own object overhead is 4 bytes (on Intel or Sparc). Since we do
   not want the overhead caused by malloc (minimum size allocated is
   16 bytes), we allocate the memory for small objects from malloced chunks
   of 8KB, maintained as a simple linked list. For large objects (>= 256B)
   we directly call malloc and free. The code is not directly portable to
   64 bit architectures since we assume that sizeof (int) = sizeof (char *)
*/
static char *ckmalloc (int size)
	{ char *new = (char *) malloc (size);
	  if (new == NULL) rts_error ("Out of memory");
	  return (new);
	};

#define CHUNK_SIZE 8192
static char *current_chunk;
static char *heap_ptr;
static void allocate_chunk (char *previous)
	{ char *new_chunk = ckmalloc (CHUNK_SIZE);
	  * (char **) new_chunk = previous;
	  current_chunk = new_chunk;
	  heap_ptr = new_chunk + sizeof (char *);
	};

#define ANCHORS_SIZE 64
static char *anchors[ANCHORS_SIZE];
static char *allocate_from_chunk (int needed)
	{ char *new;
	  int diff = (int) (heap_ptr - current_chunk);
	  if (diff + needed > CHUNK_SIZE)
	     { /* what we need does not fit in current chunk */
	       int rem = CHUNK_SIZE - diff - 2 * sizeof (short);
	       if (rem > 0)
		  { /* Check if the remainder of the current chunk fits
		       on one of the free lists; if so insert it */
		    int idx = (rem - 1) >> 2;
		    * (char **) heap_ptr = anchors[idx];
		    anchors[idx] = heap_ptr;
		  };
	       allocate_chunk (* (char **) current_chunk);
	     };
	  new = heap_ptr;
	  heap_ptr = heap_ptr + needed;
	  return (new);
	};

char *rts_malloc (int size)
	{ int needed, obj_len, idx;
	  char *new;
	  if (size <= 0) rts_error ("trying to allocate %d bytes", size);
	  obj_len = (size + sizeof (int) - 1) & ~(sizeof (int) - 1);
	  if (obj_len >= (1 << 15)) rts_error ("trying to allocate %d bytes", size);
	  needed = obj_len + 2 * sizeof (short);
	  idx = (obj_len - 1) >> 2;
	  if (idx > ANCHORS_SIZE - 1)		 /* large object */
	     new = ckmalloc (needed);
	  else
	     { new = anchors[idx];
	       if (new != NULL)
		  anchors[idx] = * (char **) new;
	       else new = allocate_from_chunk (needed);
	     };
	  ((short *) new) [0] = 1;		 /* initial ref count */
	  ((short *) new) [1] = (short) obj_len; /* actual object size */
	  return (new + 2 * sizeof (short));
	};

void rts_attach (char *ptr)
	{ short *sptr;
	  if (ptr == NULL) rts_error ("trying to attach uninitialized object");
	  sptr = (short *) (ptr - 2 * sizeof (short));
	  if (*sptr < 0) return;		 /* constant */  
	  *sptr = *sptr + 1;			 /* increment ref count */
	};

void rts_detach (char **ptr)
	{ short *sptr;
	  int idx;
	  if (ptr == NULL) rts_error ("rts_detach called with NULL pointer");
	  if ((*ptr) == NULL) return;		 /* uninitialized object */
	  sptr = (short *) ((*ptr) - 2 * sizeof (short));
	  *ptr = NULL;				 /* release object */
	  if (*sptr < 0) return;		 /* constant */
	  *sptr = *sptr - 1;			 /* decrement ref count */
	  if (*sptr) return;
	  /* the object may be freed */
	  idx = ((int) sptr[1] - 1) >> 2;
	  if (idx > ANCHORS_SIZE - 1)		 /* large object */
	     free (sptr);
	  else
	     { * (char **) sptr = anchors[idx];
	       anchors[idx] = (char *) sptr;
	     };
	};

void rts_init_gc ()
	{ int ix;
	  for (ix = 0; ix < ANCHORS_SIZE; ix++)
	     anchors[ix] = NULL;
	  allocate_chunk (NULL);
	};
