/*
   File: unparser.c
   Unparses the folded syntax tree delivered by 'cpmerge.c'
   according to the unparsing rules read by 'tuples.c'
*/

/* Global includes */
#include <stdio.h>

/* libeag includes */
#include <export.h>
#include <memalloc.h>
#include <ds.h>
#include <textparsing.h>
#include <nodenrs.h>
#include <tuples.h>

/* local includes */
#include <cpmerge.h>
#include <editor.h>
#include <unparser.h>

public char *unparse_buffer;
private char *optr;

private void find_string_size (char *s, int *dx, int *dy)
	{ char *ptr;
	  int wd = 0;
	  int ht = 0;
	  int lwd = 0;
	  for (ptr = s; *ptr; ptr++)
	     if (*ptr == '\n')
		{ if (wd < lwd) wd = lwd;
		  ht++;
		  lwd = 0;
		}
	     else lwd++;
	  if (wd < lwd) wd = lwd;
	  *dx = wd;
	  *dy = ht;
	};

#define anydir 0
#define horizontal 1
#define vertical 2
#define force_horizontal 3
/*
   The following function is heuristic. If the horizontal rule for
   this treenode does not specify any layout, the user does not expect
   vertical layout anyway. If a nonterminal has only sons that
   are layout horizontally by this same heuristic it will also be
   layout horizontally.
*/
private void test_if_rule_should_be_horizontal (treenode tree)
	{ tuple ptr;
	  int sonnr = 0;
	  int total = 0;
	  for (ptr = ruletable [tree -> nodenr] -> hrule;
	       ptr;
	       ptr = ptr -> next)
	     { if ((ptr -> hor_offset) || (ptr -> vert_offset)) return;
	       switch (ptr -> type)
		  { case type_terminal:
		       total += strlen (ptr -> text);
		       break;
		    case type_set:
		       { total += tree -> sons [sonnr] -> width;
		         sonnr++;
		       };
		       break;
		    case type_nonterminal:
		       { if (tree -> sons [sonnr] !=
				(treenode) force_horizontal) return;
		         total += tree -> sons [sonnr] -> width;
			 sonnr++;
		       };
		       break;
		    case type_forcednonterminal: return;
		  };
	     };
	  tree -> copied = (treenode) force_horizontal;
	  tree -> width = total;
	  tree -> height = 0;
	};

private void initial_estimate_tree_size (treenode tree, int *dx, int *dy,
					 int ambi)
	{ tree -> width = 0;
	  tree -> height = 0;
	  tree -> copied = (treenode) horizontal;
	  switch (tree -> type)
	     { case leafnode:
		  { find_string_size (tree -> name,
				&tree -> width, &tree -> height);
		    tree -> copied = (treenode) force_horizontal;
		  };
		  break;
	       case normalnode:
		  { int i;
		    int max_x = 0;
		    for (i=0; i < tree -> nrsons; i++)
		       { int ldx, ldy;
			 initial_estimate_tree_size
				(tree -> sons[i], &ldx, &ldy, ambi);
			 if (ldx > max_x) max_x = ldx;
		       };
		    tree -> width = max_x;
		    test_if_rule_should_be_horizontal (tree);
		  };
		  break;
	       case ambiguousnode:
		  /* actually doing too much work */
		  { int i, ldx, ldy;
		    for (i = 0; i < tree -> nrsons; i++)
		       initial_estimate_tree_size
				(tree -> sons[i], &ldx, &ldy, 1);
		    tree -> width = tree -> sons[0] -> width;
		    tree -> height = tree -> sons[0] -> height;
		    tree -> copied = tree -> sons[0] -> copied;
		  };
		  break;
	       case errornode:
		  { find_string_size (tree -> name,
				&tree -> width, &tree -> height);
		    if (tree -> height) tree -> copied = (treenode) vertical;
		    else tree -> copied = (treenode) force_horizontal;
		  };
		  break;
	       case typedplaceholdernode:
		  { /* Here we see a hack */
		    tree -> width = strlen (tree -> name + 5) +
					typed_open_symbol_len +
					typed_close_symbol_len;
		    tree -> height = 0;
		    tree -> copied = (treenode) force_horizontal;
		  };
		  break;
	       case untypedplaceholdernode:
		  { if (ambi) tree -> width = untyped_symbol_len;
		    else /* Mark the side effect */
		       { tree -> width = strlen (tree -> name + 5) +
					typed_open_symbol_len +
					typed_close_symbol_len;
			 tree -> type = typedplaceholdernode;
		       };
		    tree -> height = 0;
		    tree -> copied = (treenode) force_horizontal;
		  };
		  break;
	       case predicatenode:
	       case semipredicatenode:
		  { tree -> width = 0;
		    tree -> height = 0;
		    tree -> copied = (treenode) force_horizontal;
		  };
		  break;
	       default:
		  { fprintf (stderr, "Illegal nodetype %d\n", tree -> type);
		    exit (4);
		  };
	     };
	  *dx = tree -> width;
	  *dy = tree -> height;
	};

private int estimate_tree_size (treenode tree, int *dx, int *dy,
				int ambi, int dir, int leftmargin);
private int estimate_normalnode_in_dir (treenode tree, int ambi, int dir,
					int leftmargin)
	{ tuple current;
	  int i = 0;
	  int x = 0;
	  int y = 0;
	  int lasty = 0;
	  int max_x = 0;
	  int *x_pos;
	  int is_hor = dir == horizontal;

	  /* generated editors always have module nr 0 */
	  if (is_hor) current = ruletable [tree -> nodenr] -> hrule;
	  else current = ruletable [tree -> nodenr] -> vrule;
	  x_pos = (int *) ckcalloc (max_nr_of_tuples, sizeof (int));
	  x_pos[0] = 0;
	  
	  while (current)
	     { if (current -> relates < i)
		  { if (is_hor & (i > 0))
		      { ckfree (x_pos);
			return (0);
		      };
		    x_pos [i] = x_pos[current -> relates] +
				current -> hor_offset;
		    y = lasty + current -> vert_offset;
		  }
	       else if (is_hor) x_pos [i] = x + current -> hor_offset;
	       else if (current -> vert_offset)
		  { x_pos [i] = current -> hor_offset;
		    y = lasty + current -> vert_offset;
		  }
	       else x_pos [i] = x + current -> hor_offset;
	       x = x_pos [i];
	       switch (current -> type)
		  { case type_terminal:
		       /* what if we allow \n in terminals */
		       x += strlen (current -> text);
		       break;
		    case type_forcednonterminal:
		       { int lx, ly;
			 treenode son = tree -> sons[current -> sonnr -1];
			 if (is_hor)
			    { ckfree (x_pos);
			      return (0);
			    };
			 (void) estimate_tree_size (son, &lx, &ly, ambi,
				vertical, leftmargin + x_pos[i]);
			 y += ly;
			 x += lx;
		       };
		       break;
		    case type_nonterminal:
		    case type_set:
		       { int lx, ly;
			 int ldir = (is_hor)?horizontal:anydir;
			 treenode son = tree -> sons[current -> sonnr -1];
			 if (!estimate_tree_size (son, &lx, &ly, ambi,
				ldir, leftmargin + x_pos[i]))
			    { ckfree (x_pos);
			      return (0);
			    };
			 if (is_hor && (ly > 0))
			    { ckfree (x_pos);
			      return (0);
			    };
			 y += ly;
			 x += lx;
		       };
		       break;
		    default:
		       { fprintf (stderr, "unknown tupletype\n");
			 exit (4);
		       };
	          };
	       i++;
	       lasty = y;
	       if (max_x < x) max_x = x;
	       if (screenwidth <= max_x + leftmargin)
		  { ckfree (x_pos);
		    return (0);
		  };
	       current = current -> next;
	     };
	  tree -> width = max_x;
	  tree -> height = y;
	  ckfree (x_pos);
	  return (1);
	};

private int estimate_normalnode (treenode tree, int ambi, int dir,
				 int leftmargin)
	{ switch (dir)
	     { case horizontal:
	          if (estimate_normalnode_in_dir (tree, ambi,
				horizontal, leftmargin))
		     { tree -> copied = (treenode) horizontal;
		       return (1);
		     }
		  else return (0);
	       case anydir:
		  if (estimate_normalnode_in_dir (tree, ambi,
				horizontal, leftmargin))
		     { tree -> copied = (treenode) horizontal;
		       return (1);
		     };
	       case vertical:
		  { (void) estimate_normalnode_in_dir (tree, ambi,
				vertical, leftmargin);
		    tree -> copied = (treenode) vertical;
		  };
	     };
	  return (1);
	};

private int estimate_tree_size (treenode tree, int *dx, int *dy,
				int ambi, int dir, int leftmargin)
	{ if (tree -> copied != (treenode) force_horizontal)
	     switch (tree -> type)
	     { case normalnode:
		  if (!estimate_normalnode (tree, ambi, dir, leftmargin))
		     return (0);
		  break;
	       case ambiguousnode:
		  /* actually doing too much work */
		  { int i, ldx, ldy;
		    for (i = 0; i < tree -> nrsons; i++)
		       if (!estimate_tree_size (tree -> sons[i],
				&ldx, &ldy, 1, dir, leftmargin)) return (0);
		    tree -> width = tree -> sons[0] -> width;
		    tree -> height = tree -> sons[0] -> height;
		  };
		  break;
	       case errornode:
		  break;
	       default:
		  { fprintf (stderr, "Illegal nodetype %d\n", tree -> type);
		    exit (4);
		  };
	     };
	  *dx = tree -> width;
	  *dy = tree -> height;
	  if (dir == horizontal)
	     return ((*dy == 0) && (leftmargin + *dx < screenwidth));
	  return (1);
	};

private void print_string (char *s)
	{ char *ptr;
	  for (ptr = s; *ptr; ptr++, optr++) *optr = *ptr;
	};

private void print_spaces (int nr)
	{ int i;
	  for (i=0; i < nr; i++) *optr++ = ' ';
	};

private void print_newlines (int nr)
	{ int i;
	  for (i=0; i < nr; i++) *optr++ = '\n';
	};

private void print_tree (treenode tree, int line, int col);
private void print_normalnode (treenode tree, int line, int col)
	{ tuple current;
	  int i = 0;
	  int y = line;
	  int x = col;
	  int lasty = line;
	  int *x_pos;
	  int is_hor = (tree -> copied == (treenode) horizontal) ||
		       (tree -> copied == (treenode) force_horizontal);
	  int is_vert = (tree -> copied == (treenode) vertical);
	  if (is_hor) current = ruletable [tree -> nodenr] -> hrule;
	  else if (is_vert) current = ruletable [tree -> nodenr] -> vrule;
	  else
	     { fprintf (stderr, "printing '%s' fails\n", tree -> name);
	       current = ruletable [tree -> nodenr] -> vrule;
	     };
	  x_pos = (int *) ckcalloc (max_nr_of_tuples, sizeof (int));
	  x_pos[0] = col;

	  while (current)
	     { if (current -> relates < i)
		  { x_pos [i] = x_pos[current -> relates] +
				current -> hor_offset;
		    y = lasty + current -> vert_offset;
		    print_newlines (current -> vert_offset);
		    print_spaces (x_pos [i]);
		  }
	       else if (is_hor)
		  { x_pos [i] = x + current -> hor_offset;
		    print_spaces (current -> hor_offset);
		  }
	       else if (current -> vert_offset)
		  { x_pos [i] = current -> hor_offset;
		    y = lasty + current -> vert_offset;
		    print_newlines (current -> vert_offset);
		    print_spaces (x_pos[i]);
		  }
	       else
		  { x_pos [i] = x + current -> hor_offset;
		    print_spaces (current -> hor_offset);
		  };
	       x = x_pos [i];
	       switch (current -> type)
		  { case type_terminal:
		       /* what if we allow \n in terminals */
		       { print_string (current -> text);
		         x += strlen (current -> text);
		       };
		       break;
		    case type_forcednonterminal:
		    case type_nonterminal:
		    case type_set:
		       { treenode son = tree -> sons[current -> sonnr -1];
			 int oldy = y;
			 y += son -> height;
			 x += son -> width;
			 print_tree (son, oldy, x_pos[i]);
		       };
		       break;
		    default:
		       { fprintf (stderr, "unknown tupletype\n");
			 exit (4);
		       };
	          };
	       i++;
	       lasty = y;
	       current = current -> next;
	     };
	  ckfree (x_pos);
	};

private void print_tree (treenode tree, int line, int col)
	{ tree -> y = line;
	  tree -> x = col;
	  switch (tree -> type)
	     { case leafnode:
		  print_string (tree -> name);
		  break;
	       case normalnode:
		  print_normalnode (tree, line, col);
		  break;
	       case ambiguousnode:
		  /* some research is still necessary at this point */
		  print_tree (tree -> sons[0], line, col);
		  break;
	       case errornode:
		  print_string (tree -> name);
		  break;
	       case typedplaceholdernode:
		  { /* Yep, this is the same hack */
		    print_string (typed_open_symbol);
		    print_string (tree -> name + 5);
		    print_string (typed_close_symbol);
		  };
		  break;
	       case untypedplaceholdernode:
		  print_string (untyped_symbol);
		  break;
	       case predicatenode:
	       case semipredicatenode:
		  break;
	       default:
		  { fprintf (stderr, "Illegal nodetype %d\n", tree -> type);
		    exit (4);
		  };
	     };
	};

public int unparsed_width;
public int unparsed_height;
private void unparse_tree (treenode tree)
	{ int dx, dy;
	  optr = unparse_buffer;
	  initial_estimate_tree_size (tree, &dx, &dy, 0);
	  if (!estimate_tree_size (tree, &dx, &dy, 0, anydir, 0))
	     { fprintf (stderr, "syntax tree could not be unparsed\n");
	       exit (4);
	     };
	  unparsed_width = dx;
	  unparsed_height = dy;
	  print_tree (tree, 0, 0);
	  *optr = '\0';
#ifdef Debug
	  fprintf (stderr, "Unparse resulted in:\n%s\n", unparse_buffer);
#endif
	};

public void unparse ()
	{ unparse_tree (the_root);
	};

public void init_unparser ()
	{ unparse_buffer = (char *) ckmalloc (parsebuffer_size);
	};
