/*
   File: cpmerge.c
   Defines the actions for copying the syntax tree or merging it
   with an existing one, possibly leading to 3D trees.
*/

/* Global includes */
#include <stdio.h>

/* libdcg includes */
#include <export.h>
#include <memalloc.h>
#include <ds.h>
#include <buildaffixgraph.h>
#include <buildtree.h>

/* local includes */
#include <cpmerge.h>

public treenode the_root;
private void cp_pos_side (posnode new, posnode old, int side)
	{ int type = old -> sides[side].type;
	  new -> sides[side].type = type;
	  new -> sides[side].sill = old -> sides[side].sill;
	  switch (type)
	     { case singleaffix:
		  old -> sides[side].a.affx -> copied = affixnode_nil;
		  break;
	       case concataffix:
	       case composaffix:
		  { int i;
		    int nr = old -> sides[side].a.co.nraffs;
		    new -> sides[side].a.co.nraffs = nr;
		    new -> sides[side].a.co.affs = new_affixspace (nr);
		    for (i=0; i < nr; i++)
		       old -> sides[side].a.co.affs[i] -> copied =
			  affixnode_nil;

		  };
	     };
	};

private void cp_pos (treenode copy, treenode node)
	{ int i;
	  copy -> affs = new_posspace (node -> nraffs);
	  for (i=0; i < node -> nraffs; i++)
	     { posnode old = node -> affs[i];
	       posnode new = new_posnode ();
	       copy -> affs[i] = new;
	       new -> node = copy;
	       cp_pos_side (new, old, lower_side);
	       cp_pos_side (new, old, upper_side);
	       new -> delayed = old -> delayed;
	       new -> args = old -> args;
	       new -> dfunc = old -> dfunc;
	     };
	};

private treenode cptree (treenode node, treenode destfather)
	{ int i;
	  treenode copy;
	  if (node == treenode_nil) return (treenode_nil);
	  copy = new_treenode ();
	  copy -> type = node -> type;
	  copy -> name = node -> name;
	  copy -> nodenr = node -> nodenr;
	  copy -> nrsons = node -> nrsons;
	  copy -> sons = new_sonspace (node -> nrsons);
	  copy -> father = destfather;
	  copy -> nraffs = node -> nraffs;
	  copy -> copied = treenode_nil;
	  cp_pos (copy, node);
	  for (i=0; i < node -> nrsons; i++)
	     copy -> sons[i] = cptree (node -> sons[i], copy);
	  node -> copied = copy;
	  return (copy);
	};

private linknode cp_links (linknode old)
	{ linknode new;
	  treenode oldnode;
	  int i;
	  if (old == linknode_nil) return (linknode_nil);
	  if (old -> node -> copied == treenode_nil)
	     return (cp_links (old -> next));

	  new = new_linknode ();
	  oldnode = old -> node;
	  for (i=0; i < oldnode -> nraffs; i++)
	     if (oldnode -> affs[i] == old -> pos) break;
	  new -> node = oldnode -> copied;
	  new -> pos = oldnode -> copied -> affs[i];
	  new -> side = old -> side;
	  new -> type = old -> type;
	  new -> next = cp_links (old -> next);
	  return (new);
	};

private affixnode cp_affix (affixnode old)
	{ affixnode new;
	  if (old -> copied) return (old -> copied);
	  new = new_affixnode (old -> name);
	  new -> defined = old -> defined;
	  new -> hasval = old -> hasval;
	  new -> value = old -> value;
	  new -> copied = affixnode_nil;
	  if (old -> value) attach_valuenode (old -> value);
	  new -> mfunc = old -> mfunc;
	  new -> links = cp_links (old -> links);
	  old -> copied = new;
	  return (new);
	};

private void cp_affixes_in_side (posnode new, posnode old, int side)
	{ int type = old -> sides[side].type;
	  switch (type)
	     { case singleaffix:
		  new -> sides[side].a.affx =
			cp_affix (old -> sides[side].a.affx);
		  break;
	       case concataffix:
	       case composaffix:
		  { int i;
		    int nr = old -> sides[side].a.co.nraffs;
		    for (i=0; i < nr; i++)
		       new -> sides[side].a.co.affs[i] =
			  cp_affix (old -> sides[side].a.co.affs[i]);
		  };
		  break;
	     };
	};

private void cp_affixes_in_pos (posnode new, posnode old)
	{ cp_affixes_in_side (new, old, upper_side);
	  cp_affixes_in_side (new, old, lower_side);
	};

private void cpaffixes (treenode node)
	{ int i;
	  if (node == treenode_nil) return;
	  for (i=0; i < node -> nraffs; i++)
	     cp_affixes_in_pos (node -> copied -> affs[i], node -> affs[i]);
	  for (i=0; i < node -> nrsons; i++)
	     cpaffixes (node -> sons[i]);
	};

/*
   The syntax tree and affix graph are copied in two passes.
*/
private treenode copy_subtree (treenode node)
	{ treenode new = cptree (node, treenode_nil);
	  cpaffixes (node);
	  return (new);
	};

/*
   merge trees
*/
private treenode insert_ambiguousnode (treenode first, treenode second)
	{ int i;
	  treenode ambi = new_treenode ();
	  ambi -> type = ambiguousnode;
	  ambi -> name = first -> name;
	  ambi -> nodenr = first -> nodenr;
	  ambi -> nrsons = 2;
	  ambi -> sons = new_sonspace (2);
	  ambi -> sons [0] = first;
	  ambi -> sons [1] = second;
	  /* reparenting */
	  for (i=0; i < first -> father -> nrsons; i++)
	     if (first == first -> father -> sons[i]) break;
	  first -> father -> sons [i] = ambi;
	  ambi -> father = first -> father;
	  first -> father = ambi;
	  second -> father = ambi;
	  ambi -> x = -1;
	  ambi -> y = 0;
	  ambi -> copied = treenode_nil;
	  return (ambi);
	};

private void update_links (affixnode affx, treenode ambi, treenode first,
			   posnode new, posnode old)
	{ linknode ptr;
	  for (ptr = affx -> links; ptr != linknode_nil; ptr = ptr -> next)
	     { if (ptr -> node == first) ptr -> node = ambi;
	       if (ptr -> pos == old) ptr -> pos = new;
	     };
	};

private void add_upperside_affixes (treenode ambi, treenode first,
				    posnode new, posnode old)
	{ int type = old -> sides[upper_side].type;
	  new -> sides[upper_side].type = type;
	  switch (type)
	     { case singleaffix:
		  { affixnode affx = old -> sides[upper_side].a.affx;
		    new -> sides[upper_side].a.affx = affx;
		    update_links (affx, ambi, first, new, old);
		  };
		  break;
	       case concataffix:
	       case composaffix:
		  { int i;
		    int nr = old -> sides[upper_side].a.co.nraffs;
		    new -> sides[upper_side].a.co.affs = new_affixspace (nr);
		    for (i=0; i < nr; i++)
		       { affixnode affx = old -> sides[upper_side].a.co.affs[i];
		         new -> sides[upper_side].a.co.affs[i] = affx;
			 update_links (affx, ambi, first, new, old);
		       };
		  };
		  break;
	     };
	};

private affixnode make_daffix_in_lowerside (posnode newpos, affixnode old)
	{ affixnode new = make_undefined_affix ();
	  new -> defined = old -> defined;
	  new -> hasval = old -> hasval;
	  new -> value = old -> value;
	  if (old -> value) attach_valuenode (old -> value);
	  new -> mfunc = old -> mfunc;
	  add_dlink (new, newpos, lower_side);
	  return (new);
	};

private void add_lowerside_affixes (posnode new, posnode old)
	{ int type = old -> sides[lower_side].type;
	  new -> sides[lower_side].type = type;
	  switch (type)
	     { case singleaffix:
		  new -> sides[lower_side].a.affx =
		     make_daffix_in_lowerside (new,
			old -> sides[lower_side].a.affx);
		  break;
	       case concataffix:
	       case composaffix:
		  { int i;
		    int nr = old -> sides[lower_side].a.co.nraffs;
		    new -> sides[lower_side].a.co.affs = new_affixspace (nr);
		    for (i=0; i < nr; i++)
		       new -> sides[lower_side].a.co.affs[i] =
			  make_daffix_in_lowerside (new,
				old -> sides[lower_side].a.co.affs[i]);
		  };
		  break;
	     };
	};

private void add_affixes_to_ambiguousnode (treenode ambi, treenode first)
	{ int i;
	  ambi -> nraffs = first -> nraffs;
	  ambi -> affs = new_posspace (ambi -> nraffs);
	  for (i=0; i < ambi -> nraffs; i++)
	     { posnode old = first -> affs[i];
	       posnode new = new_posnode ();
	       ambi -> affs[i] = new;
	       new -> node = ambi;
	       add_upperside_affixes (ambi, first, new, old);
	       add_lowerside_affixes (new, old);
	     };
	};

private void add_upperside_affixes_to_first (treenode first)
	{ int i;
	  for (i=0; i < first -> nraffs; i++)
	     { posnode old = first -> affs[i];
	       old -> sides[upper_side].type = singleaffix;
	       old -> sides[upper_side].a.affx = make_undefined_affix ();
	       add_dlink (old -> sides[upper_side].a.affx, old, upper_side);
	     };
	};

private void inherit_upperside_affixes_from_first (treenode ambi, treenode new)
	{ int i;
	  treenode first = ambi -> sons[0];
	  for (i=0; i < first -> nraffs; i++)
	     { posnode old = first -> affs[i];
	       posnode newpos = new -> affs[i];
	       newpos -> sides[upper_side].type = singleaffix;
	       newpos -> sides[upper_side].a.affx =
		  	old -> sides[upper_side].a.affx;
	       add_dlink (newpos -> sides[upper_side].a.affx,
			newpos, upper_side);
	     };
	};

private void make_ambiguousnode (treenode first, treenode second)
	{ treenode ambi = insert_ambiguousnode (first, second);
	  add_affixes_to_ambiguousnode (ambi, first);
	  add_upperside_affixes_to_first (first);
	  inherit_upperside_affixes_from_first (ambi, second);
	};

private void addto_ambiguoussons (treenode ambi, treenode new)
	{ int i;
	  treenode *newsons;

	  ambi -> nrsons++;
	  newsons = new_sonspace (ambi -> nrsons);
	  for (i=0; i < ambi -> nrsons - 1; i++)
	     newsons [i] = ambi -> sons[i];
	  newsons [ambi -> nrsons - 1] = new;
	  new -> father = ambi;
	  free_sonspace (ambi -> nrsons - 1, ambi -> sons);
	  ambi -> sons = newsons;
	};

private void addto_ambiguousnode (treenode ambi, treenode new)
	{ addto_ambiguoussons (ambi, new);
	  inherit_upperside_affixes_from_first (ambi, new);
	};

/*
   for the following routine to be correct we should actually prove
   that if an ambiguous node is constructed for one of the sons,
   the whole syntax tree of 'node' is merged with that of 'dest'.
*/
private int merge_trees (treenode dest, treenode new);
private int try_merge_sons (treenode dest, treenode new)
	{ int i;
	  for (i=0; i < dest -> nrsons; i++)
	     if (merge_trees (dest -> sons[i], new -> sons[i]))
		return (1);
	  return (0);
	};

private int merge_trees (treenode dest, treenode new)
	{ int i;
	  if (new == treenode_nil) return (1);	/* ???? */
	  new -> copied = treenode_nil;		/* make sure he is uncopied */
	  if ((dest -> nodenr == new -> nodenr) &&
	      (dest -> type != ambiguousnode))
	     return (try_merge_sons (dest, new));
	  if (dest -> type == ambiguousnode)
	     { /* This is already an ambiguous node:
		  try and merge with one of its sons; */
	       for (i=0; i < dest -> nrsons; i++)
		  { treenode dson = dest -> sons[i];
		    if (dson -> nodenr == new -> nodenr)
		       if (try_merge_sons (dson, new))
			  return (1);
		  };
	       addto_ambiguousnode (dest, copy_subtree (new));
	       return (0);
	     };
	  /* dest should become ambiguous. however we should be careful
	     if dest has more than one son. if so, climb up in the syntax
	     tree until we reach another alternative.
	  */
	  if (dest -> nrsons > 1 || new -> nrsons > 1)
	     while (dest -> father -> nodenr == dest -> nodenr)
		{ new = new -> father;
		  dest = dest -> father;
		  if (dest -> father -> type == ambiguousnode)
		     dest = dest -> father;	
		};
	  make_ambiguousnode (dest, copy_subtree (new));
	  return (1);
	};

void copy_tree ()
	{ if (the_root == treenode_nil)
	     { the_root = copy_subtree (top_treenode());
	     }
	  else 
	     { (void) merge_trees (the_root, top_treenode ());
	     };
#ifdef Debug
	  fprintf (stderr, "\nResulting folded tree\n");
	  dump_parse_tree_indented (the_root, 0);
#endif
	  callq ();
	  pushq (copy_tree);
	};

public void init_cpmerge ()
	{ the_root = treenode_nil;
	};
