/*
   File: trace.c
   Provides some debugging support
*/

/* global includes */
#include <stdio.h>

/* local includes */
#include <export.h>
#include <error.h>
#include <ds.h>
#include <trace.h>
#include <nodenrs.h>
#include <textparsing.h>

public int tracing = 0;
private int indent;
public void init_trace ()
	{ indent = 0;
	  tracing = 1;
	}

private void output_char (char c)
	{ if (c == '\t') eprint_log ("\\t");
	  else if (c == '\n') eprint_log ("\\n");
	  else if (c == EOFCHAR) eprint_log ("\\e");
	  else if (c < ' ') eprint_log ("\\%o", c);
	  else eprint_log ("%c", c);
	};

private void output_indent ()
	{ eprint_log ("%d '", indent);
	  output_char (*iptr);
	  eprint_log ("': ");
	};

private void output_string (char *s)
	{ char *ptr;
	  for (ptr = s; *ptr; ptr++) output_char (*ptr);
	};

public void trace_value (valuenode v)
	{ if (v == valuenode_nil) eprint_log ("nil");
	  else
	    switch (v -> type)
	     { case undefinedtype: eprint_log ("_"); break;
	       case stringtype:
	          { eprint_log ("%c", '"');
		    output_string (v -> v.string);
		    eprint_log ("%c", '"');
		  };
		  break;
	       case numbertype: eprint_log ("%d", v -> v.number); break;
	       case compostype:
		  { int i;
		    eprint_log ("(");
		    for (i=0; i < v -> v.co.nr; i++)
		       { trace_value (v -> v.co.vals[i]);
			 if (i != v -> v.co.nr - 1) eprint_log (" * ");
		       };
		    eprint_log (")");
		  };
	     };
	};

public void trace_affix (affixnode a)
	{ eprint_log ("%s", a -> name);
	  if (a -> hasval)
	     { eprint_log ("<");
	       trace_value (a -> value);
	       eprint_log (">");
	     };
	};

public void trace_posside (posnode p, int i)
	{ int type = p -> sides[i].type;
	  output_indent ();
	  eprint_log ("%s = (", (i == lower_side)?"lo":"hi");
	  eprint_log ("sill = %d, ", p -> sides[i].sill);
	  switch (type)
	     { case undefinedaffix: eprint_log ("_"); break;
	       case singleaffix: trace_affix (p -> sides[i].a.affx); break;
	       case concataffix:
	       case composaffix:
		  { int j;
		    int nr = p -> sides[i].a.co.nraffs;
		    for (j=0; j < nr; j++)
		       { trace_affix (p -> sides[i].a.co.affs[j]);
			 if (j != nr - 1)
			    eprint_log (" %c ", (type == concataffix)?'+':'*');
		       };
		  };
	     };
	  eprint_log (")\n");
	};

private int find_position_nr (posnode p, treenode node)
	{ int i;
	  for (i=0; i < node -> nraffs; i++)
	     if (node -> affs[i] == p) return (i);
	  return (-1);
	};

public void trace_pos (posnode p)
	{ treenode node = p -> node;
	  int nodenr = node -> nodenr;
	  int pos_nr = find_position_nr (p, node);
	  output_indent();
	  eprint_log ("tracing position %d of %s in module %s\n",
		      pos_nr, node -> name, module_name_from_nodenr (nodenr));
	  trace_posside (p, lower_side);
	  trace_posside (p, upper_side);
	};

public void trace_enter (char *s)
	{ output_indent ();
	  eprint_log (">%s\n", s);
	  indent++;
	};

public void trace_alternative (char *s, int i)
	{ output_indent ();
	  eprint_log ("-%s, alt %d\n", s, i);
	};

public void trace_leave (char *s)
	{ indent--;
	  output_indent ();
	  eprint_log ("<%s\n", s);
	};
