/*
   File: stddefs.c
   Implements the standard eag library
*/

/* global includes */
#include <stdio.h>
#include <ctype.h>

/* local includes */
#include <export.h>
#include <error.h>
#include <ds.h>
#include <textstorage.h>
#include <textparsing.h>
#include <buildtree.h>
#include <buildaffixgraph.h>
#include <nodenrs.h>
#include <propagate.h>
#include <stddefs.h>

/* our module number */
private int stddefs_modnr = 1;

/*
   Meta rules
   nil, string, int, tuple, any
*/

/*
   meta defined nil
*/
public affixnode make_nil_value ()
	{ affixnode new = new_affixnode ("predef_nil");
	  valuenode newval = new_valuenode ();
	  newval -> type = compostype;
	  newval -> ref_count = 1;
	  newval -> v.co.nr = 0;
	  new -> defined = 1;
	  new -> hasval = 1;
	  new -> value = newval;
	  return (new);
	};

public void rec_nil_value ()
	{ valuenode v = popv();
	  if (v -> type == undefinedtype) callq();
	  else if (v -> type == compostype)
	     { if (v -> v.co.nr == 0) callq ();
	     };
	  pushv (v);
	  pushq (rec_nil_value);
	};

/*
   meta defined string
*/
public void rec_string_value ()
	{ valuenode v = popv();
	  if (v -> type == undefinedtype) callq();
	  else if (v -> type == stringtype) callq();
	  pushv(v);
	  pushq(rec_string_value);
	};

public affixnode make_string_value ()
	{ affixnode new = new_affixnode ("predef_string");
	  new -> defined = 1;
	  new -> mfunc = rec_string_value;
	  return (new);
	};

public void recbody_string_value ()
	{ int i;
	  for (i = 0; i <= strlen (miptr); i++)
	     { char *sptr = miptr;
	       miptr = miptr+i;
	       callq ();
	       miptr = sptr;
	     };
	  pushq (recbody_string_value);
	};

/*
   meta defined tuple
*/
public void rec_tuple_value ()
	{ valuenode v = popv();
	  if (v -> type == undefinedtype) callq();
	  else if (v -> type == compostype) callq();
	  pushv(v);
	  pushq(rec_tuple_value);
	};

public affixnode make_tuple_value ()
	{ affixnode new = new_affixnode ("predef_tuple");
	  new -> defined = 1;
	  new -> mfunc = rec_tuple_value;
	  return (new);
	};

/*
   meta defined int
*/
public void rec_int_value ()
	{ valuenode v = popv();
	  if (v -> type == undefinedtype) callq();
	  else if (v -> type == numbertype) callq();
	  pushv (v);
	  pushq (rec_int_value);
	};

public affixnode make_int_value ()
	{ affixnode new = new_affixnode ("predef_int");
	  new -> defined = 1;
	  new -> mfunc = rec_int_value;
	  return (new);
	};

public void recbody_int_value ()
	{ int i;
	  if (mtotal >= 0)
	     for (i=0; i <= mtotal; i++)
		{ int s = mtotal;
		  mtotal = mtotal - i;
		  callq ();
		  mtotal = s;
		};
	  pushq (recbody_int_value);
	};

/*
   meta defined any
*/
public void rec_any_value ()
	{ valuenode v = popv();
	  callq ();
	  pushv (v);
	  pushq (rec_any_value);
	};

public affixnode make_any_value ()
	{ affixnode new = new_affixnode ("predef_any");
	  return (new);
	};

/*
   Semi Predicates: end of sentence (0), column (1), row (2),
*/

/*
   semipredicate end of sentence
*/
public void semipred_endofsentence ()
	{ pushi (0);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 0));
	  pushq (make_semipredicatenode);
	  pushq (endofsentence);
          callq ();
	  pop (5);
          pushq (semipred_endofsentence);
        };

/*
   semipredicate column
*/
public void semipred_column ()
	{ int cc = currentcolumn ();
	  affixnode a_col = number_to_affix ("predef_column", cc);
	  pusha (a_col);
	  pushi (singleaffix);
	  pushi (1);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 1));
	  pushq (make_semipredicatenode);
	  callq ();
	  pop (6);
	  detach_valuenode (a_col -> value);
	  free_affixnode (a_col);
	  pushq (semipred_column);
	};

/*
   semi predicate row
*/
public void semipred_row ()
	{ int cc = currentrow ();
	  affixnode a_row = number_to_affix ("predef_row", cc);
	  pusha (a_row);
	  pushi (singleaffix);
	  pushi (1);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 2));
	  pushq (make_semipredicatenode);
	  callq ();
	  pop (6);
	  detach_valuenode (a_row -> value);
	  free_affixnode (a_row);
	  pushq (semipred_row);
	};

/*
   Generic predicates: equal (3), not equal (4)
*/

/*
   predicate equal
*/
private void act_equal (posnode ps1, posnode ps2)
	{ if (ps1 -> sides[lower_side].sill == 0)
	     { valuenode v1 = calc_affix_value (ps1, lower_side);
	       pushp (ps2);
	       pushv (v1);
	       pushq (propagate_predicate_value);
	       callq ();
	       pop (3);
	       detach_valuenode (v1);
	     }
	  else if (ps2 -> sides[lower_side].sill == 0)
	     { valuenode v2 = calc_affix_value (ps2, lower_side);
	       pushp (ps1);
	       pushv (v2);
	       pushq (propagate_predicate_value);
	       callq ();
	       pop (3);
	       detach_valuenode (v2);
	     }
	  else callq ();
	};

private void delayed_equal (posnode *ps)
	{ posnode ps1 = ps[0];
	  posnode ps2 = ps[1];
	  if (!position_is_defined (ps1) &&
	      !position_is_defined (ps2)) callq ();
	  else if (!position_has_value (ps1) &&
		   !position_has_value (ps2)) callq ();
	  else
	     { ps1 -> delayed = 0;
	       ps2 -> delayed = 0;
	       act_equal (ps1, ps2);
	       ps2 -> delayed = 1;
	       ps1 -> delayed = 1;
	     }
	};

public void pred_equal ()
	{ affixnode af1 = new_affixnode ("equal_af1");
	  affixnode af2 = new_affixnode ("equal_af2");
	  pushq (delayed_equal);
	  pushq (make_node_delayed);
	  pusha (af2);
	  pushi (singleaffix);
	  pusha (af1);
	  pushi (singleaffix);
	  pushi (2);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 3));		/* equal nodenr */
	  pushq (make_predicatenode);
	  callq ();
	  pop (10);
	  detach_valuenode (af1 -> value);
	  free_affixnode (af1);
	  detach_valuenode (af2 -> value);
	  free_affixnode (af2);
	  pushq (pred_equal);
	};

/*
   predicate notequal
*/
private void act_notequal (posnode ps1, posnode ps2)
	{ valuenode v1 = calc_affix_value (ps1, lower_side);
	  valuenode v2 = calc_affix_value (ps2, lower_side);
	  if (!equal_affix_value (v1, v2)) callq ();
	  detach_valuenode (v1);
	  detach_valuenode (v2);
	};

private void delayed_notequal (posnode *ps)
	{ posnode ps1 = ps[0];
	  posnode ps2 = ps[1];
	  if (!position_is_defined (ps1) ||
	      !position_is_defined (ps2)) callq ();
	  else if (!position_has_value (ps1) ||
		   !position_has_value (ps2)) callq ();
	  else
	     { ps1 -> delayed = 0;
	       ps2 -> delayed = 0;
	       act_notequal (ps1, ps2);
	       ps2 -> delayed = 1;
	       ps1 -> delayed = 1;
	     }
	};

public void pred_notequal ()
	{ affixnode af1 = new_affixnode ("notequal_af1");
	  affixnode af2 = new_affixnode ("notequal_af2");
	  pushq (delayed_notequal);
	  pushq (make_node_delayed);
	  pusha (af2);
	  pushi (singleaffix);
	  pusha (af1);
	  pushi (singleaffix);
	  pushi (2);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 4));		/* notequal nodenr */
	  pushq (make_predicatenode);
	  callq ();
	  pop (10);
	  detach_valuenode (af1 -> value);
	  free_affixnode (af1);
	  detach_valuenode (af2 -> value);
	  free_affixnode (af2);
	  pushq (pred_notequal);
	};

/*
   Numeric predicates: minus (5), mul (6), div (7), power (8)
*/

/*
   pred_minus
*/
private void act_minus (posnode ps1, posnode ps2, posnode ps3)
	{ valuenode v1 = calc_affix_value (ps1, lower_side);
	  valuenode v2 = calc_affix_value (ps2, lower_side);
	  if ((v1 -> type == numbertype) && (v2 -> type == numbertype))
	     { int new = v1 -> v.number - v2 -> v.number;
	       valuenode v3 = number_to_value (new);
	       pushp (ps3);
	       pushv (v3);
	       pushq (propagate_predicate_value);
	       callq ();
	       pop (3);
	       detach_valuenode (v3);
	     };
	  detach_valuenode (v1);
	  detach_valuenode (v2);
	};

private void delayed_minus (posnode *ps)
	{ posnode ps1 = ps[0];
	  posnode ps2 = ps[1];
	  posnode ps3 = ps[2];
	  if (!position_is_defined (ps1) ||
	      !position_is_defined (ps2)) callq ();
	  else if (!position_has_value (ps1) ||
		   !position_has_value (ps2)) callq ();
	  else
	     { ps1 -> delayed = 0;
	       ps2 -> delayed = 0;
	       ps3 -> delayed = 0;
	       act_minus (ps1, ps2, ps3);
	       ps3 -> delayed = 1;
	       ps2 -> delayed = 1;
	       ps1 -> delayed = 1;
	     }
	};

public void pred_minus ()
	{ affixnode af1 = new_affixnode ("minus_af1");
	  affixnode af2 = new_affixnode ("minus_af2");
	  affixnode af3 = new_affixnode ("minus_af3");
	  pushq (delayed_minus);
	  pushq (make_node_delayed);
	  pusha (af3);
	  pushi (singleaffix);
	  pusha (af2);
	  pushi (singleaffix);
	  pusha (af1);
	  pushi (singleaffix);
	  pushi (3);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 5));		/* minus nodenr */
	  pushq (make_predicatenode);
	  callq ();
	  pop (10);
	  detach_valuenode (af1 -> value);
	  free_affixnode (af1);
	  detach_valuenode (af2 -> value);
	  free_affixnode (af2);
	  detach_valuenode (af3 -> value);
	  free_affixnode (af3);
	  pushq (pred_minus);
	};

/*
   pred_mul
*/
private void act_mul (posnode ps1, posnode ps2, posnode ps3)
	{ valuenode v1 = calc_affix_value (ps1, lower_side);
	  valuenode v2 = calc_affix_value (ps2, lower_side);
	  if ((v1 -> type == numbertype) && (v2 -> type == numbertype))
	     { int new = v1 -> v.number * v2 -> v.number;
	       valuenode v3 = number_to_value (new);
	       pushp (ps3);
	       pushv (v3);
	       pushq (propagate_predicate_value);
	       callq ();
	       pop (3);
	       detach_valuenode (v3);
	     };
	  detach_valuenode (v1);
	  detach_valuenode (v2);
	};

private void delayed_mul (posnode *ps)
	{ posnode ps1 = ps[0];
	  posnode ps2 = ps[1];
	  posnode ps3 = ps[2];
	  if (!position_is_defined (ps1) ||
	      !position_is_defined (ps2)) callq ();
	  else if (!position_has_value (ps1) ||
		   !position_has_value (ps2)) callq ();
	  else
	     { ps1 -> delayed = 0;
	       ps2 -> delayed = 0;
	       ps3 -> delayed = 0;
	       act_mul (ps1, ps2, ps3);
	       ps3 -> delayed = 1;
	       ps2 -> delayed = 1;
	       ps1 -> delayed = 1;
	     }
	};

public void pred_mul ()
	{ affixnode af1 = new_affixnode ("mul_af1");
	  affixnode af2 = new_affixnode ("mul_af2");
	  affixnode af3 = new_affixnode ("mul_af3");
	  pushq (delayed_mul);
	  pushq (make_node_delayed);
	  pusha (af3);
	  pushi (singleaffix);
	  pusha (af2);
	  pushi (singleaffix);
	  pusha (af1);
	  pushi (singleaffix);
	  pushi (3);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 6));		/* mul nodenr */
	  pushq (make_predicatenode);
	  callq ();
	  pop (10);
	  detach_valuenode (af1 -> value);
	  free_affixnode (af1);
	  detach_valuenode (af2 -> value);
	  free_affixnode (af2);
	  detach_valuenode (af3 -> value);
	  free_affixnode (af3);
	  pushq (pred_mul);
	};

/*
   pred_div
*/
private void act_div (posnode ps1, posnode ps2, posnode ps3)
	{ valuenode v1 = calc_affix_value (ps1, lower_side);
	  valuenode v2 = calc_affix_value (ps2, lower_side);
	  if ((v1 -> type == numbertype) && (v2 -> type == numbertype))
	     { int new = v1 -> v.number / v2 -> v.number;
	       valuenode v3 = number_to_value (new);
	       pushp (ps3);
	       pushv (v3);
	       pushq (propagate_predicate_value);
	       callq ();
	       pop (3);
	       detach_valuenode (v3);
	     };
	  detach_valuenode (v1);
	  detach_valuenode (v2);
	};

private void delayed_div (posnode *ps)
	{ posnode ps1 = ps[0];
	  posnode ps2 = ps[1];
	  posnode ps3 = ps[2];
	  if (!position_is_defined (ps1) ||
	      !position_is_defined (ps2)) callq ();
	  else if (!position_has_value (ps1) ||
		   !position_has_value (ps2)) callq ();
	  else
	     { ps1 -> delayed = 0;
	       ps2 -> delayed = 0;
	       ps3 -> delayed = 0;
	       act_div (ps1, ps2, ps3);
	       ps3 -> delayed = 1;
	       ps2 -> delayed = 1;
	       ps1 -> delayed = 1;
	     }
	};

public void pred_div ()
	{ affixnode af1 = new_affixnode ("div_af1");
	  affixnode af2 = new_affixnode ("div_af2");
	  affixnode af3 = new_affixnode ("div_af3");
	  pushq (delayed_div);
	  pushq (make_node_delayed);
	  pusha (af3);
	  pushi (singleaffix);
	  pusha (af2);
	  pushi (singleaffix);
	  pusha (af1);
	  pushi (singleaffix);
	  pushi (3);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 7));		/* div nodenr */
	  pushq (make_predicatenode);
	  callq ();
	  pop (10);
	  detach_valuenode (af1 -> value);
	  free_affixnode (af1);
	  detach_valuenode (af2 -> value);
	  free_affixnode (af2);
	  detach_valuenode (af3 -> value);
	  free_affixnode (af3);
	  pushq (pred_div);
	};

/*
   pred_power
*/
private int int_power (int a, int b)
	{ int z = 1;
	  int tb = b;
	  int ta = a;
	  if (b < 0) return (0);
	  while (tb)
	     { if (tb & 1) z *= ta;
	       tb = tb >> 1;
	       ta = ta * ta;
	     };
	  return (z);
	};

private void act_power (posnode ps1, posnode ps2, posnode ps3)
	{ valuenode v1 = calc_affix_value (ps1, lower_side);
	  valuenode v2 = calc_affix_value (ps2, lower_side);
	  if ((v1 -> type == numbertype) && (v2 -> type == numbertype))
	     { int new = int_power (v1 -> v.number, v2 -> v.number);
	       valuenode v3 = number_to_value (new);
	       pushp (ps3);
	       pushv (v3);
	       pushq (propagate_predicate_value);
	       callq ();
	       pop (3);
	       detach_valuenode (v3);
	     };
	  detach_valuenode (v1);
	  detach_valuenode (v2);
	};

private void delayed_power (posnode *ps)
	{ posnode ps1 = ps[0];
	  posnode ps2 = ps[1];
	  posnode ps3 = ps[2];
	  if (!position_is_defined (ps1) ||
	      !position_is_defined (ps2)) callq ();
	  else if (!position_has_value (ps1) ||
		   !position_has_value (ps2)) callq ();
	  else
	     { ps1 -> delayed = 0;
	       ps2 -> delayed = 0;
	       ps3 -> delayed = 0;
	       act_power (ps1, ps2, ps3);
	       ps3 -> delayed = 1;
	       ps2 -> delayed = 1;
	       ps1 -> delayed = 1;
	     }
	};

public void pred_power ()
	{ affixnode af1 = new_affixnode ("div_af1");
	  affixnode af2 = new_affixnode ("div_af2");
	  affixnode af3 = new_affixnode ("div_af3");
	  pushq (delayed_power);
	  pushq (make_node_delayed);
	  pusha (af3);
	  pushi (singleaffix);
	  pusha (af2);
	  pushi (singleaffix);
	  pusha (af1);
	  pushi (singleaffix);
	  pushi (3);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 8));		/* power nodenr */
	  pushq (make_predicatenode);
	  callq ();
	  pop (10);
	  detach_valuenode (af1 -> value);
	  free_affixnode (af1);
	  detach_valuenode (af2 -> value);
	  free_affixnode (af2);
	  detach_valuenode (af3 -> value);
	  free_affixnode (af3);
	  pushq (pred_div);
	};

/*
   Conversion predicates: int to string (9), string to int (10), uppercase (11)
*/

/*
   pred_inttostring
*/
private void act_inttostring (posnode ps1, posnode ps2)
	{ valuenode v1 = calc_affix_value (ps1, lower_side);
	  if (v1 -> type == numbertype)
	     { char buf[12];
	       char *new;
	       valuenode v2;
	       sprintf (buf, "%d", v1 -> v.number);
	       new = addto_names (buf);
	       v2 = string_to_value (new);
	       pushp (ps2);
	       pushv (v2);
	       pushq (propagate_predicate_value);
	       callq ();
	       pop (3);
	       detach_valuenode (v2);
	     };
	  detach_valuenode (v1);
	};

private void delayed_inttostring (posnode *ps)
	{ posnode ps1 = ps[0];
	  posnode ps2 = ps[1];
	  if (!position_is_defined (ps1)) callq ();
	  else if (!position_has_value(ps1)) callq ();
	  else
	     { ps1 -> delayed = 0;
	       ps2 -> delayed = 0;
	       act_inttostring (ps1, ps2);
	       ps2 -> delayed = 1;
	       ps1 -> delayed = 1;
	     }
	};

public void pred_inttostring ()
	{ affixnode af1 = new_affixnode ("inttostring_af1");
	  affixnode af2 = new_affixnode ("inttostring_af2");
	  pushq (delayed_inttostring);
	  pushq (make_node_delayed);
	  pusha (af2);
	  pushi (singleaffix);
	  pusha (af1);
	  pushi (singleaffix);
	  pushi (2);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 9));		/* inttostring nodenr */
	  pushq (make_predicatenode);
	  callq ();
	  pop (10);
	  detach_valuenode (af1 -> value);
	  free_affixnode (af1);
	  detach_valuenode (af2 -> value);
	  free_affixnode (af2);
	  pushq (pred_inttostring);
	};

/*
   pred_stringtoint
*/
private void act_stringtoint (posnode ps1, posnode ps2)
	{ valuenode v1 = calc_affix_value (ps1, lower_side);
	  if (v1 -> type == stringtype)
	     { char *ptr = v1 -> v.string;
	       if (isdigit (*ptr))
		  { int value = 0;
		    do
		       { int digit = *ptr - '0';
			 value = 10 * value + digit;
			 ptr++;
		       }
		    while (isdigit (*ptr));
		    if (!(*ptr))
		       { valuenode v2 = number_to_value (value);
			 pushp (ps2);
			 pushv (v2);
			 pushq (propagate_predicate_value);
			 callq ();
			 pop (3);
			 detach_valuenode (v2);
		       };
		  };
	     };
	  detach_valuenode (v1);
	};

private void delayed_stringtoint (posnode *ps)
	{ posnode ps1 = ps[0];
	  posnode ps2 = ps[1];
	  if (!position_is_defined (ps1)) callq ();
	  else if (!position_has_value(ps1)) callq ();
	  else
	     { ps1 -> delayed = 0;
	       ps2 -> delayed = 0;
	       act_stringtoint (ps1, ps2);
	       ps2 -> delayed = 1;
	       ps1 -> delayed = 1;
	     }
	};

public void pred_stringtoint ()
	{ affixnode af1 = new_affixnode ("stringtoint_af1");
	  affixnode af2 = new_affixnode ("stringtoint_af2");
	  pushq (delayed_stringtoint);
	  pushq (make_node_delayed);
	  pusha (af2);
	  pushi (singleaffix);
	  pusha (af1);
	  pushi (singleaffix);
	  pushi (2);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 10));	/* stringtoint nodenr */
	  pushq (make_predicatenode);
	  callq ();
	  pop (10);
	  detach_valuenode (af1 -> value);
	  free_affixnode (af1);
	  detach_valuenode (af2 -> value);
	  free_affixnode (af2);
	  pushq (pred_stringtoint);
	};

/*
   predicate uppercase
   converts all lowercase letters into uppercase
*/
private void act_uppercase (posnode ps1, posnode ps2)
	{ valuenode v1 = calc_affix_value (ps1, lower_side);
	  if (v1 -> type == stringtype)
	     { char *ptr1, *ptr2, *new;
	       valuenode v2;
	       for (ptr1 = v1 -> v.string, ptr2 = strstore;
		    *ptr1; ptr1++, ptr2++)
		  if (islower (*ptr1)) *ptr2 = toupper (*ptr1);
		  else *ptr2 = *ptr1;
	       *ptr2 = '\0';
	       new = addto_names (strstore);
	       v2 = string_to_value (new);
	       pushp (ps2);
	       pushv (v2);
	       pushq (propagate_predicate_value);
	       callq ();
	       pop (3);
	       detach_valuenode (v2);
	     };
	  detach_valuenode (v1);
	};

private void delayed_uppercase (posnode *ps)
	{ posnode ps1 = ps[0];
	  posnode ps2 = ps[1];
	  if (!position_is_defined (ps1)) callq ();
	  else if (!position_has_value(ps1)) callq ();
	  else
	     { ps1 -> delayed = 0;
	       ps2 -> delayed = 0;
	       act_uppercase (ps1, ps2);
	       ps2 -> delayed = 1;
	       ps1 -> delayed = 1;
	     }
	};

public void pred_uppercase ()
	{ affixnode af1 = new_affixnode ("uppercase_af1");
	  affixnode af2 = new_affixnode ("uppercase_af2");
	  pushq (delayed_uppercase);
	  pushq (make_node_delayed);
	  pusha (af2);
	  pushi (singleaffix);
	  pusha (af1);
	  pushi (singleaffix);
	  pushi (2);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 11));	/* uppercase nodenr */
	  pushq (make_predicatenode);
	  callq ();
	  pop (10);
	  detach_valuenode (af1 -> value);
	  free_affixnode (af1);
	  detach_valuenode (af2 -> value);
	  free_affixnode (af2);
	  pushq (pred_uppercase);
	};

/*
   predicate not in reserved word list (12)
   checks whether its first argument is not present in its second
*/
private int string_in_value (char *string, valuenode val)
	{ switch (val -> type)
	     { case stringtype:
		  return (strcmp (string, val -> v.string) == 0);
	       case compostype:
		  { int i;
		    for (i=0; i < val -> v.co.nr; i++)
		       if (string_in_value (string, val -> v.co.vals[i]))
			  return (1);
		    return (0);
		  };
	       default: return (0);
	     };
	};

private void act_notinreservedwordlist (posnode ps1, posnode ps2)
	{ valuenode v1 = calc_affix_value (ps1, lower_side);
	  valuenode v2 = calc_affix_value (ps2, lower_side);
	  if ((v1 -> type == stringtype) && (v2 -> type == compostype))
	     { char *string = v1 -> v.string; 
	       if (!string_in_value (string, v2)) callq ();
	     };
	  detach_valuenode (v1);
	  detach_valuenode (v2);
	};

private void delayed_notinreservedwordlist (posnode *ps)
	{ posnode ps1 = ps[0];
	  posnode ps2 = ps[1];
	  if (!position_is_defined (ps1) ||
	      !position_is_defined (ps2)) callq ();
	  else if (!position_has_value (ps1) ||
		   !position_has_value (ps2)) callq ();
	  else
	     { ps1 -> delayed = 0;
	       ps2 -> delayed = 0;
	       act_notinreservedwordlist (ps1, ps2);
	       ps2 -> delayed = 1;
	       ps1 -> delayed = 1;
	     }
	};

public void pred_notinreservedwordlist ()
	{ affixnode af1 = new_affixnode ("notinreservedwordlist_af1");
	  affixnode af2 = new_affixnode ("notinreservedwordlist_af2");
	  pushq (delayed_notinreservedwordlist);
	  pushq (make_node_delayed);
	  pusha (af2);
	  pushi (singleaffix);
	  pusha (af1);
	  pushi (singleaffix);
	  pushi (2);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 12));	/* nodenr */
	  pushq (make_predicatenode);
	  callq ();
	  pop (10);
	  detach_valuenode (af1 -> value);
	  free_affixnode (af1);
	  detach_valuenode (af2 -> value);
	  free_affixnode (af2);
	  pushq (pred_notinreservedwordlist);
	};

/*
   predicate dumpaffix (13)
*/
private void act_dumpaffix (posnode ps1)
	{ valuenode v1 = calc_affix_value (ps1, lower_side);
	  dump_affixvalue_on_file (v1, stderr);
	  detach_valuenode (v1);
	};

private void delayed_dumpaffix (posnode *ps)
	{ posnode ps1 = ps[0];
	  if (!position_is_defined (ps1)) callq ();
	  else if (!position_has_value(ps1)) callq ();
	  else
	     { ps1 -> delayed = 0;
	       act_dumpaffix (ps1);
	       ps1 -> delayed = 1;
	     }
	};

public void pred_dumpaffix ()
	{ affixnode af1 = new_affixnode ("dumpaffix_af1");
	  pushq (delayed_dumpaffix);
	  pushq (make_node_delayed);
	  pusha (af1);
	  pushi (singleaffix);
	  pushi (1);
	  pushi (0);
	  pushi (mk_nodenr (stddefs_modnr, 13));	/* dumpaffix nodenr */
	  pushq (make_predicatenode);
	  callq ();
	  pop (10);
	  detach_valuenode (af1 -> value);
	  free_affixnode (af1);
	  pushq (pred_dumpaffix);
	};

/*
   Module table of nodenrs
*/
public char *stddefs_name_from_nodenr (int nodenr)
	{ int lnodenr = lnodenr_from_nodenr (nodenr);
	  switch (lnodenr)
	     { case 0: return ("semipred_endofsentence");
	       case 1: return ("semipred_column");
	       case 2: return ("semipred_row");
	       case 3: return ("pred_equal");
	       case 4: return ("pred_notequal");
	       case 5: return ("pred_minus");
	       case 6: return ("pred_mul");
	       case 7: return ("pred_div");
	       case 8: return ("pred_power");
	       case 9: return ("pred_inttostring");
	       case 10: return ("pred_stringtoint");
	       case 11: return ("pred_uppercase");
	       case 12: return ("pred_notinreservedwordlist");
	       case 13: return ("pred_dumpaffix");
	       default: panic ("strange nodenr %d in stddefs\n", lnodenr);
	     };
	  return (NULL);
	};

public void init_stddefs (int modnr)
	{ stddefs_modnr = modnr;
	};
