/*
   File: topdown.c
   Defines a topdown recursive backup parser generator
*/

/* global includes */
#include <stdio.h>

/* libeag includes */
#include <export.h>
#include <error.h>

/* local includes */
#include <sizes.h>
#include <tree.h>
#include <numbering.h>
#include <main.h>
#include <common.h>
#include <codemeta.h>
#include <editor.h>
#include <gentemplates.h>
#include <typecheck.h>
#include <topdown.h>

/*
   Forward declare all generated rules
*/
private void generate_rule_declaration (hyper_rule rule)
	{ if (!rule -> reachable) return;
	  fprintf (out, "%s void %s_%s ();\n",
		rule -> ext?"import":"private",
		rule_qualifier (rule), rule -> nonterminal);
	};

private void generate_rule_declarations ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     generate_rule_declaration (all_hyper_rules[i]);
	};

/*
   Generate code for parsing members
*/
private void generate_call (hyper_rule rule, call c, int *sonnr, int *nr_pushes)
	{ int nrofps;
	  if (layoutflag && (c -> def == layout_rule))
	     { fprintf (out, "\t  pushq (rule_%s);\n", c -> nonterminal);
	       *nr_pushes += 1;
	       return;
	     };
	  if (rule -> kind & h_predicate)
	     { fprintf (out, "\t  pushi (%d);\n", *sonnr);
	       fprintf (out, "\t  pusht (pnode);\n");
	       fprintf (out, "\t  pushq (link_predicateson);\n");
	       *nr_pushes += 3;
	     }
	  else 
	     { fprintf (out, "\t  pushi (%d);\n", *sonnr);
	       fprintf (out, "\t  pushq (link_son);\n");
	       *nr_pushes += 2;
	     };
	  nrofps = (c -> display == pos_list_nil)?0:c -> display -> nrofps;
	  if (nrofps)
	     { generate_display (c -> display, nr_pushes);
	       fprintf (out, "\t  pushq (make_affix_link);\n");
	       *nr_pushes += 1;
	     };
	  fprintf (out, "\t  pushq (%s_%s);\n", rule_qualifier (c -> def),
		c -> nonterminal);
	  *nr_pushes += 1;
	  *sonnr -= 1;
	};

private void generate_member (hyper_rule rule, member m,
			      int *sonnr, int *nr_pushes, int code_build)
	{ switch (m -> tag)
	     { case tag_call:
		  generate_call (rule, m -> u.cl, sonnr, nr_pushes); break;
	       case tag_terminal:
		  generate_terminal (m -> u.terminal, nr_pushes); break;
	       case tag_semiterminal:
		  generate_semiterminal (m -> u.semi, sonnr,
			nr_pushes, code_build); break;
	       case tag_cut:
		  generate_cut (nr_pushes);
	       default: break;
	     };
	};

private void generate_rhs (hyper_rule rule, member_list mems,
			   int nrsons, int *nr_pushes, int code_build)
	{ int i;
	  int sonnr = nrsons;
	  if (mems == member_list_nil) return;
	  for (i=mems -> nrofms - 1; 0 <= i; i--)
	     generate_member (rule, mems -> ms[i], &sonnr,
			nr_pushes, code_build);
	};

private void generate_rule_lhs (hyper_rule rule, alt a, int *nr_pushes)
	{ int nrofps = (a -> display == pos_list_nil)?0:a -> display -> nrofps;
	  if (!(rule -> kind & h_nonpredicate) || nrofps)
	     generate_display (a -> display, nr_pushes);
	  fprintf (out, "\t  pushi (%d);\n", a -> nrsons);
	  fprintf (out, "\t  pushi (%d);\n", a -> nodenr);
	  if (rule -> kind & h_nonpredicate)
	     fprintf (out, "\t  pushq (make_%snode);\n",
		(nrofps)?"normal":"simple");
	  else if (rule -> kind & h_semipredicate)
	     fprintf (out, "\t  pushq (make_semipredicatenode);\n");
	  *nr_pushes += 3;
	};

private int alt_contains_cut (member_list mems)
	{ int i;
	  if (mems == member_list_nil) return (0);
	  for (i=0; i < mems -> nrofms; i++)
	     if (mems -> ms[i] -> tag == tag_cut) return (1);
	  return (0);
	};

private void generate_alt_code (hyper_rule rule, int i, alt a, int *needs_leave)
	{ int nrpushes = 0;
	  int code_build = !layoutflag || (rule != layout_rule);
	  int cut = alt_contains_cut (a -> members);
	  if (rule -> kind & h_nonpredicate)
	     may_generate_lookahead_check (rule, a -> director);
	  generate_alt_header (rule, i, a, code_build, cut);
	  generate_rhs (rule, a -> members, a -> nrsons,
			&nrpushes, code_build);
	  if (code_build) generate_rule_lhs (rule, a, &nrpushes);
	  generate_alt_trailer (rule, i, a, nrpushes, code_build, cut);
	  if (cut) *needs_leave = 1;
	};

private void generate_alts_code (hyper_rule rule, alt_list alts,
				 int *needs_leave)
	{ int i;
	  for (i=0; i < alts -> nrofas; i++)
	     generate_alt_code (rule, i, alts -> as[i], needs_leave);
	};

private void generate_rule_header (hyper_rule rule)
	{ fprintf (out, "private void %s_%s ()\n", rule_qualifier (rule),
			rule -> nonterminal);
	  fprintf (out, "\t{\n");
	};

private void may_generate_trace_enter (hyper_rule rule)
	{ if (traceflag)
	     fprintf (out, "\t trace_enter (\"%s_%s\");\n",
			rule_qualifier (rule), rule -> nonterminal);
	};

private void generate_rule_trailer (hyper_rule rule, int needs_leave)
	{ if (needs_leave)
	     fprintf (out, "leave_%s:\n", rule -> nonterminal);
	  if (traceflag)
	     fprintf (out, "\t trace_leave (\"%s_%s\");\n",
			rule_qualifier (rule), rule -> nonterminal);
	  fprintf (out, "\t  pushq(%s_%s);\n", rule_qualifier (rule),
			rule -> nonterminal);
	  fprintf (out, "\t};\n\n");
	};

/*
   The following routines generate code to parse placeholders,
*/
private void generate_untyped_placeholder_code (hyper_rule rule)
	{ int nrpushes = 4;
	  generate_placeholder_alt_header (rule, 1, untyped_symbol[0]);
	  generate_buildplaceholdernode (rule, 1, &nrpushes);
	  fprintf (out, "\t  pushq (rule_layout);\n");
	  fprintf (out, "\t  pushs (\"%s\");\n", untyped_symbol);
	  fprintf (out, "\t  pushq (parse_terminal);\n");
	  fprintf (out, "\t  pushq (rule_layout);\n");
	  generate_placeholder_alt_trailer (rule, 1, nrpushes);
	};

private void generate_typed_placeholder_code (hyper_rule rule)
	{ int nrpushes = 8;
	  generate_placeholder_alt_header (rule, 0, typed_open_symbol[0]);
	  generate_buildplaceholdernode (rule, 0, &nrpushes);
	  fprintf (out, "\t  pushq (rule_layout);\n");
	  fprintf (out, "\t  pushs (\"%s\");\n", typed_close_symbol);
	  fprintf (out, "\t  pushq (parse_terminal);\n");
	  fprintf (out, "\t  pushs (\"%s\");\n", rule -> nonterminal);
	  fprintf (out, "\t  pushq (parse_terminal);\n");
	  fprintf (out, "\t  pushs (\"%s\");\n", typed_open_symbol);
	  fprintf (out, "\t  pushq (parse_terminal);\n");
	  fprintf (out, "\t  pushq (rule_layout);\n");
	  generate_placeholder_alt_trailer (rule, 0, nrpushes);
	};

private void generate_placeholder_code (hyper_rule rule)
	{ if (!rule -> placeholder) return;
	  generate_untyped_placeholder_code (rule);
	  generate_typed_placeholder_code (rule);
	};

private void generate_rule_topdown (hyper_rule rule)
	{ int needs_leave = 0;
	  generate_rule_header (rule);
	  may_generate_trace_enter (rule);
	  if (placeholderflag) generate_placeholder_code (rule);
	  generate_alts_code (rule, rule -> alts, &needs_leave);
	  generate_rule_trailer (rule, needs_leave);
	};

/*
   Generate code for predicates
*/
private void generate_predicate_alt_code (hyper_rule rule, int i, alt a,
					  int *needs_leave)
	{ int nr_pushes = 0;
	  int cut = alt_contains_cut (a -> members);
	  generate_alt_header (rule, i, a, 1, cut);
	  generate_rhs (rule, a -> members, a -> nrsons, &nr_pushes, 1);
	  generate_display (a -> display, &nr_pushes);
	  fprintf (out, "\t  pushpp(args);\n");
	  fprintf (out, "\t  pushi(%d);\n", a -> nrsons);
	  fprintf (out, "\t  pushi(%d);\n", a -> nodenr);
	  fprintf (out, "\t  pushq(update_predicatenode);\n");
	  generate_alt_trailer (rule, i, a, nr_pushes + 4, 1, cut);
	  if (cut) *needs_leave = 1;
	};

private void generate_actual_predicate_code (hyper_rule rule)
	{ int i;
	  int needs_leave = 0;
	  fprintf (out, "private void act_%s (posnode *args)\n",
			rule -> nonterminal);
	  if (rule -> has_sons)
	     fprintf (out, "\t{ treenode pnode = args[0] -> node;\n");
	  else fprintf (out, "\t{\n");
	  for (i=0; i < rule -> alts -> nrofas; i++)
	     generate_predicate_alt_code (rule, i, rule -> alts -> as[i],
				&needs_leave);
	  if (needs_leave) fprintf (out, "leave_%s:", rule -> nonterminal);
	  fprintf (out, "\t}\n\n");
	};

private void generate_check_inherited_affixes (char *cond, hyper_rule rule,
					       int nrps, char *check)
	{ int first = 1;
	  int i;
	  fprintf (out, "\t  %s", cond);
	  for (i=0; i < nrps; i++)
	     { pos p = rule -> proto_display -> ps[i];
	       if (p -> kind == inherited)
		  { fprintf (out, "%s!%s(paf%d)",
			(first)?"":" || ", check, i);
	            first = 0;
		  };
	     };
	  fprintf (out, ")\n\t     callq();\n");
	};

private void generate_delayed_predicate_code (hyper_rule rule)
	{ int i;
	  int nrps = rule -> proto_display -> nrofps;
	  fprintf (out, "private void delayed_%s (posnode *args)\n",
			rule -> nonterminal);
	  fprintf (out, "\t{\n");
	  for (i=0; i < nrps; i++)
	     fprintf (out, "\t  posnode paf%d = args[%d];\n", i, i);
	  generate_check_inherited_affixes
		("if (", rule, nrps, "position_is_defined");
	  generate_check_inherited_affixes
		("else if (", rule, nrps, "position_has_value");
	  fprintf (out, "\t  else {\n");
	  for (i=0; i < nrps; i++)
	     fprintf (out, "\t     paf%d -> delayed = 0;\n", i);
	  fprintf (out, "\t     act_%s (args);\n", rule -> nonterminal);
	  for (i=0; i < nrps; i++)
	     fprintf (out, "\t     paf%d -> delayed = 1;\n", i);
	  fprintf (out, "\t  };\n");
	  fprintf (out, "\t}\n\n");
	};

private void generate_enter_predicate_code (hyper_rule rule)
	{ int i;
	  int nrps = rule -> proto_display -> nrofps;
	  generate_rule_header (rule);
	  for (i=0; i < nrps; i++)
	     fprintf (out,
	     "\t  affixnode paf%d = new_affixnode (\"pred_%s_paf%d\");\n",
		i, rule -> nonterminal, i);
	  may_generate_trace_enter (rule);
	  fprintf (out, "\t  pushq(delayed_%s);\n", rule -> nonterminal);
	  fprintf (out, "\t  pushq(make_node_delayed);\n");
	  for (i = nrps - 1; 0 <= i; i--)
	     { fprintf (out, "\t  pusha(paf%d);\n", i);
	       fprintf (out, "\t  pushi(singleaffix);\n");
	     };
	  fprintf (out, "\t  pushi(%d);\n", nrps);
	  fprintf (out, "\t  pushi(0);\n");
	  fprintf (out, "\t  pushi(%d);\n",
				rule -> alts -> as[0] -> nodenr - 1);
	  fprintf (out, "\t  pushq(make_predicatenode);\n");
	  fprintf (out, "\t  callq();\n");
	  fprintf (out, "\t  pop(%d);\n", 2 * nrps + 6);
	  for (i=0; i < nrps; i++)
	     { fprintf (out, "\t  detach_valuenode (paf%d -> value);\n", i);
	       fprintf (out, "\t  free_affixnode (paf%d);\n", i);
	     };
	  generate_rule_trailer (rule, 0);
	};

public void generate_predicate_topdown (hyper_rule rule)
	{ generate_actual_predicate_code (rule);
	  generate_delayed_predicate_code (rule);
	  generate_enter_predicate_code (rule);
	};

public void generate_semipredicate_topdown (hyper_rule rule)
	{ generate_rule_topdown (rule);
	};

/*
   Generate code for topdown parsers
*/
private void generate_rule (hyper_rule rule)
	{ if (rule -> ext) return;
	  if (!rule -> reachable) return;
	  if (rule -> kind & h_nonpredicate) generate_rule_topdown (rule);
	  else if (rule -> kind & h_predicate)
		generate_predicate_topdown (rule);
	  else if (rule -> kind & h_semipredicate)
		generate_semipredicate_topdown (rule);
	};

private void generate_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     generate_rule (all_hyper_rules[i]);
	};

public void generate_topdown_parser (char *fname, char **predefs, int nrpreds)
	{ warning ("generating topdown parser...");
	  open_output_file (fname, "topdown", "c");
	  generate_std_includes (predefs, nrpreds);
	  generate_meta_rules ();
	  generate_rule_declarations ();
	  code_nodenrs (fname);
	  if (editor) generate_enter_templates ();
	  warning ("coding syntax rules...");
	  generate_rules ();
	  warning ("coding postamble and main...");
	  generate_start_rule ();
	  generate_module_interface (fname, predefs, nrpreds);
	  generate_main (fname, predefs, nrpreds);
	  close_output_file ();
	};
