/*
   File: rules.c
   Generates the unparsing rules
*/

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

/* libeag includes */
#include <export.h>
#include <error.h>

/* libedt includes */
#include <editor.h>

/* local includes */
#include <tree.h>
#include <sizes.h>
#include <main.h>
#include <prepare.h>
#include <numbering.h>
#include <common.h>
#include <layout.h>
#include <rules.h>

private FILE *rfile;
private void code_rule (char *format, ...)
	{ char buf[MAXSTRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);
	  fprintf (rfile, buf);
	};

private void open_rule_file (char *gram)
	{ char totalname[MAXFNAME];
	  sprintf (totalname, "%s.rules", gram);
	  if (!(rfile = fopen (totalname, "w")))
	     panic ("can't open rulefile %s", totalname);
	  code_rule ("number of rules: %d\n", max_nodenr);
	  code_rule ("number of tuples: %d\n", max_members_per_alt);
	};
#define close_rule_file() fclose (rfile)

private void generate_member (member m)
	{ if (is_an_invisible_member (m)) return;
	  switch (m -> tag)
	     { case tag_call:
		  code_rule ("%s%s%s", typed_open_symbol,
			     m -> u.cl -> nonterminal, typed_close_symbol);
		  break;
	       case tag_terminal:
		  output_string (rfile, m -> u.terminal);
		  break;
	       case tag_semiterminal: code_rule ("{}");
	       default: break;
	     };
	};

private int rule_is_unoriented (alt a)
	{ int i;
	  member_list mems = a -> members;
	  if (mems == member_list_nil) return (1);
	  for (i = 0; i < mems -> nrofms; i++)
	     if (mems -> ms[i] -> followlayout) return (0);
	  return (1);
	};

private void generate_unoriented_rule (alt a)
	{ member_list mems = a -> members;
	  int i;
	  code_rule ("::\n");
	  if (mems != member_list_nil)
	     for (i=0; i < mems -> nrofms; i++) 
	        generate_member (mems -> ms[i]);
	  code_rule ("\n");
	};

private void generate_default_horizontal_rule (member_list mems)
	{ int i;
	  for (i=0; i < mems -> nrofms; i++)
	     { generate_member (mems -> ms[i]);
	       if (mems -> ms[i] -> followlayout) code_rule (".");
	     };
	};

private void generate_default_vertical_rule (member_list mems)
	{ int i;
	  for (i=0; i < mems -> nrofms; i++)
	     { generate_member (mems -> ms[i]);
	       if (mems -> ms[i] -> followlayout) code_rule ("\n");
	     };
	};

/*
   Heuristic rules for horizontal layout only kill layout.
   We therefore unmark these members and then generate
   according to the nonheuristic rules. This also has
   the wanted side effect in changing the corresponding
   vertical rule and template.
*/
private int heuristic_horizontal_rule (member_list mems)
	{ int i;
	  member m;
	  for (i=0; i < mems -> nrofms; i++)
	     if (!is_an_invisible_member (mems -> ms[i])) break;
	  m = mems -> ms[i];
	  if (m -> tag != tag_terminal) return (0);
	  if (strlen (m -> u.terminal) != 1) return (0);
	  m -> followlayout = 0;
	  for (i++; i < mems -> nrofms; i++)
	     { int j;
	       if (mems -> ms[i] -> followlayout)
	       for (j = i+1; j < mems -> nrofms; j++)
		  { m = mems -> ms[j];
		    if ((m -> tag == tag_terminal) &&
			(strlen (m -> u.terminal) == 1))
		       mems -> ms[i] -> followlayout = 0;
		    if (!is_an_invisible_member (m)) break;
		  };
	     };
	  generate_default_horizontal_rule (mems);
	  return (1);
	};

private void generate_horizontal_rule (alt a)
	{ member_list mems = a -> members;
	  code_rule (":H:\n");
	  if (!heuristic_horizontal_rule (mems))
	     generate_default_horizontal_rule (mems);
	  code_rule ("\n");
	};

/*
   A leading or trailing terminal of size >= 2 is treated as
   a bracket. These occur frequently in usual programming
   language constructs
*/
#define is_large_terminal(m)\
	(((m) -> tag == tag_terminal) && (strlen ((m) -> u.terminal) > 1))

private int alt_has_brackets (member_list mems)
	{ int i;
	  for (i=0; i < mems -> nrofms; i++)
	     { if (is_large_terminal (mems -> ms[i])) return (1);
	       if (!is_an_invisible_member (mems -> ms[i])) return (0);
	     };
	  for (i=mems -> nrofms - 1; 0 <= i; i--)
	     { if (is_large_terminal (mems -> ms[i])) return (1);
	       if (!is_an_invisible_member (mems -> ms[i])) return (0);
	     };
	  return (0);
	};

private int heuristic_vertical_rule (member_list mems)
	{ int i;
	  member prev;
	  if (!alt_has_brackets (mems)) return (0);
	  for (i=0; i < mems -> nrofms; i++)
	     if (!is_an_invisible_member (mems -> ms[i])) break;
	  prev = mems -> ms[i];
	  generate_member (prev);
	  for (i++; i < mems -> nrofms; i++)
	     { member cur = mems -> ms[i];
	       if (!is_an_invisible_member (cur))
		  { if (prev -> followlayout)
		       { if ((cur -> tag == tag_call) &&
			     (prev -> tag == tag_terminal))
			    code_rule ("\n.  ");
		         else code_rule ("\n");
		       };
		    generate_member (cur);
		    prev = cur;
		  };
	     };
	  return (1);
	};

private void generate_vertical_rule (alt a)
	{ member_list mems = a -> members;
	  code_rule (":V:\n");
	  if (!heuristic_vertical_rule (mems))
	     generate_default_vertical_rule (mems);
	  code_rule ("\n");
	};

private void generate_alt_type_and_sonnrs (alt a)
	{ int i;
	  member_list mems = a -> members;
	  code_rule ("type %d", a -> nodenr);
	  if (mems != member_list_nil)
	     for (i=0; i < mems -> nrofms; i++)
	        { member m = mems -> ms [i];
	          if (!is_an_invisible_member (m) &&
		      ((m -> tag == tag_call) ||
		       (m -> tag == tag_semiterminal)))
		     code_rule (", %d", m -> sonnr);
	        };
	  code_rule ("\n");
	};

private void generate_unparsing_rule_for_alt (alt a)
	{ generate_alt_type_and_sonnrs (a);
	  if (rule_is_unoriented (a)) generate_unoriented_rule (a);
	  else
	     { generate_horizontal_rule (a);
	       generate_vertical_rule (a);
	     };
	};

private void generate_unparsing_rule_for_rule (hyper_rule rule)
	{ int i;
	  if (!rule -> placeholder) return;
	  for (i=0; i < rule -> alts -> nrofas; i++)
	     generate_unparsing_rule_for_alt (rule -> alts -> as[i]);
	};

private void generate_unparsing_rule_for_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     generate_unparsing_rule_for_rule (all_hyper_rules[i]);
	};

private void generate_unparsing_rule_for_startrule ()
	{ code_rule ("type 0, 1\n");
	  code_rule ("::\n");
	  code_rule ("%s%s%s\n", typed_open_symbol,
			start_rule -> nonterminal, typed_close_symbol);
	};

public void generate_unparsing_rules (char *gram)
	{ warning ("generating unparsing rules...");
	  open_rule_file (gram);
	  generate_unparsing_rule_for_startrule ();
	  generate_unparsing_rule_for_rules ();
	  close_rule_file ();
	};
