/*
   File: main.c
   Defines main program
*/

/* Global includes */
#include <stdio.h>
#include <string.h>

/* libeag includes */
#include <export.h>
#include <error.h>
#include <memalloc.h>
#include <textstorage.h>

/* local includes */
#include <sizes.h>
#include <tree.h>
#include <parser.h>
#include <ident.h>
#include <typecheck.h>
#include <reduce.h>
#include <prepare.h>
#include <empty.h>
#include <eag_unparser.h>
#include <numbering.h>
#include <layout.h>
#include <lr.h>
#include <flow.h>
#include <templates.h>
#include <rules.h>
#include <lookahead.h>
#include <leftcorner.h>
#include <topdown.h>
#include <recursion.h>
#include <placeholder.h>
#include <meta.h>
#include <version.h>
#include <main.h>

public int showversion;
public int topdown;
public int leftcorner;
public int interface_to_c;
public int lr;
public int editor;
public int layoutflag;
public int placeholderflag;
public int indirect_templates;
public int traceflag;
public int matchflag;
public int dumpflag;
public int lookahead;
public int lookahead_error_messages;
public int qstackflag;

private void init_options ()
	{ showversion = 0;
	  topdown = 0;
	  leftcorner = 1;
	  interface_to_c = 0;
	  lr = 0;
	  editor = 0;
	  layoutflag = 0;
	  placeholderflag = 0;
	  indirect_templates = 0;
	  traceflag = 0;
	  matchflag = 0;
	  dumpflag = 0;
	  lookahead = 1;
	  lookahead_error_messages = 0;
	  qstackflag = 0;
	};

private char* basename_from_name (char *name)
	{ char basename[MAXFNAME];
	  char *sptr, *dptr;
	  for (sptr = name, dptr = basename;
	       (*sptr != '\0') && (*sptr != '.');
	       sptr++, dptr++)
	     *dptr = *sptr;
	  *dptr = '\0';
	  return (addto_names (basename));
	};

private FILE *open_input_file (char *name)
	{ char fname[MAXFNAME];
	  FILE *fd;

	  sprintf (fname, "%s.eag", name);
	  if ((fd = fopen (fname, "r"))) return (fd);
	  sprintf (fname, "%s.g", name);
	  if ((fd = fopen (fname, "r"))) return (fd);
	  sprintf (fname, "%s.b", name);
	  if ((fd = fopen (fname, "r"))) return (fd);
	  sprintf (fname, "%s", name);
	  if ((fd = fopen (fname, "r"))) return (fd);
	  panic ("could not open inputfile %s", name);
	  return ((FILE *) NULL);
	};

#ifndef PATH
#define PATH "/home/marcs/eag/include"
#endif
private FILE *open_predefines_file (char *name)
	{ char fname[MAXFNAME];
	  FILE *fd;

	  sprintf (fname, "./%s.eag", name);
	  if ((fd = fopen (fname, "r"))) return (fd);
	  sprintf (fname, "%s/%s.eag", PATH, name);
	  if ((fd = fopen (fname, "r"))) return (fd);
	  panic ("could not open predefines file %s.eag", name);
	  return ((FILE *) NULL);
	};

private void print_usage ()
	{ wlog ("usage: eag-compile [flags] [filename]");
	  wlog ("-h:  provide this help");
	  wlog ("-V:  show version");
	  wlog ("-v:  verbose");
	  wlog ("-fv: full verbose");
	  wlog ("-p [filename]: include filename as extra prelude");
	  wlog ("-ic: generate interface to C");
	  wlog ("-ed: generate editor");
	  wlog ("-td: generate topdown parser");
	  wlog ("-lc: generate leftcorner parser");
	  wlog ("-lr: generate lr parser (not yet implemented)");
	  wlog ("-nl: do not use lookahead");
	  wlog ("-gl: generate code to report error messages on lookahead failure");
	  wlog ("-il: shortcuts rule for layout");
	  wlog ("-pp: generate code to parse placeholders");
	  wlog ("-it: generate indirect templates");
	  wlog ("-D:  generate code to dump parse tree");
	  wlog ("-M:  generate code to count number of matches");
	  wlog ("-T:  generate code to enable tracing");
	  wlog ("-qc: generate code to check the runtime qstack");
	  exit (4);
	};

private void syntax_error (char *syn_error)
	{ error ("error on command line: %s", syn_error);
	  print_usage ();
	};

#define PREDFNAME "stddefs"
private char **predefs;
private int nr_of_predefs;
private void init_predefs (int argc)
	{ int max_nr = argc/2 + 1;		/* stddefs.eag */
	  predefs = (char **) ckcalloc (max_nr, sizeof (char *));
	  predefs [0] = PREDFNAME;
	  nr_of_predefs = 1;
	};

private void new_prelude (int *i, int argc, char **argv)
	{ *i += 1;
	  if (*i == argc) syntax_error ("missing prelude name");
	  predefs [nr_of_predefs] = argv[*i];
	  nr_of_predefs++;
	};

private void scan_option (char *ptr, int *i, int argc, char **argv)
	{ if (strcmp (ptr, "td") == 0)
	     { topdown = 1; leftcorner = 0; lr = 0; }
	  else if (strcmp (ptr, "lc") == 0)
	     { topdown = 0; leftcorner = 1; lr = 0; }
	  else if (strcmp (ptr, "lr") == 0)
	     { topdown = 0; leftcorner = 0; lr = 1; }
	  else if (strcmp (ptr, "ed") == 0)
	     { editor = 1; layoutflag = 1;
	       placeholderflag = 1; interface_to_c = 0; }
	  else if (strcmp (ptr, "it") == 0) indirect_templates = 1;
	  else if (strcmp (ptr, "il") == 0) layoutflag = 1;
	  else if (strcmp (ptr, "ic") == 0)
	     { interface_to_c = 1; editor = 0; }
	  else if (strcmp (ptr, "pp") == 0)
	     { layoutflag = 1; placeholderflag = 1; }
	  else if (strcmp (ptr, "p") == 0) new_prelude (i, argc, argv);
	  else if (strcmp (ptr, "h") == 0) print_usage ();
	  else if (strcmp (ptr, "V") == 0) showversion = 1;
	  else if (strcmp (ptr, "v") == 0) { showversion = 1; verbose = 1; }
	  else if (strcmp (ptr, "fv") == 0)
		{ showversion = 1; verbose = 1; full_verbose = 1; }
	  else if (strcmp (ptr, "T") == 0) traceflag = 1;
	  else if (strcmp (ptr, "M") == 0) matchflag = 1;
	  else if (strcmp (ptr, "D") == 0) dumpflag = 1;
	  else if (strcmp (ptr, "nl") == 0) lookahead = 0;
	  else if (strcmp (ptr, "gl") == 0) lookahead_error_messages = 1;
	  else if (strcmp (ptr, "qc") == 0) qstackflag = 1;
	  else syntax_error ("illegal option specified");
	};

private void report_version ()
	{ if (!showversion) return;
	  wlog ("This is EAG, C Version %d.%d, (C) University of Nijmegen",
		ReleaseNr, VersionNr);
	};

public void main (int argc, char **argv)
	{ FILE *in;
	  int i;
	  char *fname = NULL;
	  char *basename;
	  /* initialization */
	  init_error ();
	  init_options ();
	  init_predefs (argc);
	  init_textstorage ();
	  init_tree ();
	  /* command line parsing */
	  for (i=1; i < argc; i++)
	     { char *arg = argv[i];
	       if (arg[0] == '-') scan_option (arg+1, &i, argc, argv);
	       else if (fname == NULL) fname = arg;
	       else syntax_error ("too many filenames specified");
	     };
	  if (fname == NULL) { fname = "_stdin"; in = stdin; }
	  else in = open_input_file (fname);
	  basename = basename_from_name (fname);
	  report_version ();
	  /* input parsing: context free analysis */
	  for (i=0; i < nr_of_predefs; i++)
	     { FILE *pred = open_predefines_file (predefs [i]);
	       parse_eag (pred, 1);
	     };
	  parse_eag (in, 0);
	  /* context sensitive analysis */
	  do_numbering ();
	  identification ();
	  do_empty_classification ();
	  analyze_metagrammar ();
	  flow_check ();
	  type_check ();
	  /* prepare for parser generation */
	  reduce_grammar ();
	  check_recursion ();
	  if (placeholderflag) decide_placeholders ();
	  if (lookahead) determine_lookahead_sets ();
	  prepare_grammar ();
	  if (editor) analyze_layout ();
	  if (editor) generate_unparsing_rules (basename);
	  unparse_eag ();
	  if (topdown)
	     generate_topdown_parser (basename, predefs, nr_of_predefs);
	  else if (leftcorner)
	     generate_leftcorner_parser (basename, predefs, nr_of_predefs);
	  else if (lr) generate_lr_parser (basename, predefs, nr_of_predefs);
	  exit (0);
	};
