/*
   File: lookahead.c
   Determines FIRST and FOLLOW sets (on character basis)
   to form director sets which may limit backtracking.
*/

/* Global includes */
#include <stdio.h>
#include <string.h>

/* Libeag includes */
#include <export.h>
#include <error.h>
#include <textstorage.h>
#include <textparsing.h>

/* Libedt includes */
#include <editor.h>

/* Local includes */
#include <main.h>
#include <tree.h>
#include <numbering.h>
#include <lookahead.h>

/*
   Make an initial assumption about the first sets of rules
*/
private void initial_first_sets_in_terminal (member m)
	{ char first[2];
	  if (strlen (m -> u.terminal) != 0)
	     { first[0] = m -> u.terminal[0];
	       first[1] = '\0';
	       m -> first = addto_names (first);
	     }
	  else m -> first = "";
	};

private int char_in_set (char c, char *s)
	{ register char *sptr;
	  for (sptr = s; *sptr; sptr++)
	     if (c == *sptr) return (1);
	  return (0);
	};

private char *make_complementaryset (char *s)
	{ char buf[129];
	  char *dptr = buf;
	  int i;
	  for (i=1; i <= 127; i++)
	     if (!char_in_set (i, s))
		*dptr++ = i;
	  *dptr = '\0';
	  dptr = addto_names (buf);
	  return (dptr);
	};

private void initial_first_sets_in_semi (member m)
	{ set s = m -> u.semi -> s;
	  if (s -> kind & non) m -> first = make_complementaryset (s -> string);
	  else m -> first = s -> string;
	};

private void initial_first_sets_in_member (member m)
	{ switch (m -> tag)
	     { case tag_call: break;
	       case tag_terminal: initial_first_sets_in_terminal (m); break;
	       case tag_semiterminal: initial_first_sets_in_semi (m); break;
	       case tag_cut: m -> first = "";
	       default: break;
	     }
	};

private void initial_first_sets_in_members (member_list mems)
	{ int j;
	  if (mems == member_list_nil) return;
	  for (j=0; j < mems -> nrofms; j++)
	     initial_first_sets_in_member (mems -> ms[j]);
	};

private void initial_first_sets_in_alts (alt_list alts)
	{ int i;
	  for (i=0; i < alts -> nrofas; i++)
	     initial_first_sets_in_members (alts -> as[i] -> members);
	};

private void initial_first_sets_in_rule (hyper_rule rule)
	{ char buf[3];
	  if (rule -> ext) return;
	  if (rule -> kind & h_predicate) return;
	  if (rule -> kind & h_semipredicate) return;
	  initial_first_sets_in_alts (rule -> alts);
	  if (!rule -> placeholder) return;
	  buf[0] = untyped_symbol[0];
	  if (typed_open_symbol[0] != untyped_symbol[0])
	     { buf[1] = typed_open_symbol[0];
	       buf[2] = '\0';
	     }
	  else buf[1] = '\0';
	  rule -> first = addto_names (buf);
	};

private void initial_first_sets_in_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     initial_first_sets_in_rule (all_hyper_rules[i]);
	};

/*
   Incrementally add to the first sets of the rules
   until we reach a fixed point
*/
private void addcharto_set (char *dst, char c)
	{ char *dptr;
	  for (dptr = dst; *dptr; dptr++)
	     if (*dptr == c) return;
	  *dptr++ = c;
	  *dptr = '\0';
	};

private void addto_set (char *dst, char *src)
	{ char *sptr;
	  if (src == string_nil) return;
	  for (sptr = src; *sptr; sptr++)
	     addcharto_set (dst, *sptr);
	};

private void add_first_sets_for_call (char *dst, call c)
	{ hyper_rule def = c -> def;
	  addto_set (dst, def -> first);
	};

private void add_first_sets_for_member (char *dst, member m)
	{ switch (m -> tag)
	     { case tag_call: add_first_sets_for_call (dst, m -> u.cl); break;
	       case tag_terminal:
	       case tag_semiterminal: addto_set (dst, m -> first); break;
	       case tag_cut:
	       default: break;
	     };
	};

private void add_first_sets_for_members (char *dst, member_list mems)
	{ int i;
	  if (mems == member_list_nil) return;
	  for (i = 0; i < mems -> nrofms; i++)
	     { add_first_sets_for_member (dst, mems -> ms[i]);
	       if (mems -> ms[i] -> empty == h_neverproducesempty) break;
	     };
	};

private int change;
private void add_first_sets_for_alt (char *dst, alt a)
	{ char altset[256];
	  altset[0] = '\0';
	  addto_set (altset, a -> first);
	  add_first_sets_for_members (altset, a -> members);
	  if ((a -> first == string_nil) ||
	      (strcmp (a -> first, altset) != 0))
	     { a -> first = addto_names (altset);
	       change = 1;
	     };
	  addto_set (dst, a -> first);
	};

private void find_first_sets_for_rule (hyper_rule rule)
	{ char ruleset[256];
	  int i;
	  if (rule -> ext) return;
	  if (rule -> kind & h_predicate) return;
	  if (rule -> kind & h_semipredicate) return;
	  ruleset[0] = '\0';
	  addto_set (ruleset, rule -> first);
	  for (i=0; i < rule -> alts -> nrofas; i++)
	     add_first_sets_for_alt (ruleset, rule -> alts -> as[i]);
	  if ((rule -> first == string_nil) ||
	      (strcmp (rule -> first, ruleset) != 0))
	     { rule -> first = addto_names (ruleset);
	       change = 1;
	     };
	};

private void find_first_sets_for_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     find_first_sets_for_rule (all_hyper_rules[i]);
	};

private void determine_first_sets ()
	{ int nrpasses = 0;
	  warning ("determining first sets...");
	  initial_first_sets_in_rules ();
	  do
	     { change = 0;
	       nrpasses++;
	       find_first_sets_for_rules ();
	     }
	  while (change);
	  hint ("used %d passes for first sets", nrpasses);
	};

/*
   Make an initial assumption about the follow sets of all rules
*/
private void initiate_follow_sets ()
	{ char buf[2];
	  buf[0] = EOFCHAR;
	  buf[1] = '\0';
	  start_rule -> def -> follow = addto_names (buf);
	  if (!placeholderflag) return;
	  buf[0] = untyped_symbol[0];
	  if (typed_open_symbol[0] != buf[0])
	     { buf[1] = typed_open_symbol[0];
	       buf[2] = '\0';
	     }
	  else buf[1] = '\0';
	  lookup_hyper_rule ("layout") -> follow = addto_names (buf);
	};

private void add_first_of_member_to_follow_set (member m, char *s)
	{ switch (m -> tag)
	     { case tag_call:
		  addto_set (s, m -> u.cl -> def -> first); break;
	       case tag_terminal:
	       case tag_semiterminal: addto_set (s, m -> first); break;
	       case tag_cut:
	       default: break;
	     };
	};

private void add_first_of_members_to_follow_set (member_list mems, int from,
						 char *fset)
	{ int i;
	  for (i = from; i < mems -> nrofms; i++)
	     { add_first_of_member_to_follow_set (mems -> ms[i], fset);
	       if (mems -> ms[i] -> empty == h_neverproducesempty) return;
	     };
	};

private void try_add_rule_set_to_follow_set (hyper_rule rule, member_list mems,
					     int from, char *fset)
	{ int i;
	  for (i = from; i < mems -> nrofms; i++)
	     if (mems -> ms[i] -> empty == h_neverproducesempty) return;
	  addto_set (fset, rule -> follow);
	};

private void try_update_follow_set (hyper_rule rule, member_list mems,
				    int from, hyper_rule target)
	{ char fset[256];
	  fset[0] = '\0';
	  if ((target -> kind & h_predicate) ||
	      (target -> kind & h_semipredicate)) return;
	  addto_set (fset, target -> follow);	/* first yourself */
	  add_first_of_members_to_follow_set (mems, from, fset);
	  try_add_rule_set_to_follow_set (rule, mems, from, fset);
	  if (strcmp (target -> follow, fset) != 0)
	     { target -> follow = addto_names (fset);
	       change = 1;
	     };
	};

private void determine_follow_sets_in_members (hyper_rule rule,
					       member_list mems)
	{ int i;
	  if (mems == member_list_nil) return;
	  for (i=0; i < mems -> nrofms; i++)
	     if (mems -> ms[i] -> tag == tag_call)
	        try_update_follow_set (rule, mems, i + 1,
			mems -> ms[i] -> u.cl -> def);
	};

private void determine_follow_sets_in_alts (hyper_rule rule, alt_list alts)
	{ int i;
	  for (i=0; i < alts -> nrofas; i++)
	     determine_follow_sets_in_members (rule, alts -> as[i] -> members);
	};

private void determine_follow_sets_in_rule (hyper_rule rule)
	{ if ((rule -> kind & h_predicate) ||
	      (rule -> kind & h_semipredicate)) return;
	  determine_follow_sets_in_alts (rule, rule -> alts);
	};

private void determine_follow_sets_in_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     determine_follow_sets_in_rule (all_hyper_rules[i]);
	};

private void determine_follow_sets ()
	{ int nrpasses = 0;
	  warning ("determining follow sets...");
	  initiate_follow_sets ();
	  nrpasses = 0;
	  do
	     { change = 0;
	       nrpasses++;
	       determine_follow_sets_in_rules ();
	     }
	  while (change);
	  hint ("used %d passes for follow sets", nrpasses);
	};

/*
   Combine first and follow sets into director sets
*/
private void determine_director_sets_in_alt (hyper_rule rule, alt a)
	{ char fset[256];
	  fset [0] = '\0';
	  addto_set (fset, a -> first);
	  if (a -> empty == h_mayproduceempty)
	     addto_set (fset, rule -> follow);
	  a -> director = addto_names (fset);
	};

private void determine_director_set (hyper_rule rule)
	{ int i;
	  if ((rule -> kind & h_predicate) ||
		(rule -> kind & h_semipredicate)) return;
	  for (i = 0; i < rule -> alts -> nrofas; i++)
	     determine_director_sets_in_alt (rule, rule -> alts -> as[i]);
	};

private void determine_director_sets ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     determine_director_set (all_hyper_rules[i]);
	};

/*
   Report on director sets
*/
private void report_director_set (hyper_rule rule)
	{ int i;
	  wlog ("first (%s) = %s", rule -> nonterminal, rule -> first);
	  wlog ("follow (%s) = %s", rule -> nonterminal, rule -> follow);
	  for (i=0; i < rule -> alts -> nrofas; i++)
	     wlog ("direct_alt (%d) = %s", i, 
		   rule -> alts -> as[i] -> director);
	};

private void report_director_sets ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     report_director_set (all_hyper_rules[i]);
	};

public void determine_lookahead_sets ()
	{ determine_first_sets ();
	  determine_follow_sets ();
	  determine_director_sets ();
	  if (full_verbose) report_director_sets ();
	};

public char *gather_lcin_director_set (hyper_rule rule, member_list mems, int j)
	{ char fset[256];
	  fset[0] = '\0';
	  add_first_of_members_to_follow_set (mems, j + 1, fset);
	  try_add_rule_set_to_follow_set (rule, mems, j + 1, fset);
	  return (addto_names (fset));
	};
