/*
   File: ident.c
   identifies all hyper_rule applications and metadefined affix values
   checks the number of affix positions for every application
   identifies all local affixes, generates proper metadefinitions
   for affix sets occurring in hyper rules.
*/

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

/* libeag includes */
#include <export.h>
#include <error.h>
#include <textstorage.h>
#include <memalloc.h>

/* local includes */
#include <sizes.h>
#include <tree.h>
#include <gentree.h>
#include <numbering.h>
#include <ident.h>
#include <main.h>

/*
   set up error administration
*/
private int ident_errors;
private int generated_nr;
private void init_ident ()
	{ ident_errors = 0;
	  generated_nr = 0;
	};

private void ident_error (char *format, ...)
	{ char buf[MAXSTRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);

	  ident_errors++;
	  error ("identification error: %s", buf);
	};

/*
   Do hyper rule identification
*/
private void identify_call (hyper_rule rule, int i, int j, call c)
	{ hyper_rule callee = lookup_hyper_rule (c -> nonterminal);
	  c -> def = callee;
	  if (callee == hyper_rule_nil)
	     ident_error (
		    "in rule %s, alt %d, member %d: %s could not be identified",
		    rule -> nonterminal, i+1, j+1, c -> nonterminal);
	};

private void identify_in_member (hyper_rule rule, int i, int j, member m)
	{ switch (m -> tag)
	     { case tag_call: identify_call (rule, i, j, m -> u.cl); break;
	       case tag_terminal: return;
	       case tag_semiterminal: return;
	       case tag_cut: return;
	       default: panic ("strange member tag %d", m -> tag);
	     };
	};

private void identify_in_members (hyper_rule rule, int i, member_list ml)
	{ int j;
	  if (ml == member_list_nil) return;	/* empty alternative */
	  for (j=0; j < ml -> nrofms; j++)
	     identify_in_member (rule, i, j, ml -> ms[j]);
	};

private void deduce_nr_lhs_positions (hyper_rule rule, int i, pos_list pl)
	{ int nrofps;
	  if (pl == pos_list_nil) nrofps = 0;
	  else nrofps = pl -> nrofps;
	  if (rule -> proto_display == pos_list_nil)
	     rule -> proto_display = new_empty_pos_list (nrofps);
	  else if (rule -> proto_display -> nrofps != nrofps)
	     ident_error (
	          "rule %s, alt %d: inconsistent number of lhs affix positions",
		  rule -> nonterminal, i+1);
	};

private void identify_in_alt (hyper_rule rule, int i, alt a)
	{ deduce_nr_lhs_positions (rule, i, a -> display);
	  identify_in_members (rule, i, a -> members);
	};

private void identify_in_alts (hyper_rule rule, alt_list alts)
	{ int i;
	  for (i=0; i < alts -> nrofas; i++)
	     identify_in_alt (rule, i, alts -> as[i]);
	};

private void identify_in_hyper_rule (hyper_rule rule)
	{ identify_in_alts (rule, rule -> alts);
	};

private void identify_in_hyper_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     identify_in_hyper_rule (all_hyper_rules[i]);
	};

private void identify_start_rule ()
	{ start_rule -> def = lookup_hyper_rule (start_rule -> nonterminal);
	  if (start_rule -> def == hyper_rule_nil)
	     ident_error ("could not identify start rule");
	};

/*
   Promote affix terminals, semiterminals and numbers to
   anonymous affix variables with a (predefined) value.
   Also introduce an anonymous affix for a semiterminal
   without display
*/
private int affix_occurs_on_locals_list (alt a, affix affx, affix *fd_affx)
	{ affix ptr;
	  for (ptr = a -> locals; ptr != affix_nil; ptr = ptr -> next)
	     if (strcmp (affx -> name, ptr -> name) == 0)
		{ *fd_affx = ptr;
		  return (1);
	        };
	  return (0);
	};

private void gather_affix_variable (alt a, affix *affx)
	{ affix current = *affx;
	  char name[MAXSTRLEN+3];

	  strcpy (name, "a_");
	  strcat (name, current -> u.var -> variable);
	  current -> name = name;
	  if (affix_occurs_on_locals_list (a, current, affx)) ckfree (current);
	  else
	     { current -> name = addto_names (name);
	       current -> next = a -> locals;
	       a -> locals = current;
	     };
	};

private char *generate_anonymous_affixname ()
	{ char name[20];
	  sprintf (name, "a_gen_%d", generated_nr);
	  generated_nr++;
	  return (addto_names (name));
	};

private void gather_anonymous_affix (alt a, affix *affx)
	{ affix current = *affx;
	  char *name = generate_anonymous_affixname ();

	  if (current -> tag == tag_affix_set)
	     { /* introduce a proper meta definition */
	       meta_alt a2[1];
	       meta_alt_list ml2;
	       meta_rule newmeta;

	       affix_variable newvar = new_affix_variable (name);
	       affix newaffix = new_affix_nonterminal (newvar);
	       expr e2 = new_expr_single (current);

	       a2[0] = new_meta_alt (e2);
	       ml2 = new_meta_alt_list (1, a2);
	       newmeta = new_meta_rule (name, ml2);
	       (void) enter_meta_rule (newmeta);
	       newaffix -> name = name;
	       *affx = newaffix;
	       newaffix -> next = a -> locals;
	       a -> locals = newaffix;
	     }
	  else
	     { current -> name = name;
	       current -> next = a -> locals;
	       a -> locals = current;
	     };
	};

private pos_list gather_anonymous_display (alt a)
	{ expr newexpr;
	  pos npos;
	  pos_list new = new_pos_list ();

	  char *name = generate_anonymous_affixname ();
	  affix_variable newvar = new_affix_variable (name);
	  affix newaffix = new_affix_nonterminal (newvar);
	  newaffix -> name = name;
	  newaffix -> next = a -> locals;
	  a -> locals = newaffix;

	  newexpr = new_expr_single (newaffix);
	  npos = new_pos (noflow, newexpr);
	  app_pos_list (new, npos);
	  return (new);
	};

private void gather_affix (alt a, affix *affx)
	{ if ((*affx) -> tag == tag_affix_nonterminal)
	     gather_affix_variable (a, affx);
	  else gather_anonymous_affix (a, affx);
	};

private void gather_affix_list (alt a, affix_list affl)
	{ int i;
	  for (i = 0; i < affl -> nrofas; i++)
	     gather_affix (a, &affl -> as[i]);
	};

private void gather_affixes_in_expr (alt a, expr e)
	{ switch (e -> tag)
	     { case tag_single: gather_affix (a, &e -> u.single); break;
	       case tag_compos: gather_affix_list (a, e -> u.compos); break;
	       case tag_concat: gather_affix_list (a, e -> u.concat); break;
	     };
	};

private void gather_affixes_in_display (alt a, pos_list pl)
	{ int i;
	  if (pl == pos_list_nil) return;
	  for (i=0; i < pl -> nrofps; i++)
	     gather_affixes_in_expr (a, pl -> ps[i] -> ex);
	};

private void gather_affixes_in_semi (alt a, semiterminal sm)
	{ if (sm -> display -> nrofps)
	     gather_affixes_in_display (a, sm -> display);
	  else
	     sm -> display = gather_anonymous_display (a);
	};

private void gather_affixes_in_member (alt a, member m)
	{ switch (m -> tag)
	     { case tag_call:
		  gather_affixes_in_display (a, m -> u.cl -> display);
		  break;
	       case tag_terminal: break;
	       case tag_semiterminal:
		  gather_affixes_in_semi (a, m -> u.semi);
	       default: break;
	     };
	};

private void gather_affixes_in_members (alt a, member_list mems)
 	{ int i;
	  if (mems == member_list_nil) return;
	  for (i=0; i < mems -> nrofms; i++)
	     gather_affixes_in_member (a, mems -> ms[i]);
	};

private void gather_affixes_in_alt (alt a)
	{ gather_affixes_in_display (a, a -> display);
	  gather_affixes_in_members (a, a -> members);
	};

private void gather_affixes_in_alts (alt_list alts)
	{ int i;
	  for (i=0; i < alts -> nrofas; i++)
	     gather_affixes_in_alt (alts -> as[i]);
	};

private void gather_affixes_in_hyper_rule (hyper_rule rule)
	{ if (rule -> ext) return;
	  gather_affixes_in_alts (rule -> alts);
	};

private void gather_affixes_in_hyper_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     gather_affixes_in_hyper_rule (all_hyper_rules[i]);
	};

private void check_affix_variable (char *rule_name, affix_variable v, int meta)
	{ meta_rule meta_def = lookup_meta_rule (v -> variable);
	  v -> def = meta_def;
	  if (meta && (meta_def == meta_rule_nil))
	     ident_error ("meta_rule %s: affix %s is not metadefined",
			  rule_name, v -> variable);
	};

private void check_affix (char *rule_name, affix a, int meta)
	{ switch (a -> tag)
	     { case tag_affix_nonterminal:
		  check_affix_variable (rule_name, a -> u.var, meta);
		  break;
	       case tag_affix_terminal: break;
	       case tag_affix_number: break;
	       case tag_affix_set: break;
	     };
	};

private void check_affix_list (char *rule_name, affix_list al, int meta)
	{ int i;
	  for (i=0; i < al -> nrofas; i++)
	     check_affix (rule_name, al -> as[i], meta);
	};

private void check_expression (char *rule_name, expr e, int meta)
	{ if (e == expr_nil) return;		/* empty meta alt */
	  switch (e -> tag)
	     { case tag_single:
		  check_affix (rule_name, e -> u.single, meta); break;
	       case tag_concat:
		  check_affix_list (rule_name, e -> u.concat, meta); break;
	       case tag_compos:
		  check_affix_list (rule_name, e -> u.compos, meta); break;
	     };
	};

private void check_position (char *rule_name, pos p)
	{ check_expression (rule_name, p -> ex, 0);
	};

private void check_positions (char *rule_name, pos_list pl)
	{ int i;
	  if (pl == pos_list_nil) return;	/* empty display */
	  for (i=0; i < pl -> nrofps; i++)
	     check_position (rule_name, pl -> ps[i]);
	};

private void check_positions_in_call (char *rule_name, int i, call c)
	{ hyper_rule callee = c -> def;
	  pos_list pl = c -> display;
	  int nrofps = (pl == pos_list_nil)?0:pl -> nrofps;

	  if (callee == hyper_rule_nil) return;	/* unidentifiable rule */
	  if (nrofps != callee -> proto_display -> nrofps) 
	     ident_error (
      "rule %s, alt %d: application of rule %s has wrong number of positions",
		rule_name, i+1, callee -> nonterminal);
	  check_positions (rule_name, pl);
	};

private void check_positions_in_semiterminal (char *rule_name, int i,
					      semiterminal s)
	{ if (s -> display -> nrofps != 1)	/* should be one */
	     ident_error (
		    "rule %s, alt %i: semiterminal has %d affix positions",
	       	    rule_name, i+1, s -> display -> nrofps);
	  check_positions (rule_name, s -> display);
	};

private void check_positions_in_member (char *rule_name, int i, member m)
	{ switch (m -> tag)
	     { case tag_call: check_positions_in_call (rule_name, i, m -> u.cl);
			      break;
	       case tag_terminal: break;
	       case tag_semiterminal:
		  check_positions_in_semiterminal (rule_name, i, m -> u.semi);
		  break;
	       case tag_cut: break;
	       default: panic ("strange member tag %d", m -> tag);
	     };
	};

private void check_positions_in_members (char *rule_name, int i, member_list ml)
	{ int j;
	  if (ml == member_list_nil) return;	/* empty alt */
	  for (j=0; j < ml -> nrofms; j++)
	     check_positions_in_member (rule_name, i, ml -> ms[j]);
	};

private void check_positions_in_alt (char *rule_name, int i, alt a)
	{ check_positions (rule_name, a -> display);	/* can't be wrong */
	  check_positions_in_members (rule_name, i, a -> members);
	};

private void check_positions_in_alts (char *rule_name, alt_list alts)
	{ int i;
	  for (i=0; i < alts -> nrofas; i++)
	     check_positions_in_alt (rule_name, i, alts -> as[i]);
	};

private void check_positions_in_hyper_rule (hyper_rule rule)
	{ check_positions_in_alts (rule -> nonterminal, rule -> alts);
	};

private void check_positions_in_hyper_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     check_positions_in_hyper_rule (all_hyper_rules[i]);
	};

private void check_start_rule_positions ()
	{ int nrofps, nrofps2;
	  if (start_rule -> def == hyper_rule_nil) return; /* unidentified */
	  nrofps = start_rule -> def -> proto_display -> nrofps;
	  nrofps2 = (start_rule -> display == pos_list_nil)?0:
			start_rule -> display -> nrofps;
	  if (nrofps != nrofps2)
	     ident_error ("start rule has a wrong number of affix positions");
	};

private void check_expressions_in_meta_alts (char *rule_name, meta_alt_list al)
	{ int i;
	  if (al == meta_alt_list_nil) return; 	/* predefined metarule */
	  for (i=0; i<al -> nrofas; i++)
	     check_expression (rule_name, al -> as[i] -> e, 1);
	};

private void check_expressions_in_meta_rule (meta_rule rule)
	{ check_expressions_in_meta_alts (rule -> meta_nonterminal,
					  rule -> meta_alts);
	};

private void check_expressions_in_meta_rules ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     check_expressions_in_meta_rule (all_meta_rules[i]);
	};

private void ident_layout_member (member m)
	{ if (m -> tag == tag_call)
	     ident_error ("rule layout contains call");
	};

private void ident_layout_members (member_list mems)
	{ int i;
	  if (mems == member_list_nil) return;
	  for (i=0; i < mems -> nrofms; i++)
	     ident_layout_member (mems -> ms[i]);
	};

private void ident_layout_alts (alt_list alts)
	{ int i;
	  for (i=0; i < alts -> nrofas; i++)
	     ident_layout_members (alts -> as[i] -> members);
	};

private void identify_layout_rule ()
	{ layout_rule = lookup_hyper_rule ("layout");
	  if (!layout_rule)
	     ident_error ("rule layout could not be identified");
	  else if (layout_rule -> proto_display -> nrofps)
	     ident_error ("rule layout has affixes");
	  else ident_layout_alts (layout_rule -> alts);
	};

public void identification ()
	{ warning ("identification...");
	  init_ident ();
	  identify_in_hyper_rules ();
	  identify_start_rule ();
	  gather_affixes_in_hyper_rules ();
	  gather_affixes_in_alt (start_alt);
	  check_positions_in_hyper_rules ();
	  check_start_rule_positions ();
	  check_expressions_in_meta_rules ();
	  hint ("generated %d anonymous affix%s",
		generated_nr, (generated_nr == 1)?"":"es");
	  if (layoutflag) identify_layout_rule ();
	  if (ident_errors == 0) return;
	  panic ("%d identification error%s w%s found",
		 ident_errors, (ident_errors == 1)?"":"s",
		 (ident_errors == 1)?"as":"ere");
	};
