/*
   File: empty.c
   classifies the hyper_rules into predicates, non predicates and
   rules starting with a semi terminal.
*/

/* global includes */
#include <stdio.h>

/* libeag includes */
#include <export.h>
#include <error.h>

/* local includes */
#include <tree.h>
#include <numbering.h>
#include <empty.h>
#include <main.h>

private void initial_terminal_classification (member m)
	{ if (strlen (m -> u.terminal) != 0) m -> empty = h_neverproducesempty;
	  else m -> empty = h_alwaysproducesempty;
	};

private void initial_semiterminal_classification (member m)
	{ set s = m -> u.semi -> s;
	  if ((strlen (s -> string) == 0) && !(s -> kind & non))
	     m -> empty = h_alwaysproducesempty;
	  else if (s -> kind & star) m -> empty = h_mayproduceempty;
	  else m -> empty = h_neverproducesempty;
	};

private void initial_cut_classification (member m)
	{ m -> empty = h_alwaysproducesempty;
	};

private void initial_member_classification (hyper_rule rule, member m)
	{ switch (m -> tag)
	     { case tag_call: return;
	       case tag_terminal:
		  initial_terminal_classification (m); break;
	       case tag_semiterminal:
		  initial_semiterminal_classification (m); break;
	       case tag_cut:
		  initial_cut_classification (m); break;
	       default: break;
	     };
	  rule -> kind = h_nonpredicate;
	};

private void initial_alt_classification (hyper_rule rule, alt a)
	{ member_list mems = a -> members;
	  if (mems == member_list_nil) a -> empty = h_mayproduceempty;
	  else
	     { int i;
	       for (i=0; i < mems -> nrofms; i++)
		  initial_member_classification (rule, mems -> ms[i]);
	     };
	  if (a -> empty) rule -> empty = h_mayproduceempty;
	};

private void initial_rule_classification (hyper_rule rule)
	{ alt_list alts = rule -> alts;
	  int i;

	  if (rule -> ext) rule -> empty = h_alwaysproducesempty;
	  else
	     for (i=0; i < alts -> nrofas; i++)
	        initial_alt_classification (rule, alts -> as[i]);
	};

private void do_initial_rule_classification ()
	{ int ix;
	  hint ("initial rule classification...");
	  for (ix = 0; ix < nr_of_hyper_rules; ix++)
	     initial_rule_classification (all_hyper_rules [ix]);
	};

private int change;
private void unify_empty (hyper_rule rule, int empty)
	{ if (rule -> empty) return;
	  if (!empty) return;
	  rule -> empty = h_mayproduceempty;
	  change = 1;
	};

private int member_may_produce_empty (member m)
	{ if (m -> tag == tag_call)
	     { int empty = m -> u.cl -> def -> empty;
	       if (empty == h_mayproduceempty) return (1);
	       return (empty == h_alwaysproducesempty);
	     }
	  else return (m -> empty != h_neverproducesempty);
	};

private int alt_may_produce_empty (alt a)
	{ int i;
	  member_list mems = a -> members;
	  if (mems == member_list_nil) return (1);
	  for (i=0; i < mems -> nrofms; i++)
	     if (!member_may_produce_empty (mems -> ms[i])) return (0);
	  return (1);
	};

private void detect_if_rule_produces_empty (hyper_rule rule)
	{ int i;
	  alt_list alts = rule -> alts;

	  if (rule -> ext) return;
	  for (i=0; i < alts -> nrofas; i++)
	     unify_empty (rule, alt_may_produce_empty (alts -> as[i]));
	};

private void detect_empty_producing_rules ()
	{ int nr_passes;
	  int ix;
	  hint ("detecting empty producing rules...");
	  nr_passes = 0;
	  do { change = 0;
	       for (ix = 0; ix < nr_of_hyper_rules; ix++)
		  detect_if_rule_produces_empty (all_hyper_rules [ix]);
	       nr_passes++;
	     }
	  while (change);
	  hint ("needed %d pass%s for empty detection",
		nr_passes, (nr_passes == 1)?"":"es");
	};

private void mark_if_rule_never_produces_empty (hyper_rule rule)
	{ if (rule -> empty) return;
	  rule -> empty = h_neverproducesempty;
	  rule -> kind = h_nonpredicate;
	};

private int member_always_produces_empty (member m)
	{ if (m -> tag == tag_call)
	     { hyper_rule def = m -> u.cl -> def;
	       if (def -> empty == h_neverproducesempty) return (0);
	       return (def -> kind != h_nonpredicate);
	     }
	  else return (m -> empty == h_alwaysproducesempty);
	};

private int alt_always_produces_empty (alt a)
	{ int i;
	  member_list mems = a -> members;
	  if (mems == member_list_nil) return (1);

	  for (i=0; i < mems -> nrofms; i++)
	     if (!member_always_produces_empty (mems -> ms[i]))
		return (0);
	  return (1);
	};

private void detect_if_rule_always_produces_empty (hyper_rule rule)
	{ alt_list alts = rule -> alts;
	  int i;

	  if (rule -> ext) return;
	  if (rule -> kind == h_nonpredicate) return;
	  for (i=0; i < alts -> nrofas; i++)
	     if (!alt_always_produces_empty (alts -> as[i]))
	        { rule -> kind = h_nonpredicate;
		  change = 1;
		};
	};

private void mark_if_rule_always_produces_empty (hyper_rule rule)
	{ if (rule -> ext) return;
	  if (rule -> kind == h_nonpredicate) return;
	  rule -> empty = h_alwaysproducesempty;
	  rule -> kind = h_predicate;
	};

private void determine_user_predicates ()
	{ int nr_passes;
	  int ix;
	  for (ix = 0; ix < nr_of_hyper_rules; ix++)
	     mark_if_rule_never_produces_empty (all_hyper_rules [ix]);
	  hint ("detecting user defined predicates...");
	  nr_passes = 0;
	  do { change = 0;
	       for (ix = 0; ix < nr_of_hyper_rules; ix++)
		  detect_if_rule_always_produces_empty (all_hyper_rules [ix]);
	       nr_passes++;
	     }
	  while (change);
	  hint ("needed %d pass%s for predicate detection",
		nr_passes, (nr_passes == 1)?"":"es");
	  for (ix = 0; ix < nr_of_hyper_rules; ix++)
	     mark_if_rule_always_produces_empty (all_hyper_rules [ix]);
	};

private void finish_marking_in_member (member m)
	{ if (m -> tag != tag_call) return;
	  m -> empty = m -> u.cl -> def -> empty;
	};

private void finish_marking_in_alt (alt a)
	{ member_list mems = a -> members;
	  int i;
	  if (mems == member_list_nil) return;
	  for (i=0; i < mems -> nrofms; i++)
	     finish_marking_in_member (mems -> ms[i]);
	  for (i=0; i < mems -> nrofms; i++)
	     if (mems -> ms[i] -> empty == h_neverproducesempty) return;
	  a -> empty = h_mayproduceempty;
	};

private void finish_marking (hyper_rule rule)
	{ alt_list alts = rule -> alts;
	  int i;
	  if (rule -> ext) return;
	  for (i=0; i < alts -> nrofas; i++)
	     finish_marking_in_alt (alts -> as[i]);
	};

private void finish_rule_marking ()
	{ int ix;
	  for (ix = 0; ix < nr_of_hyper_rules; ix++)
	     finish_marking (all_hyper_rules [ix]);
	};

private void check_layout_rule ()
	{ if (!layoutflag) return;
	  if (layout_rule -> kind & h_nonpredicate) return;
	  panic ("rule layout should be a non predicate");
	};

private void report_empty_detection_for_rule (hyper_rule rule)
	{ if (rule -> kind & h_predicate) eprint_log ("predicate");
	  else if (rule -> kind & h_semipredicate) eprint_log ("semipredicate");
	  else if (rule -> kind & h_nonpredicate) eprint_log ("rule");
	  else eprint_log ("?? rule ??");
	  eprint_log (" %s ", rule -> nonterminal);
	  switch (rule -> empty)
	     { case h_mayproduceempty:
		  eprint_log ("may produce empty");
		  break;
	       case h_neverproducesempty:
		  eprint_log ("never produces empty");
		  break;
	       case h_alwaysproducesempty:
		  eprint_log ("always produces empty");
		  break;
	       default: eprint_log ("may or may not produce empty");
	     };
	  eprint_log ("\n");
	};

private void try_report_empty_detection ()
	{ int ix;
	  if (!full_verbose) return;
	  wlog ("Hyper rules have been classified as follows:");
	  for (ix = 0; ix < nr_of_hyper_rules; ix++)
	     report_empty_detection_for_rule (all_hyper_rules [ix]);
	};

void do_empty_classification ()
	{ warning ("empty detection and rule classification...");
	  do_initial_rule_classification ();
	  detect_empty_producing_rules ();
	  determine_user_predicates ();
	  finish_rule_marking ();
	  check_layout_rule ();
	  try_report_empty_detection ();
	};
