/*
 * cdl3rts - CDL3 runtime system routines
 * Copyright (C) 2000 C.H.A. Koster
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <cdl3rts.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/times.h>
#include <signal.h>
#include <string.h>
#include <assert.h>

#define MAXSIZE 50
#define CHUNK 20

#ifndef CLK_TCK
#include <sys/param.h>
#define CLK_TCK HZ
#endif

extern void (*errhandler)();
#if 0
void (*errhandler)() = NULL;
#endif

long modulenr = -1;
long linenr = 0;
int received_signal = 0;

#if !defined(CDLASM) || !defined(__i386__)
#define STATIC static
#else
#define STATIC /**/
#endif

STATIC value freelists[MAXSIZE];
STATIC value backuplist[MAXSIZE];

long attaches = 0;
long detaches = 0;
STATIC long used = 0;
STATIC long reused = 0;
STATIC long requested = 0;
STATIC long freed = 0;
long frees = 0;

long saved_calls = 0;
long array_ref = 0;
long array_done = 0;

#if 0
STATIC value membase = 0x1000000; /* Base address */
STATIC long mainsize = 0;
STATIC value bitmap = 0x4000000;	/* Base address */
STATIC long bitmapsize = 0;
#endif
long chunksize = 1024 * 1024;	/* Size of data chunks */

STATIC value remerge = NULL;
STATIC value maxlist = NULL;
STATIC value backupmax = NULL;
/*STATIC long memsize = 0; */
STATIC long totsize = 0;
STATIC value recycle = NULL;
STATIC value allmem = NULL;

/* Status information */
long CDL3Usage = 0;
long CDL3FancyErrors = 0;
long CDL3Compiler = 0;

int argument_count;
char **arguments;

#if !defined(CDLASM) || !defined(__i386__)

static value
mmalloc (size)
     int size;
{
  value v;
  if ((v = (value) calloc (1, size * sizeof (value))) == NULL) {
    fprintf (stderr, "Out of memory... (Allocating %d).\n", size);
    quit (1);
  };
  return v;
}

#endif

#if !defined(CDLASM) || !defined(__i386__)

value getmem (long size)
{
    value v;
    if (size < MAXSIZE) {
        if (freelists[size] != NULL) {
            /* reuse */
            v = freelists[size];
            freelists[size] = (value) (*v);
            reused += (long) (size * sizeof (value));
        } else {
            /* MULTIPLE MALLOC */
            register int i;
/*
   if ((v = (value) malloc (size * sizeof (value) * CHUNK)) == NULL) {
   fputs ("Out of memory...", stderr);
   quit (1);
   };
 */
            v = (value) mmalloc (size * CHUNK);
            used += size * sizeof (value) * CHUNK;

            for (i = 0; i < CHUNK; i++, v += size) {
                *v = (long) (freelists[size]);
                freelists[size] = v;
            };

            v = freelists[size];
            freelists[size] = (value) (*v);
        };
    } else {
        /* large structure malloc */
        v = mmalloc (size);
        used += (long) (size * sizeof (value));
    };

    requested += (long) (size * sizeof (value));

    return v;
}

void
freemem (v, size)
     value v;
     long size;
{
  if (size < MAXSIZE) {
    *v = (long) (freelists[size]);
    freelists[size] = v;
  } else {
    free (v);
  };
  freed += (long) (size * sizeof (value));
}
#endif /* CDLASM */

value
ctext (t)
     char *t;
{
  int len =  strlen(t);
  value v;
#ifndef CDLASM
  v = getmem (3);
  v[0] = Tag (0, T_TEXT);
  v[1] = len;
  v[2] = (long)getmem(Length(len+1));
  strcpy(V_TEXT(v),t);
#else
  v = getmem (2+Length(len+1));
  v[0] = Tag (0, T_TEXT);
  v[1] = len;
  strcpy(v+2,t);
#endif
  return v;
}

value
concat (s1, s2)
     value s1;
     value s2;
{
  value r;

  int l1, l2;
  char *v;

  l1 = TextLength (s1);
  l2 = TextLength (s2);

#ifndef CDLASM
  r = getmem (3);
  r[0] = Tag (0, T_TEXT);
  r[1] = l1+l2;
  r[2] = (long)getmem(Length(l1+l2+1));
#else
  r = getmem (2+l1+l2);
  r[0] = Tag (0, T_TEXT);
  r[1] = l1+l2;
#endif
  v = V_TEXT(r);
  strncpy (v, V_TEXT (s1), l1);
  strncpy (v + l1, V_TEXT (s2), l2);
  v[l1+l2] = '\0';
  
  return r;
}

/* signal handler for control-C & segfaults */
#ifdef PC86
static void
sigerr (sig)
     int sig;
#ifndef NSIG
#define NSIG 19
#endif
#else /* PC86 */
static void
sigerr (sig, code, scp)
     int sig, code;
     struct sigcontext *scp;
#ifndef NSIG
#define NSIG 32
#endif
#endif
{
#ifdef ATARI
  static char *error[NSIG] =
  {"NULL", "ALRM", "BUS", "ODD", "ILL", "ZDIV",
   "CHK", "TRAP", "PRIV", "TRACE", "INT", "QUIT",
   "TSTP", "FPE", "HUP", "ABRT", "TERM", "USR1", "USR2"};
#else
  static char *error[NSIG] =
  {"BAD", "HUP", "INT", "QUIT", "ILL", "TRAP", "IOT",
   "EMT", "FPE", "KILL", "BUS", "SEGV", "SYS", "PIPE",
   "ALRM", "TERM", "URG", "STOP", "TSTP", "CONT", "CHLD",
   "TTIN", "TTOU", "IO", "XCPU", "XFSZ", "VTALRM", "PROF",
   "WINCH", "LOST", "USR1", "USR2"};
#endif
  fflush (stdout);
  fflush (stderr);
  if (modulenr > 0)
    fprintf (stderr, "SIG%s interrupt in line %ld of %s\n",
	     error[sig], linenr, "<unknown>");
  else {
    if (sig == 2 && CDL3FancyErrors) {
      static void FancyError (void);
      FancyError ();
    } else
      fprintf (stderr, "SIG%s interrupt\n", error[sig]);
  }
  received_signal = sig;
  quit (1);
}

static char *FancyErrors[38] =
{
  "You burned CDL3 to death.",
  "Your lightning bolt kills CDL3.",
  "You chill CDL3, remember to put flowers on its grave.",
  "CDL3 is dissolved by your goodness.",
  "CDL3 crumples as you kill it by draining its energy.",
  "Your fireball hits CDL3 with full force, causing an immediate death.",
  "CDL3 reels from the sheer power of your spell, and is destroyed.",
  "Your lightning bolt shatters CDL3 to pieces.",
  "The magic missile tears away the remaining life of CDL3.",
  "CDL3 dies dies while looking rather shocked.",
  "CDL3 makes a strange sound but is suddenly very silent, as you place your dagger in its back.",
  "Your kick at CDL3's face splits its head open - yummy.",
  "CDL3 doesn't recover from your bash ! - CDL3 is dead.",
  "Your light wounds are enough to send CDL3 over the brink....",
  "You critically kill CDL3.",
  "Your flaming power chars CDL3 to death!",
  "CDL3 is dissolved by your evilness.",
  "CDL3 is dissolved into a sticky ooze.",
  "CDL3 is frozen to death by your cone of cold.",
  "CDL3 is whacked by your swarm of meteors.",
  "CDL3 is ripped apart by your storm of ice.",
  "CDL3 dies instantly from the power of your holy word.",
  "CDL3 dies instantly from the power of your unholy word.",
  "CDL3 dies instantly from the power of your word.",
  "CDL3 is struck by your ray of sunlight, and destroyed instantly.",
  "CDL3 is killed instantly by the dreaded quivering palm.",
  "CDL3 is killed by your fiery breath.",
  "CDL3 is killed by your gas breath.",
  "CDL3 is frozen to death by your frost breath.",
  "CDL3 melts into a pile of stick ooze before your eyes.",
  "CDL3 is killed by your lightning breath.",
  "You punch CDL3's head with a blow, that crushes its skull.",
  "You punch CDL3 in its kidneys resulting in its immediate death.",
  "You bludgeon CDL3 to death.",
  "You succesfully pierce CDL3. The dead body falls to the ground.",
  "You pierce CDL3's heart, you heartbreaker you.",
  "You beautifully slash CDL3 into two parts - both dead.",
  "Your slash at CDL3 results in blood spurting all over, and its immediate death."
};

static void
FancyError (void)
{
  time_t Time;

  Time = time ((time_t *) NULL);
  srand (Time);
  fprintf (stderr, "%s\n"
	   "CDL3 is dead! R.I.P.\n"
	   "Your blood freezes as you hear CDL3's death cry.\n", FancyErrors[rand () % 38]);
}

void
start_rts (argc, argv)
     int argc;
     char *argv[];
{
  register int i;
  argument_count = argc;
  arguments = argv;
  for (i = 0; i < MAXSIZE; i++)
    freelists[i] = NULL;
#ifdef NEW_GETMEM
  freelist = NULL;
#endif

  if (getenv ("CDL3USAGE") != NULL) {
    CDL3Usage = 1;
    chunksize = 1024 * 16;
  }
  if (getenv ("CDL_FANCY_ERRORS") != (char *) NULL) {
    CDL3FancyErrors = 1;
  }
  if (strncmp (argv[0], "cdl", 3) == 0) {
    CDL3Compiler = 1;
  }
  if (strstr (argv[0], "/cdl") != (char *) NULL) {
    CDL3Compiler = 1;
  }
  /* INSTALL SIGNAL HANDLERS */
  for (i = 1; i < NSIG; i++)
    switch (i) {
#if !defined(ATARI) && !defined(PC86) && !defined(PC386)
    case SIGCHLD:
    case SIGTSTP:
    case SIGCONT:
    case SIGWINCH:
    case SIGPROF:
      break;
#endif
    default:
      (void) signal (i, sigerr);
    };
}

void
abort_rts (long line, char* module_name)
{
    fflush (stdout);
    fflush (stderr);

    if (CDL3FancyErrors && CDL3Compiler) {
        fprintf (stderr, "Internal compiler error in line %ld of module %s\n"
                 "Please contact the development team.\n", line, module_name);
    } else {
        fprintf (stderr, "Aborted in line %ld of %s\n", line, module_name);
    }

#ifdef ABORT_BY_ASSERT
    assert(0);
#endif

    quit (2);
}

void stop_rts ()
{
  quit (0);
}

extern int cdl_main ();

int main (int argc, char *argv[])
{
  start_rts (argc, argv);
  cdl_main ();
  stop_rts ();
  return (0);
}

void quit (int e)
{
  fflush (stdout);
  fflush (stderr);
  if ((e>0)&&(errhandler!=NULL))
    errhandler();

#ifndef PC386
  if (CDL3Usage) {
    struct tms t;
#ifdef EXTRAGC
    extern char _edata;
    used = ((char *)sbrk(0))-&_edata;
#endif

    times (&t);

    fprintf (stderr,
	     "\nRequested Allocated Freed     Attaches  Detaches  Frees     User    System\n");
    fprintf (stderr,
	     "%-9ld %-9ld %-9ld %-9ld %-9ld %-9ld %-7.2f %-7.2f\n",
	     requested, used, freed, attaches, detaches, frees,
	     0.01 * (float) ((t.tms_utime * 100) / CLK_TCK),
	     0.01 * (float) ((t.tms_stime * 100) / CLK_TCK));
    if (saved_calls) {
      fprintf (stderr, "Saved %ld calls thru inlining.\n", saved_calls);
    }
    if (array_ref) {
      fprintf (stderr, "Saved %ld array references, left %ld.\n", array_ref, array_done);
    }
  }
#endif
  exit (e);
}

void error(char *message)
{
  fputs(message,stderr);
}
