/*
   File: predefrts.c
   Implementation of the standard library routines

   Copyright (C) 2000-2006 C.H.A. Koster
  
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   CVS ID: "$Id: predefrts.c,v 1.12 2006/01/04 11:35:45 marcs Exp $"
*/

/* Check to include config.h */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef WIN32
#include <windows.h>
#include <io.h>
#else
#include <termios.h>
#include <unistd.h>
#endif

#include "cdl3rts.h"
#include "predef.h"
#define MAXLINE 10240

int E200_get_key_TEXT (value *VTEXT)
{ char v[2];
#ifndef WIN32
  int fd;
  int ret;
  char c;
  struct termios old,new;
  fflush (stdout);
  if ((fd = open ("/dev/tty", O_RDONLY)) < 0)
    { v[0] = '\0';
      *VTEXT = C_TEXT(v);
      return (1);
    };
  tcgetattr (fd,&old);
  new = old;
  new.c_lflag &= ~ICANON;
  new.c_lflag &= ~ECHO;
  new.c_cc[VMIN] = 1;   /* one char */
  new.c_cc[VTIME] = 0;  /* no timing */
  new.c_cc[VQUIT] = 0;  /* ignore ^\ */
  new.c_cc[VINTR] = 0;  /* ignore ^C */
  new.c_cc[VSUSP] = 0;  /* ignore ^Z */
  new.c_cc[VSTOP] = 0;  /* ignore ^Q */
  new.c_cc[VSTART] = 0; /* ignore ^S */
  tcsetattr (fd, TCSANOW, &new);
  ret = read (fd, &c, 1);
  tcsetattr (fd, TCSANOW, &old);
  close (fd);
  if (ret != 1)
    { v[0] = '\0';
      *VTEXT = C_TEXT(v);
      return (1);
    };
  v[0] = c;
  v[1] = '\0';
  *VTEXT = C_TEXT(v);
  return (1);
#else 
  fflush (stdout);
  /* v[0] = getkey() */;
  /* MS: find out what the WIN32 equivalent is */
  v[1] = '\0';
  *VTEXT = C_TEXT(v);
  return (1);
#endif
}

value C_FILE (FILE *f)
{ value v = getmem(2);
  v[0] = Tag (0, T_FILE);
  v[1] = (long) ((value) f);
  return (v);
}

#ifdef ASM
void D_FILE(value v) 
{ if (v != NULL) 
    { if (Refs(v) == 0) Free(v,2); 
      else DecrRefs(v);
    };
}
#endif

#ifdef ASM
int EQ_FILE (value v1, value v2)
{ return (V_FILE(v1)==V_FILE(v2));
}
#endif

#ifdef ASM 
void W_FILE(value v1)
{ WRITE("FILE");
}
#endif

#ifdef ASM
void A_FILE(value v1)
{ ATTACH(v1);
}
#endif

#ifdef ASM
void D_TEXT(value v1)
{ if (v1 != NULL)
    { if (Refs(v1) == 0) Free(v1, L_TEXT(v1) + 2);
      else DecrRefs(v1);
    }
}

int EQ_TEXT(value v1, value v2)
{ return (strcmp (V_TEXT(v1), V_TEXT(v2)) == 0);
}
#endif

static char line[MAXLINE];
char *safecopy(char *c)
{ char *sc;
  sc = (char *) getmem (Length (strlen(c)+1));
  strcpy (sc,c);
  return (sc);
}

int E8_pipe_to_command_TEXT_FILE (value V_TEXT, value *V_FILE)
{ FILE *handle;
#ifdef WIN32
  if ((handle = _popen (Text(V_TEXT), "wb")) == NULL)
    return 0;
#else
  if ((handle = popen (Text(V_TEXT), "w")) == NULL)
    return 0;
#endif

  *V_FILE = C_FILE(handle);
  return (1);
}

int E28_pipe_from_command_TEXT_FILE (value V_TEXT, value *V_FILE)
{ FILE *handle;
#ifdef WIN32
  if ((handle = _popen (Text(V_TEXT), "rb")) == NULL)
    return 0;
#else
  if ((handle = popen (Text(V_TEXT), "r")) == NULL)
    return 0;
#endif

  *V_FILE = C_FILE(handle);
  return (1);
}

int E1_open_input_file_TEXT_FILE (value V_TEXT, value *V_FILE)
{ FILE *handle = fopen (Text(V_TEXT), "rb");
  if (handle == NULL) return (0);

  *V_FILE = C_FILE(handle);
  return (1);
}

int E2_open_output_file_TEXT_FILE(value V_TEXT, value *V_FILE)
{ FILE *handle = fopen (Text(V_TEXT), "wb");
  if (handle == NULL) return (0);

  *V_FILE = C_FILE(handle);
  return (1);
}

int E3_open_standard_output_FILE (value *V_FILE)
{ *V_FILE = C_FILE(stdout);
  return (1);
}

int E4_open_standard_error_FILE (value *V_FILE)
{ *V_FILE = C_FILE(stderr);
  return (1);
}

int E5_open_standard_input_FILE (value *V_FILE)
{ *V_FILE = C_FILE(stdin);
  return (1);
}

int E6_close_FILE (value V_FILE)
{ return (fclose (V_FILE(V_FILE)) == 0); 
}

int E7_seek_FILE_INT (value V_FILE, value V_INT)
{ return (fseek (V_FILE(V_FILE), Int(V_INT), 0) == 0);
}

int E9_close_pipe_FILE (value V_FILE)
{
#ifdef WIN32
  return (_pclose (V_FILE(V_FILE)) == 0);
#else
  return (pclose (V_FILE(V_FILE)) == 0); 
#endif
}

static char strbuf[16];
static char *strstart;

static void convert_long (long n)
{ int sign;
  strstart = strbuf + 15;
  *strstart = '\0';
  if (n<0) {
    sign = 1;
    n = -n;
  } else
  if (n==0) {
    strstart--;
    *strstart = '0';
    return;
  } else
    sign = 0;

  while (n>0) {
    strstart--;
    *strstart = '0'+(n%10);
    n /= 10;
  }; 
  if (sign) {
    strstart--;
    *strstart = '-';
  };
}

int E10_write_TEXT (value V_TEXT)
{ fputs (Text(V_TEXT), stdout);
#ifdef FLUSH_ALL
  fflush (stdout);
#endif
  return (1);
}

int E11_write_INT (value V_INT)
{ convert_long (Int(V_INT));
  fputs (strstart, stdout);
  return (1);
}

int E12_write_FILE_TEXT (value V_FILE, value V_TEXT)
{ fputs (Text(V_TEXT), V_FILE(V_FILE));
  return (1);
}

int E13_write_FILE_INT (value V_FILE, value V_INT)
{ convert_long (Int(V_INT));
  fputs (strstart, V_FILE(V_FILE));
  return (1);
}

int E14_formatted_write_FILE_FORMAT (value V_FILE, value V_FORMAT)
{ switch (Alternative(V_FORMAT))
    { case 0: /* FORMAT TEXT */
        E14_formatted_write_FILE_FORMAT (V_FILE, Part(V_FORMAT,1));
        fputs (Text(Part(V_FORMAT,2)), V_FILE(V_FILE));
        break;
      case 1: /* FORMAT INT */
        E14_formatted_write_FILE_FORMAT (V_FILE, Part(V_FORMAT,1));
        convert_long (Int(Part(V_FORMAT,2)));
        fputs (strstart, V_FILE(V_FILE));
        break;
      case 2: /* TEXT */
        fputs (Text(Part(V_FORMAT,1)), V_FILE(V_FILE));
        break;
      case 3: /* INT */
        convert_long (Int(Part(V_FORMAT,1)));
        fputs (strstart, V_FILE(V_FILE));
        break;
    };
  return (1);
}

/* MS: changed fprintf (.., "%c", c) into a fputc */
int E15_write_char_FILE_INT(value V_FILE, value V_INT)
{ unsigned char c = (Int(V_INT) & 0xff);
  fputc (c, V_FILE(V_FILE));
  return (1);
};

int E16_write_char_INT (value V_INT)
{ unsigned char c = (Int(V_INT) & 255);
  fputc (c, stdout);
  return (1);
};

int E20_trace_TEXT (value V_TEXT)
{ fflush (stdout);
  fputs (Text(V_TEXT), stderr);
  fflush (stderr);
  return (1);
};

int E21_trace_INT (value V_INT)
{ fflush (stdout);
  convert_long (Int(V_INT));
  fputs (strstart, stderr);
  fflush (stderr);
  return (1);
}

int E22_formatted_trace_FILE_FORMAT (value V_FILE, value V_FORMAT)
{ (void) E14_formatted_write_FILE_FORMAT (V_FILE,V_FORMAT);
  return (1);
};

int E30_read_line_TEXT(value *V_TEXT)
{ if (fgets (line, MAXLINE-1, stdin) == NULL)
    return (0);

  line[MAXLINE-1] = '\0'; /* just in case */
  *V_TEXT = C_TEXT(line);
  return (1);
}

int E31_read_line_FILE_TEXT (value V_FILE, value *V_TEXT)
{ if (fgets (line, MAXLINE-1, V_FILE(V_FILE)) == NULL)
    return (0);

  line[MAXLINE-1] = '\0'; /* just in case */
  *V_TEXT = C_TEXT(line);
  return (1);
}

static int real_read_char (FILE* infile, value* V_TEXT)
{ int c = fgetc(infile);
  char str[2];

  /* EOF reached, or other error (file not open among things)? */
  if (c == EOF)
    return (0);

  str[0] = (char) (c & 0xff);
  str[1] = '\0';
  *V_TEXT = C_TEXT(str);

  return (1);
}

static int real_read_char_int (FILE* infile, value* V_INT)
{ int c = fgetc (infile);

  /* EOF reached, or other error (file not open among things)? */
  if (c == -1)
    return (0);

  *V_INT = C_INT (c);
  return (1);
}

int E32_read_char_FILE_TEXT (value V_FILE, value* V_TEXT)
{ return (real_read_char (V_FILE(V_FILE), V_TEXT));
}

int E33_read_char_TEXT (value* V_TEXT)
{ return (real_read_char (stdin, V_TEXT));
}

int E34_read_char_FILE_INT (value V_FILE, value *V_INT)
{ return (real_read_char_int (V_FILE(V_FILE), V_INT));
}

int E35_read_char_INT (value *V_INT)
{ return (real_read_char_int (stdin, V_INT));
}

int E40_less_INT_INT (value V_INT, value V_INT1)
{ return (Int(V_INT) < Int(V_INT1)); 
}

int E41_lesseq_INT_INT (value V_INT, value V_INT1)
{ return (Int(V_INT) <= Int(V_INT1));
}

int E42_greater_INT_INT (value V_INT, value V_INT1)
{ return (Int(V_INT) > Int(V_INT1));
}

int E43_greatereq_INT_INT (value V_INT, value V_INT1)
{ return (Int(V_INT) >= Int(V_INT1));
}

int E51_extract_TEXT_INT_INT_TEXT (value v_TEXT, value v_INT1, value v_INT2, value *v_TEXT1)
{ int size = Int(v_INT2) - Int(v_INT1);
  char *t;
  if (size < 1)
    { /* Construct empty TEXT */
      size = 0;
      t = (char *) getmem(1);
      t[0] = '\0';
    }
  else
    { t = (char *) getmem (Length (size+1));
      strncpy (t, Text(v_TEXT) + Int(v_INT1), size); 
      t[size] = '\0';
    };

  *v_TEXT1 = C_TEXT(t);
  freemem ((value) t, Length(size+1));	/* Cast to keep compiler from complaining */
  return (1);
}

int E52_is_subtext_TEXT_INT_TEXT (value V_TEXT, value *V_INT, value V_TEXT1)
{ int l = TextLength(V_TEXT1);
  if (Int(*V_INT) > TextLength(V_TEXT)) return (0);
  if (strncmp (Text(V_TEXT) + Int(*V_INT), Text(V_TEXT1), l)) return (0);

  *V_INT = C_INT(Int(*V_INT) + (long)l);
  return (1);
}

int E53_between_TEXT_INT_TEXT_TEXT (value V_TEXT, value V_INT, value V_TEXT1, value V_TEXT2)
{ char *c = Text(V_TEXT);
  int i;

  for (i = Int(V_INT); ((*c != '\0') && (i > 0)); i--, c++) ;
  if (*c)
    return ((*(Text(V_TEXT1)) <= *c) && (*c <= *(Text(V_TEXT2))));
  return (0);
}

int E54_not_between_TEXT_INT_TEXT_TEXT (value V_TEXT, value V_INT, value V_TEXT1, value V_TEXT2)
{ char *c = Text(V_TEXT);
  int i;

  for (i = Int(V_INT); ((*c != '\0') && (i > 0)); i--, c++) ; 
  if (*c)
    return ((*c < *(Text(V_TEXT1))) || (*c > *(Text(V_TEXT2))));
  return 0;
}

int E55_before_TEXT_TEXT (value V_TEXT, value V_TEXT1)
{ return (strcmp (Text(V_TEXT), Text(V_TEXT1)) < 0);
}

int E56_is_prefix_TEXT_TEXT_TEXT (value v_TEXT, value v_TEXT1, value *v_TEXT2)
{ int l1 = TextLength(v_TEXT);
  int l2 = TextLength(v_TEXT1);

  if (l1 > l2) return (0);
  if (strncmp (Text(v_TEXT), Text(v_TEXT1), l1))
    return (0);

  *v_TEXT2 = C_TEXT(Text(v_TEXT1) + l1);
  return (1);
}

int E57_prefix_TEXT_INT_TEXT_TEXT (value v_TEXT, value v_INT, value *v_TEXT1, value *v_TEXT2)
{ char *t;
  int tlen;

  if (Int(v_INT) > TextLength(v_TEXT))
    return (0);

  /* Construct the prefix */
  tlen = Length(Int(v_INT)+1);
  t = (char*) getmem (tlen);
  strncpy (t, Text(v_TEXT), Int(v_INT));
  t[Int(v_INT)] = '\0';

  /* Free is ok, because C_TEXT makes a copy */
  *v_TEXT1 = C_TEXT(t);
  freemem ((value) t, tlen);	/* Cast to keep compiler happy */
  *v_TEXT2 = C_TEXT(Text(v_TEXT) + Int(v_INT));
  return (1);
}

int E58_length_TEXT_INT(value V_TEXT, value *V_INT)
{ *V_INT = C_INT(TextLength(V_TEXT));
  return (1);
}

int E59_asciicode_TEXT_INT_INT (value V_TEXT, value V_INT1, value *V_INT2)
{ if (Int(V_INT1) >= TextLength(V_TEXT)) return (0);
  *V_INT2 = C_INT((long) *(Text(V_TEXT) + Int(V_INT1)));
  return (1);
}

int E60_bindec_INT_TEXT (value V_INT, value *V_TEXT)
{ sprintf (line, "%ld", Int(V_INT));
  *V_TEXT = C_TEXT(line); 
  return (1);
}

int E61_decbin_TEXT_INT (value V_TEXT, value *V_INT)
{ long n;
  if (sscanf (Text(V_TEXT), "%ld", &n) != 1) return (0);
  *V_INT = C_INT(n);
  return (1);  
}

int E70_hash_TEXT_INT_INT (value V_TEXT, value V_INT1, value *V_INT2)
{ unsigned char *t = Text(V_TEXT);
  register long v; 

  for (v = 0; *t; t++) v += (long) (*t); 
  *V_INT2 = C_INT(v % Int(V_INT1));
  return (1);
}

int E80_command_arg_INT_TEXT(value V_INT, value *V_TEXT)
{ if ((Int(V_INT) < 0) || (Int(V_INT) >= (long) argument_count))
    return (0);

  *V_TEXT = C_TEXT(arguments[Int(V_INT)]);
  return (1);
}

int E81_exit_INT (value V_INT) 
{ quit (Int(V_INT));
  return (0);		/* Unreachable code */
}

int E82_get_from_environment_TEXT_TEXT (value V_TEXT1, value *V_TEXT2)
{ char *t = getenv (Text(V_TEXT1));
  if (t == NULL) return (0);
  *V_TEXT2 = C_TEXT(t);
  return (1);
}

int E83_time_INT(value *V_INT)
{ *V_INT = C_INT((long) time (NULL));
  return (1);
}

int E85_file_date_TEXT_INT (value V_TEXT, value *V_INT)
{
#ifdef WIN32
  struct _stat filestat;
  if (_stat (Text(V_TEXT), &filestat)) return (0);
#else
  struct stat filestat;
  if (stat (Text(V_TEXT), &filestat)) return (0);
#endif
  *V_INT = C_INT((long) (filestat.st_mtime));
  return (1);
}

int E84_execute_TEXT (value V_TEXT)
{ return (!system (Text(V_TEXT)));
}

extern int received_signal;
int E90_killed ()
{ return (received_signal);
}

