/*
   File: lxcn_nfa.h
   Interface for the bitwise NFA functions implementing a regular
   edit distance (optionally including transpositions)

   Copyright 2009 Radboud University of Nijmegen
 
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$"
*/
#ifndef IncLxcnNfa
#define IncLxcnNfa

#include <abase_porting.h>
#include <limits.h>

#define ALPHABET_SIZE    (2<<CHAR_BIT)
#define MAX_EDITDISTANCE 2
#define TRANSPOSITIONS   0

/* 
   Note: definitions cannot be made opaque, because we want to be able to
   allocate (at least) the nfa_states directly

   When this code will be replaced with similar code operating on a
   generalized edit distance, this may become problematic.
*/
typedef u_int64 nfa_transition_table[ALPHABET_SIZE]; 
typedef u_int64 nfa_state[MAX_EDITDISTANCE+1 + TRANSPOSITIONS*MAX_EDITDISTANCE];

typedef struct nfa_static_info {  
    nfa_transition_table mask;
    int height;
    int width;
} AutomatonData;

/* 
   The pointer to nfa_static_info should be removed from this definition and
   passed to the functions that accept AutomatonState's explicitly to make
   this structure more compact.
 */
typedef struct nfa_dynamic_info {
    nfa_state reg;
    struct nfa_static_info *info;
} AutomatonState;

/* Maximum size of pattern is 63 characters (will complain) */
extern void lxcn_nfa_create(char *pattern, int distance, AutomatonData*, AutomatonState*);
extern AutomatonState *lxcn_nfa_deepcopy(AutomatonState *dest, AutomatonState *src);

extern int lxcn_nfa_feed_char_copy(AutomatonState*, char c, AutomatonState *new);
extern int lxcn_nfa_feed_string(AutomatonState*, char *str);

#define lxcn_nfa_feed_char(state, c)		\
    	lxcn_nfa_feed_char_copy(state, c, state)

#define lxcn_nfa_accepts_at(state, dist)	\
	(((state)->reg[dist] >> (state)->info->width & 1) != 0)
#define lxcn_nfa_accepts(state)			\
	lxcn_nfa_accepts_at(state, (state)->info->height)

#define lxcn_nfa_rejects_at(state, dist)	\
	(((state)->reg[dist] & ((2ULL << (state)->info->width)-1)) == 0)
#define lxcn_nfa_rejects(state)			\
	lxcn_nfa_rejects_at(state, (state)->info->height)

/* returns the remainder of an accepted string, or NULL if not accepted */
extern char *lxcn_nfa_match_shortest(AutomatonState*, char *str);
extern char *lxcn_nfa_match_longest (AutomatonState*, char *str);

/* returns negative value if nfa is nondeterministic, or position in original input */
extern int lxcn_nfa_deterministic_pos(AutomatonState*, int dist);
extern void lxcn_nfa_get_max_pos(AutomatonState*, int *x, int *y);
extern void lxcn_nfa_get_min_pos(AutomatonState*, int *x, int *y);

/* examines whether the nfa is still only in its initial states */
extern int lxcn_nfa_is_trivial(AutomatonState*);

/* OBSOLETE */
extern int lxcn_nfa_compare(AutomatonState *a, AutomatonState *b);

/* Debugging functions */
extern void lxcn_nfa_dump(AutomatonState*);
extern int  lxcn_nfa_feed_string_dump(AutomatonState*, char *str);

#endif 

