/* 
   File: abase_memalloc.c
   Defines memory allocation routines

   Copyright 2009 Radboud University of Nijmegen

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$"
*/

/* standard includes */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

/* Local includes */
#include "abase_porting.h"
#include "abase_error.h"
#include "abase_memalloc.h"

/*
   Conditionally include a memory logging function
*/
#ifdef MEMORY_LOG
static FILE *log_file = NULL;
void log_memory (char *format, ...)
	{ char *log_fname = "agfl_memory.log";
	  char buf[MAX_FMT_LEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsnprintf (buf, MAX_FMT_LEN, format, arg_ptr);
	  va_end (arg_ptr);
	  
	  if (log_file == NULL)
	     log_file = fopen (log_fname, "w");
	  if (log_file == NULL)
	     abs_abort ("log_memory", "could not open memory log '%s'", log_fname);
	  fprintf (log_file, "%s\n", buf);
	};

#endif /* MEMORY_LOG */

void *abs_malloc (const size_t size, const char *place)
	{ void* result = malloc (size);
	  if ((result == NULL) && (size != 0))
	     abs_abort ("abs_malloc", "Failed to allocate %lu bytes of memory at %s.\n",
			(long) size, place);
#ifdef MEMORY_LOG
	  log_memory ("at %p, %s: malloced %lu bytes", result, place, (long) size);
#endif
	  return (result);
	};

void *abs_calloc (const size_t nr, const size_t size, const char *place)
	{ size_t nr_bytes = nr * size;
	  void *result;
	  result = malloc (nr_bytes);
	  if ((result == NULL) && (nr_bytes != 0))
	     abs_abort ("abs_calloc", "Failed to allocate %lu bytes of memory at %s.\n",
			(long) (nr_bytes), place);
#ifdef MEMORY_LOG
	  log_memory ("at %p, %s: calloced %lu * %lu bytes", result, place, (long) nr, (long) size);
#endif
	  return (result);
	};

void *abs_realloc (void *ptr, const size_t size, const char *place)
	{ void* result = realloc (ptr, size);
	  if ((result == NULL) && (size != 0))
	     abs_abort ("abs_realloc", "Failed to reallocate %lu bytes of memory at %s.\n",
			(long) size, place);
#ifdef MEMORY_LOG
	  log_memory ("at %p, %s: resized %lu bytes", ptr, place, (long) size);
	  RLOG_MEM (ptr, "abs_realloc, resize", (long) size, place);
	  if (result != ptr)
	     { log_memory ("at %p, %s: memory has been freed", ptr, place);
	       log_memory ("at %p, %s: memory has been reallocated", result, place);
	     };
#endif
	  return (result);
	};

void abs_free (void *ptr, const char *place)
	{ if (ptr == NULL) return;
#ifdef MEMORY_LOG
	  log_memory ("at %p, %s: memory has been freed", ptr, place);
#endif
	  free (ptr);
	};

char *abs_new_string (const char *old, const char *place)
	{ char *result;
	  if (old == NULL)
	     abs_abort ("abs_new_string", "Called with NULL argument at %s", place);
	  result = (char *) abs_malloc (strlen (old) + 1, place);
	  strcpy (result, old);
	  return (result);
	};

char *abs_new_fmtd_string (const char *place, const char *format, ...)
	{ char buf[MAX_FMT_LEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsnprintf (buf, MAX_FMT_LEN, format, arg_ptr);
	  va_end (arg_ptr);
	  return (abs_new_string (buf, place));
	};
