/*
   File: driver.c
   Compiler driver for agfl with new codegenerator
   Implements make-like functionality for grammar and lexicon.
  
   Copyright 2009-2010 Radboud University of Nijmegen

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id$"
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>

/* libabase includes */
#include <abase_version.h>
#include <abase_porting.h>
#include <abase_error.h>
#include <abase_process.h>
#include <abase_fileutil.h>
#include <abase_memalloc.h>
#include <abase_dstring.h>

/*------------------------------------------------------------------------------
// Alternatively, use the new code generator to generate machine independent
// assembler, assemble it using agfl-as and then use the agfl-run interpreter
//----------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------
// Update later:
// Developers may use agfl-maint instead of agfl-coder. It has more options
// available and does not pass optimization options to agfl by default.
-------------------------------------------------------------------------*/
#define NEGMEMO_FLAG		0

#if MAINTAINER_MODE
#  define DIRECTORS_FLAG	0
#else
#  define DIRECTORS_FLAG	1
#endif

static void show_version()
{ abs_message ("Agfl version %s, Copyright 2010, Radboud University of Nijmegen.", AGFL_VERSION);
}

static char agfl_help_mesg[] =
  "Usage: agfl [-cfGHhmNpsVvw] <grammar>\n"
  "\n"
  "Options are:\n"
  "    -f           force new generation\n"
  "    -G           generate generative grammar (not supported in this version)\n"
  "    -H           generate hybrid parser\n"
  "    -h           show this help, and exit\n"
  "    -p           profile grammar\n"
  "    -s           add Unix #!-line to coded grammar\n"
  "    -V           show version\n"
  "    -v           be verbose\n"
  "    -w           suppress compiler warnings\n"
  "    -N           use negative memoizing\n"
  "    -Xtrp:modulename add TRIPLES file\n"
  "    -Xvar:XYZ    select variant lines X, Y and Z\n"
  "    -Xlexrule    show LEX rules in parse tree\n"
  "\n"
  "Arguments are:\n"
  "    <grammar>    name of grammar file  (.gra)\n";

#if MAINTAINER_MODE
static char agflmaint_help_mesg[] =
  "Additional options for debug version:\n"
  "    -d           debug\n"
  "    -l           write log file\n"
  "    -S           leave generated assembler file\n"
  "    -t           show compile time\n"
  "    -A           suppress well-formedness analysis\n"
  "    -D           use director set lookahead\n"
  "    -F           left-factorize grammar\n";
#endif

static void show_help ()
{ show_version();
  abs_message (agfl_help_mesg);

#if MAINTAINER_MODE
  abs_message (agflmaint_help_mesg);
#endif
}

#define NXTRP	16

typedef struct
{ char *grammar_file;
  int force;                          /* -f */
  int debug;                          /* -d */
  int generative_grammar;             /* -G */
  int hybrid_parser;		      /* -H */
  int write_log;                      /* -l */
  int profile;                        /* -p */
  int keep_assembler_file;            /* -S */
  int show_time;                      /* -t */
  int verbose;                        /* -v */
  int suppress_warnings;              /* -w */
  int suppress_analysis;              /* -A */
  int directors;                      /* -D */
  int left_factorize;                 /* -F */
  int negative_memo;                  /* -N */
  int shebang;			      /* -s */
  int nXtrp;			      /* -s */
  char *Xtrp[NXTRP];		      /* -Xtrp: */
  char *Xvar;			      /* -Xvar:XYZ */
  char *Xlexrule;		      /* -Xlexrule */
} Options;

void init_options (Options* options)
{ int i;
  options -> grammar_file = NULL;
  options -> force = 0;
  options -> debug = 0;
  options -> generative_grammar = 0;
  options -> hybrid_parser = 0;
  options -> write_log = 0;
  options -> profile = 0;
  options -> keep_assembler_file = 0;
  options -> show_time = 0;
  options -> verbose = 0;
  options -> suppress_warnings = 0;
  options -> suppress_analysis = 0;
  options -> directors = DIRECTORS_FLAG;
  options -> left_factorize = 0;
  options -> negative_memo = NEGMEMO_FLAG;
  options -> shebang = 0;
  options -> nXtrp = 0;
  for (i = 0; i < NXTRP; i++) options -> Xtrp[i] = NULL;
  options -> Xvar = NULL;
  options -> Xlexrule = NULL;
}

void process_extended_option(char *str, Options *options)
{ if (strncmp(str, "-Xtrp:", 5) == 0)
  { if (options -> nXtrp < NXTRP)
      { options -> Xtrp[options -> nXtrp] = str;
	options -> nXtrp++;
      } else
      { abs_error ("Too many -Xtrp: options");
	abs_exit (1);
      }
  } else
  if (strncmp(str, "-Xvar:", 5) == 0)
  { options -> Xvar = str;
  } else
  if (strncmp(str, "-Xlexr", 5) == 0)
  { options -> Xlexrule = str;
  } else
  { abs_error ("unknown extended option `%s'", str);
    abs_error ("use `agfl -h' for help");
    abs_exit (1);
  }
}

void process_options (int argc, char** argv, Options* options)
{ char *opt;
  char c;

  /*
     Check whether options specified
  */
  if (argc == 1)
    { show_version ();
      abs_exit (1);
    };

  /*
     Process options
  */
  while ((opt = *++argv))
    { if ((opt[0] == '-') && (opt[1] != '\0'))
	{ while((c = *++opt))
	    { switch (c)
		{ case 'f':
                    options -> force++;
                    break;
                  case 'G':
		    abs_error ("This version of AGFL does not support building generating grammars.");
		    abs_exit (1);
                    options -> generative_grammar = 1;
                    options -> directors = 0;
                    options -> negative_memo = 0;
                    break;
		  case 'H':
		    options -> hybrid_parser = 1;
		    break;
                  case 'h':
                  case '?':
                    show_help(stdout);
                    exit (1);
                  case 'p':
                    options -> profile++;
                    break;
                  case 's':
                    options -> shebang++;
                    break;
                  case 'V':
                    show_version();
                    exit (0);
                  case 'v':
                    options -> verbose++;
                    break;
                  case 'w':
                    options -> suppress_warnings++;
                    break;
#if MAINTAINER_MODE
                  case 'd':
                    options -> debug++;
                    break;
                  case 'l':
                    options -> write_log++;
                    break;
                  case 'S':
                    options -> keep_assembler_file++;
                    break;
                  case 't':
                    options -> show_time++;
                    break;
                  case 'A':
                    options -> suppress_analysis++;
                    break;
                  case 'D':
                    options -> directors++;
                    break;
                  case 'F':
                    options -> left_factorize++;
                    break;
#endif /* MAINTAINER_MODE */
                  case 'N':
                    options -> negative_memo++;
                    break;
		  case 'X':
		    process_extended_option(opt-1, options);
		    goto nextarg;	/* break won't suffice */
                  default:
                    abs_error ("unknown option `-%c'", c);
                    abs_error ("use `agfl -h' for help");
                    abs_exit (1);
                };
            };
        }
      else
        { if (options -> grammar_file == NULL)
            options -> grammar_file = opt;
          else
	    { abs_error ("multiple input files specified"
                         " (`%s' and `%s')", options -> grammar_file, opt);
              abs_exit (1);
            };
        }
nextarg: ;
    };

  /*
     Check whether options valid
  */
  if (options -> grammar_file == NULL)
    { abs_error ("no input file specified");
      abs_exit (1);
    };

  if (options -> verbose >= 2)
    { abs_printf ("  %s ",
#if MAINTAINER_MODE
	    "agflmaint");
#else
	    "agfl");
#endif /* MAINTAINER_MODE */
      
      if (options -> force              ) abs_printf (" -f");
      if (options -> debug              ) abs_printf (" -d");
      if (options -> generative_grammar ) abs_printf (" -G");
      if (options -> hybrid_parser      ) abs_printf (" -H");
      if (options -> write_log          ) abs_printf (" -l");
      if (options -> profile            ) abs_printf (" -p");
      if (options -> keep_assembler_file) abs_printf (" -S");
      if (options -> show_time          ) abs_printf (" -t");
      if (options -> verbose > 0        ) abs_printf (" -v");
      if (options -> verbose > 1        ) abs_printf (" -v");
      if (options -> suppress_warnings  ) abs_printf (" -w");
      if (options -> suppress_analysis  ) abs_printf (" -A");
      if (options -> directors          ) abs_printf (" -D");
      if (options -> left_factorize     ) abs_printf (" -F");
      if (options -> negative_memo      ) abs_printf (" -N");
      if (options -> shebang            ) abs_printf (" -s");
      if (options -> nXtrp)
      { int i;
	for (i = 0; i < options -> nXtrp; i++)
	{ abs_printf(" %s", options -> Xtrp[i]);
	}
      }
      if (options -> Xvar               ) abs_printf (" %s", options -> Xvar);
      if (options -> grammar_file       ) abs_printf (" %s", options -> grammar_file);
      abs_message ("");
    };
}

void strip_extension (char *path, char *ext)
{ int len1 = (int) strlen (path);
  int len2 = (int) strlen (ext);

  if ((len1 > len2) && strcmp (path + len1 - len2, ext) == 0)
    path[len1 - len2] = '\0';
}

void get_last_dir (char* name, char *path)
{ char *last_dir;
  char *p;

  last_dir = path;
  for (p = path; p[0] != '\0'; p++)
    if (((p[0] == '/') || (p[0] == DIRSEP)) && (p[1] != '\0'))
      last_dir = p;
  strcpy(name, last_dir);
}

#ifdef WIN32
#define DIR_SEP "\\"
#else
#define DIR_SEP "/"
#endif

/*
   This is very dubious code, doing something special in the case
   that the argument is a directory. What was the original intention?
*/
void get_base_name (char* base, char *name, char *ext)
{ char path[MAXPATHLEN];

  strcpy (base, name);
  strip_extension (base, ext);
#ifdef WIN32
  /* Unsafe copy: no snprintf */
  sprintf (path, "%s%s", base, ext);
#else
  snprintf (path, MAXPATHLEN, "%s%s", base, ext);
#endif

  if (!abs_file_exists (path))
    { if (abs_is_directory (base))
        { char name[MAXPATHLEN];

          get_last_dir(name, base);
          strcat(base, DIR_SEP);
          strcat(base, name);
#ifdef WIN32
          /* Unsafe copy */
          sprintf (path, "%s%s", base, ext);
#else
          snprintf (path, MAXPATHLEN, "%s%s", base, ext);
#endif
        };

      if (!abs_file_exists (path))
        { abs_error ("cannot locate input file `%s'", path);
          abs_exit (1);
        };
    };

  if (!abs_is_normal_file (path))
    { abs_error ("`%s' is not a regular file", path);
      abs_exit (1);
    };

  if (strlen (base) + 4 > MAXPATHLEN)
    { abs_error ("`%s' is too long", base);
      abs_exit (1);
    };
}

int have_lexicon (char* base)
{ return (abs_file_ext_exists (base, "lif"));
}

/*
   Check whether file with path1 is newer than file with path2.
   Return false if file with path1 does not exist.
   Return true if file with path2 does not exist.
*/
int file_is_newer (char* path1, char* path2)
{ time_t time1 = 0;
  time_t time2 = 0;
  if (abs_file_mtime (path1, &time1)) return (0);
  (void) abs_file_mtime (path2, &time2);

  return (time1 > time2);
}

void check_ext_file_exists (char *name, char *ext)
{ if (!abs_file_ext_exists (name, ext))
    { abs_error ("cannot locate input file `%s.%s'", name, ext);
      exit(1);
    };
}

void delete_file (const char* path, int verbose)
{ if (verbose >= 2)
    abs_message ("delete file '%s'", path);

  /* Note remove is an existing C function, both in Unix as Win32 */
  remove (path);
}

void try_dump_argv (char *argv[], Options *opt)
{ char **argptr;
  if (!opt -> verbose) return;
  for (argptr = argv; (*argptr) != NULL; argptr++)
    { if (argptr != argv) abs_printf (" ");
      abs_printf ("%s", *argptr);
    };
  abs_message ("");
}

#define NRARGS	(32 + NXTRP)

void compile_grammar (char *base, Options *opt)
{ char *coder_cmdline[NRARGS];
  char asm_file[MAXPATHLEN];
  int nrargs = 0;
  check_ext_file_exists (base, "gra");

  /* Compose the command line */
  coder_cmdline[nrargs++] = "agfl-coder";
  if (opt -> debug)               coder_cmdline[nrargs++] = "-d";
  if (opt -> generative_grammar)  coder_cmdline[nrargs++] = "-G";
  if (opt -> hybrid_parser)	  coder_cmdline[nrargs++] = "-H";
  if (opt -> write_log)           coder_cmdline[nrargs++] = "-l";
  if (opt -> profile)             coder_cmdline[nrargs++] = "-p";
  if (opt -> show_time)           coder_cmdline[nrargs++] = "-t";
  if (opt -> verbose)             coder_cmdline[nrargs++] = "-v";
  if (opt -> verbose > 1)         coder_cmdline[nrargs++] = "-v";
  if (opt -> suppress_warnings)   coder_cmdline[nrargs++] = "-w";
  if (opt -> suppress_analysis)   coder_cmdline[nrargs++] = "-A";
  if (opt -> directors)           coder_cmdline[nrargs++] = "-D";
  if (opt -> left_factorize)      coder_cmdline[nrargs++] = "-F";
  if (opt -> negative_memo)       coder_cmdline[nrargs++] = "-N";
  if (opt -> nXtrp)
  { int i;
    for (i = 0; i < opt -> nXtrp && nrargs < NRARGS - 3; i++)
    { coder_cmdline[nrargs++] = opt -> Xtrp[i];
    }
  }
  if (opt -> Xvar)                coder_cmdline[nrargs++] = opt -> Xvar;
  if (opt -> Xlexrule)            coder_cmdline[nrargs++] = opt -> Xlexrule;
  coder_cmdline[nrargs++] = base;
  coder_cmdline[nrargs] = NULL;

  /* Try and execute agfl-coder */
  try_dump_argv (coder_cmdline, opt);
  if (abs_execute (coder_cmdline) == 0)
    return;

  /* If we reached this point, something went wrong, delete .s file */
  sprintf (asm_file, "%s.s", base);
  delete_file (asm_file, opt -> verbose);
  abs_exit (1);
}

void assemble_code (char *base, Options *opt)
{ char *as_cmdline[16];
  char asm_file[MAXPATHLEN];
  char obj_file[MAXPATHLEN];
#if defined (INSTALLDIR)
  char shebang_text[MAXPATHLEN];
#endif
  int nrargs = 0;
  int as_exitstatus;

#ifdef WIN32
  sprintf (asm_file, "%s.s", base);
  sprintf (obj_file, "%s.aob", base);
#else
  snprintf (asm_file, MAXPATHLEN, "%s.s", base);
  snprintf (obj_file, MAXPATHLEN, "%s.aob", base);
#endif
  check_ext_file_exists (base, "s");
#if defined (INSTALLDIR)
  snprintf (shebang_text, MAXPATHLEN, "%s/agfl-run", INSTALLDIR);
#endif
  
  /* Compose the assembler line */
  as_cmdline[nrargs++] = "agfl-as";
  as_cmdline[nrargs++] = asm_file;
  as_cmdline[nrargs++] = "-o";
  as_cmdline[nrargs++] = obj_file;
#if defined (INSTALLDIR)
  if (opt -> shebang)
    { as_cmdline[nrargs++] = "-sh";
      as_cmdline[nrargs++] = shebang_text;
    }
#endif
  as_cmdline[nrargs] = NULL;

  /* Try and execute agfl-as */
  try_dump_argv (as_cmdline, opt);
  as_exitstatus = abs_execute (as_cmdline);

  /* Keep the assembler file, if requested */
  if (!opt -> keep_assembler_file)
    delete_file (asm_file, opt -> verbose);

  if (as_exitstatus)
    { /* Assembly went wrong, stop */
      abs_exit (1);
    };

#if defined (INSTALLDIR)
  if (opt -> shebang)
    symlink (obj_file, base);		/* ignore failure, such as EEXIST */
#endif
}

void compile_lexica (char* base, Options *opt)
{ char blx_file[MAXPATHLEN];
  char *lexgen_cmdline[10];
  int nrargs = 0;

  /* Compose lexgen command line */
  if (!have_lexicon (base)) return;
  lexgen_cmdline[nrargs++] = "agfl-lexgen";
  if (opt -> verbose)     lexgen_cmdline[nrargs++] = "-v";
  if (opt -> verbose > 1) lexgen_cmdline[nrargs++] = "-v";
  if (opt -> force)       lexgen_cmdline[nrargs++] = "-f";
  lexgen_cmdline[nrargs++] = base;
  lexgen_cmdline[nrargs] = NULL;

  /* Try and execute lexgen */
  try_dump_argv (lexgen_cmdline, opt);
  if (abs_execute (lexgen_cmdline) == 0)
    return;

  /* Some error occurred, delete blx (blf) file */
#ifdef WIN32
  sprintf (blx_file, "%s%s", base, ".blx");
#else
  snprintf (blx_file, MAXPATHLEN, "%s%s", base, ".blx");
#endif
  delete_file (blx_file, opt -> verbose);
  abs_exit (1);
}

/*
 * To make sure that the agfl-coder, agfl-as we execute belong to the
 * same installation of AGFL as this driver, prepend some directories to
 * the search $PATH:
 *
 * - The path with which the driver was called (if any).
 *   If this is a relative path it should stay valid since we never
 *   change directories.
 * - As a fallback, the directory where this AGFL was originally
 *   installed.
 *
 * This is done before options processing so that potentially an option
 * can override this by prepending even another directory name.
 */
void adjust_search_path (char *programname)
{
#if HAVE_SETENV && defined(INSTALLDIR)
  char *path = getenv ("PATH");
  char *installdir = INSTALLDIR;
  char *slash, *newenv;
  dstring newpath;

  slash = strrchr (programname, '/');
  newpath = abs_init_dstring (strlen (path) + 1 +
			      strlen (installdir) + 1 +
			      (slash ? slash-programname+1 : 0));
  if (slash)
    { /* Append the given pathname up to the last /, */
      char *c = programname;
      while (c < slash) abs_append_dstring_c (newpath, *c++);
      abs_append_dstring_c (newpath, ':');
    }
  /* then append the installation directory, */
  abs_append_dstring (newpath, installdir);
  abs_append_dstring_c (newpath, ':');
  /* then append the old setting of $PATH, */
  abs_append_dstring (newpath, path);

  /* and finally put the result in the environment. */
  newenv = abs_finish_dstring (newpath);
  setenv ("PATH", newenv, 1);
  abs_free (newenv, "new PATH");
#endif /* HAVE_SETENV */
}

int main (int argc, char* argv[])
{ Options options;
  char base_name[MAXPATHLEN];

  /* adjust search path */
  adjust_search_path (argv[0]);

  /* Parse options */
  init_options (&options);
  process_options (argc, argv, &options);

  /* Pick the right grammar and compile it */
  get_base_name (base_name, options.grammar_file, ".gra");
  compile_grammar (base_name, &options);
  assemble_code (base_name, &options);

  /* lexicon generator decides if it needs to make a lexicon..: */
  compile_lexica (base_name, &options);
  return (0);
}
