/*
   File: agfl-lexgen.c
   Defines main program for agfl-lexgen

   Copyright 2008 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id: agfl-lexgen.c,v 1.12 2008/12/18 14:25:28 marcs Exp $"   
*/

/* System includes */
#include <stdio.h>
#include <stdlib.h>

/* Libabase includes */
#include <abase_error.h>
#include <abase_meminfo.h>
#include <abase_fileutil.h>

/* Local includes */
#include "options.h"
#include "globals.h"
#include "dyn_array.h"
#include "lif_parser.h"
#include "affix_values.h"
#include "entries.h"
#include "nonterminals.h"
#include "datfct_parser.h"

static void generate_info_output ()
{ FILE *info = abs_fopen (info_fname, "w");
  generate_affix_values_info (info);
  generate_nonterminals_info (info);
  generate_entries_info (info);
  fclose (info);
};

static void generate_dump_output ()
{ FILE *dump = abs_fopen (dump_fname, "w");
  dump_vocabulary_entries (dump);
  dump_fact_tables (dump);
  fclose (dump);
};

static void write_lexicon_file ()
{ BinFile blx = abs_bin_fopen (lexicon_fname, "w");
  abs_bin_save_version (blx, "binary lexicon");
  bin_save_text_array (blx, lexicon_names);
  bin_save_text_array (blx, fact_table_names);
  bin_save_affix_values (blx);
  bin_save_nonterminals (blx);
  bin_save_calls (blx);
  bin_save_entries (blx);
  bin_save_vocabularies (blx);
  bin_save_fact_tables (blx);
  abs_bin_save_eof (blx);
  abs_bin_fclose (blx);
}

static int verify_new_lif_file (BinFile old_blx)
{ abs_bin_verify_version (old_blx, "binary lexicon");
  if (bin_cmp_text_array (old_blx, lexicon_names)) return (1);
  if (bin_cmp_text_array (old_blx, fact_table_names)) return (1);
  if (has_new_lif_affix_values (old_blx)) return (1);
  if (has_new_lif_nonterminals (old_blx)) return (1);
  return (0);
}

static int verify_dat_file_age (char *name, int fct)
{ char path[MAXPATHLEN + 1];
  time_t blx_age, file_age;

  /* We should not be able to fail on this one, having opened it recently */
  if (!abs_file_mtime (lexicon_fname, &blx_age)) return (1);
  sprintf (path, "%s.%s", name, (fct)?FCT_SUFFIX:DAT_SUFFIX);

  /* If the source file does not exist anymore, die */
  if (!abs_file_mtime (path, &file_age))
    abs_fatal ("could not determine age of file '%s'", path);
  
  /* Recompile if blx is older than dat */
  return (blx_age <= file_age);
}

static int recompilation_necessary ()
{ BinFile old_blx;
  int stat, ix;

  /* If forced, do it */
  if (lexgen_forced) return (1);

  /* Check if the old binary lexicon matches with the lif */
  if (!abs_file_exists (lexicon_fname)) return (1);
  old_blx = abs_bin_fopen (lexicon_fname, "r");
  stat = verify_new_lif_file (old_blx);
  abs_bin_fclose (old_blx);
  if (stat) return (1);
  if (verbose) abs_message ("    lexicon interface did not change");

  /* Check if the old binary lexicon is younger than the source lexicon files */
  for (ix = 0; ix < lexicon_names -> size; ix++)
    { stat = verify_dat_file_age (lexicon_names -> array[ix], 0);
      if (stat && verbose)
	abs_message ("    lexicon %s changed since last binary lexicon generation",
		     lexicon_names -> array[ix]);
      if (stat) return (1);
    };

  /* Check if the old binary lexicon is younger than the source fact files */
  for (ix = 0; ix < fact_table_names -> size; ix++)
    { stat = verify_dat_file_age (fact_table_names -> array[ix], 1);
      if (stat && verbose)
	abs_message ("    fact table %s changed since last binary lexicon generation",
		     fact_table_names -> array[ix]);
      if (stat) return (1);
    };

  if (verbose) abs_message ("    lexicon and fact table files did not change");
  return (0);
}

int main (int argc, char **argv)
{ parse_command_line (argc, argv);
  init_affix_values ();
  init_nonterminals ();

  parse_lexicon_interface ();
  calculate_affix_values_closure ();
  if (!recompilation_necessary ())
    return (0);

  init_entries ();
  parse_datfct_files ();

  if (generate_info) generate_info_output ();
  if (generate_dump) generate_dump_output ();
  write_lexicon_file ();

  if (verbose) abs_report_meminfo ();
  return (0);
}
