/*
   File: nonterminals.c
   Maintains the administration of all lexicon nonterminals with
   their formal arguments (lexicon nonterminal declarations) as
   well as all occurences of a nonterminal with actuals i.e.
   the lexicon calls occurring in a lexicon.

   Copyright 2007 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id: nonterminals.c,v 1.7 2007/10/31 14:57:50 marcs Exp $"
*/

/* system includes */
#include <stdio.h>
#include <string.h>

/* libabase includes */
#include <abase_repr.h>
#include <abase_error.h>
#include <abase_memalloc.h>
#include <abase_fileutil.h>

/* local includes */
#include "options.h"
#include "dyn_array.h"
#include "affix_values.h"
#include "nonterminals.h"

/*
   We have to maintain an administration of all nonterminals with
   their arity and formal parameters in such a way, that each
   nonterminal is stored only once.

   The basic administration will therefore use hashing techniques for lookup.
   Once a nonterminal has been stored, its index in the administration
   is used as its identification.
*/
typedef struct nonterminal_rec
{ char *nont_name;
  int nont_nr;
  int_array formals;
} *nonterminal;

typedef struct call_rec
{ int nont_id;
  int_array actuals;
} *call;

static ptr_array all_nonterminals;
static ptr_array all_calls;
static hash_container nonterminal_hash_container;
static hash_container call_hash_container;

#define streq(s1,s2) (strcmp((s1),(s2)) == 0)
static int hash_nonterminal (char *name, int arity, int hsize)
{ unsigned value = (unsigned) arity;
  unsigned result;
  char *ptr;

  for (ptr = name; *ptr; ptr++) value = (131 * value + ((unsigned int) (*ptr))) & 0x7ffffff;
  result = value % ((unsigned) hsize);
  return ((int) result);
};

static int hash_call (int nont_id, int_array actuals, int hsize)
{ unsigned value = (unsigned) nont_id;
  unsigned result;
  int ix;

  for (ix = 0; ix < actuals -> size; ix++)
    value = (997 * value + ((unsigned) actuals -> array[ix])) & 0x1fffff;
  result = value % ((unsigned) hsize);
  return ((int) result);
};

int register_new_nonterminal (char *nont_name, int nont_nr, int_array formals)
{ int hindex = hash_nonterminal (nont_name, formals -> size, affix_hash_size);
  hash_list hl = nonterminal_hash_container[hindex];
  nonterminal new_nont;
  int nont_index, ix;

  /* Locate nonterminal on hash list */
  for (ix = 0; ix < hl -> size; ix++)
    { nonterminal nont;
      int arity;
      nont_index = hl -> array[ix];
      nont = (nonterminal) (all_nonterminals -> array[nont_index]);
      arity = nont -> formals -> size;
      if (streq (nont_name, nont -> nont_name) && (formals -> size == arity))
	return (-1);	/* This should not happen */
    };

  /* Nonterminal + arity combination is new: create a unique copy */
  nont_index = all_nonterminals -> size;
  new_nont = (nonterminal) abs_malloc (sizeof (struct nonterminal_rec), "register_new_nonterminal");
  new_nont -> nont_name = nont_name;	/* lif_parser did alloc */
  new_nont -> nont_nr = nont_nr;	/* Coder nonterminal nr */
  new_nont -> formals = formals;
  app_ptr_array (all_nonterminals, (void *) new_nont);
  app_hash_list (hl, nont_index);
  return (nont_index);
}

int register_new_call (int nont_id, int_array actuals)
{ int hindex = hash_call (nont_id, actuals, affix_hash_size);
  hash_list hl = call_hash_container[hindex];
  int call_index, ix;
  call new_call;

  /* Locate call on hash list */
  for (ix = 0; ix < hl -> size; ix++)
    { int_array call_actuals;
      call this_call;

      /* Check if this call equals the one we seek */
      call_index = hl -> array[ix];
      this_call = (call) all_calls -> array[call_index];
      if (nont_id != this_call -> nont_id) continue;
      call_actuals = this_call -> actuals;

#ifdef DEBUG
      if (call_actuals -> size != actuals -> size)
	abs_bug ("register_new_call",
		 "Arity mismatch for nont %d between registered %d and actual %d",
		  nont_id, call_actuals -> size, actuals -> size);
#endif

      /* Nonterminals id match, now check the actuals */
      if (equal_int_array (call_actuals, actuals))
        return (call_index);
    };

  /* Nonterminal id + actuals combination is new: create a new call */
  call_index = all_calls -> size;
  new_call = (call) abs_malloc (sizeof (struct call_rec), "register_new_call");
  new_call -> nont_id = nont_id;
  new_call -> actuals = init_int_array (actuals -> size);
  for (ix = 0; ix < actuals -> size; ix++)
    app_int_array (new_call -> actuals, actuals -> array[ix]);
  app_ptr_array (all_calls, (void *) new_call);
  app_hash_list (hl, call_index);
  return (call_index);
}

/*
   Lookup of nonterminals
*/
int lookup_nonterminal (char *nont_name, int arity)
{ int hindex = hash_nonterminal (nont_name, arity, affix_hash_size);
  hash_list hl = nonterminal_hash_container[hindex];
  int ix;
  for (ix = 0; ix < hl -> size; ix++)
    { int nont_index = hl -> array[ix];
      nonterminal nont = (nonterminal) (all_nonterminals -> array[nont_index]);
      if (streq (nont_name, nont -> nont_name) && (arity == nont -> formals -> size))
	return (nont_index);
    };

  /* Not found */
  return (-1);
}

int_array formals_from_nonterminal (int nont_index)
{ nonterminal nont = (nonterminal) (all_nonterminals -> array[nont_index]);
  return (nont -> formals);
}

int nr_of_nonterminals ()
{ return (all_nonterminals -> size);
}

int nr_of_calls ()
{ return (all_calls -> size);
}

/*
  Generate info file
*/
void generate_nonterminals_info (FILE *info)
{ int ix;
  fprintf (info, "Nonterminal table:\n");
  for (ix = 0; ix < all_nonterminals -> size; ix++)
    { nonterminal nont = (nonterminal) (all_nonterminals -> array[ix]);
      int arity = nont -> formals -> size;
      fprintf (info, "%4d: [%d] %s/%d", ix, nont -> nont_nr, nont -> nont_name, arity);
      if (arity)
	{ int iy;
	  fprintf (info, " (");
	  for (iy = 0; iy < arity; iy++)
	    { if (iy) fprintf (info, ", ");
	      fprintf (info, "%d", nont -> formals -> array[iy]);
	    };
	  fprintf (info, ")");
        };
      fprintf (info, "\n");
    };

  fprintf (info, "Call table:\n");
  for (ix = 0; ix < all_calls -> size; ix++)
    { call the_call = (call) (all_calls -> array[ix]);
      int nont_id = the_call -> nont_id;
      int_array actuals = the_call -> actuals;
      fprintf (info, "%4d: %d", ix, nont_id);
      if (actuals -> size)
        { int iy;
          fprintf (info, " (");
          for (iy = 0; iy < actuals -> size; iy++)
            { if (iy) fprintf (info, ", ");
              fprintf (info, "%d", actuals -> array[iy]);
            };
          fprintf (info, ")");
        };
      fprintf (info, "\n");
    };
}

/*
   Dumping of information
*/
void dump_call (FILE *dump, int call_id)
{ call the_call = (call) (all_calls -> array[call_id]);
  int nont_id = the_call -> nont_id;
  nonterminal nont = (nonterminal) (all_nonterminals -> array[nont_id]);
  int_array fpars = nont -> formals;
  int_array actuals = the_call -> actuals;
  int ix;
  fprintf (dump, "%s", nont -> nont_name);
  if (!fpars -> size) return;
  fprintf (dump, " (");
  for (ix = 0; ix < fpars -> size; ix++)
    { if (ix) fprintf (dump, ", ");
      dump_affix_value (dump, fpars -> array[ix], actuals -> array[ix]);
    };
  fprintf (dump, ")");
}

/*
   Binary save the nonterminals
   Binary verify the lexicon header against the lif
*/
void bin_save_nonterminals (BinFile bf)
{ int ix;
  abs_bin_save_int (bf, all_nonterminals -> size);
  for (ix = 0; ix < all_nonterminals -> size; ix++)
    { nonterminal nont = (nonterminal) (all_nonterminals -> array[ix]);
      abs_bin_save_string (bf, nont -> nont_name);
      abs_bin_save_int (bf, nont -> nont_nr);
      bin_save_int_array (bf, nont -> formals);
    };
}

void bin_save_calls (BinFile bf)
{ int ix, iy;
  abs_bin_save_int (bf, all_calls -> size);
  for (ix = 0; ix < all_calls -> size; ix++)
    { call the_call = (call) (all_calls -> array[ix]);
      int_array actuals = the_call -> actuals;
      abs_bin_save_int (bf, the_call -> nont_id);
      for (iy = 0; iy < actuals -> size; iy++)
	abs_bin_save_int (bf, actuals -> array[iy]);
    };
}

static int is_new_nonterminal (BinFile bf, nonterminal nont)
{ int ival, size, nont_nr, ix;
  char *sval;
  abs_bin_load_string (bf, &sval);
  if (!streq (sval, nont -> nont_name)) return (1);
  abs_free (sval, "is_new_nonterminal");
  abs_bin_load_int (bf, &nont_nr);
  if (nont_nr != nont -> nont_nr) return (1);
  abs_bin_load_int (bf, &size);
  if (size != nont -> formals -> size) return (1);
  for (ix = 0; ix < nont -> formals -> size; ix++)
    { abs_bin_load_int (bf, &ival);   
      if (ival != nont -> formals -> array[ix])
	return (1);
    };
  return (0);
}

int has_new_lif_nonterminals (BinFile bf)
{ int ix, value;
  abs_bin_load_int (bf, &value);
  if (value != all_nonterminals -> size) return (1);
  for (ix = 0; ix < all_nonterminals -> size; ix++)
    if (is_new_nonterminal (bf, (nonterminal) (all_nonterminals -> array[ix])))
      return (1);
  return (0);
}

/*
   Initialization stuff
*/
void init_nonterminals ()
{ int ix;
  all_nonterminals = init_ptr_array (affix_hash_size);
  nonterminal_hash_container =
	(hash_container) abs_calloc (affix_hash_size, sizeof (hash_list), "init_nonterminals");
  for (ix = 0; ix < affix_hash_size; ix++)
    nonterminal_hash_container[ix] = new_hash_list ();
  all_calls = init_ptr_array (param_hash_size);
  call_hash_container =
        (hash_container) abs_calloc (param_hash_size, sizeof (hash_list), "init_calls");
  for (ix = 0; ix < param_hash_size; ix++)
    call_hash_container[ix] = new_hash_list ();

}
