/*
   File: lxcn_lexicon_impl.h
   Defines the internal structures to store a lexicon

   Copyright 2007 Radboud University of Nijmegen
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   CVS ID: "$Id: lxcn_lexicon_impl.h,v 1.4 2007/10/31 14:58:06 marcs Exp $"
*/
#ifndef IncLexiconImpl
#define IncLexiconImpl

/* Libabase includes */
#include <abase_repr.h>

/* Local includes */
#include "lxcn_vocabulary.h"
#include "lxcn_lexicon.h"

#ifdef __cplusplus
extern "C" {
#endif

/*-------------------------------------------------------------------------
// Structure for storing the set affixes
//
// name:    Affix name
// bitset:  The bitset associated with this affix
//	    Note that we read and store the value as an unsigned 32 bit
//	    which is not the same as the unsigned long still used in
//	    the runtime system.
// is_nont: If true, this affix is a affix nonterminal
// lhs[]:   The indexes of the left hand sides (affix nonterminals),
//	    of which this affix is a subset. The nr_lhs_sides indicates
//	    the cardinality of this set.
//
// The INT and TEXT affixes are saved as rows of INTs/TEXTs
//-----------------------------------------------------------------------*/
typedef struct
{ char *	name;
  Bitset32	bitset;
  int		is_nont;
  int		nr_lhs_sides;
  int *		lhs;
} SetAffix;

/*-------------------------------------------------------------------------
// Structure for storing the nonterminals and calls
// Nonterminal:
// name:  Nonterminal name
// nont_nr: Nonterminal number given by coder
// arity: Number of parameters
// fpars: The formal parameters of the nonterminal:
//	  Note: INT is coded as -1, TEXT as -2,
//	        a set affix is coded as its index in the set_affixes
//
// Call:
// nont_id:	Entry in the nonterminal table
// actuals:	Actual arguments, coded as index in the affix tables
//		Note the formal type can be obtained from the Nonterminal
//-----------------------------------------------------------------------*/
#define FormalINT   (-1)
#define FormalTEXT  (-2)
typedef struct
{ char *	name;
  int		nont_nr;
  int 		arity;
  int *		fpars;
} Nonterminal;

typedef struct
{ int		nont_id;
  int *		actuals;
} Call;

struct new_lexicon_rec
{ int 		nr_int_affixes;
  int *		int_affixes;
  int		nr_text_affixes;
  char **	text_affixes;
  int		nr_set_affixes;
  SetAffix *	set_affixes;
  int		nr_nonterminals;
  Nonterminal *	nonterminals;
  int		nr_calls;
  Call *	calls;
  int		nr_lex_entries;
  int **	all_entries;
  int		nr_vocabularies;
  int *		all_lexeme_markers;
  Vocabulary *  all_vocabularies;
};

/*------------------------------------------------------------------------------
// Format of lexicon file (as delivered by agfl-lexgen)
//
// 0) lexicon header
//
// The affix table: the nr of lif affixes is saved to check for lexicon recompilation.
//
// 1a) int		nr_lif_affixes
// 1b) int      	nr_set_affixes
// 1c) setaffix		set_affixes[]
// 2a) int		nr_int_affixes
// 2b) int		int_affixes[]
// 3a) int		nr_text_affixes
// 3b) char *		text_affixes[]
//
// The nonterminal and call table:
//
// 4a) int		nr_nonterminals
// 4b) nonterminal	nonterminals[]
// 5a) int		nr_calls
// 5b) call		calls[]
//
// The mapping from list nr to calls
// 6a) int		nr_entries
// 6b) entry		all_entries[] (saved as int *)
//
// The vocabularies
// 7a) int		nr_vocabularies
// 7b) int		all_lexeme_markers
// 7a) int		nr_vocabularies
// 7b) int		all_lexeme_markers[]
// 7c) vocabulary	all_vocabularies[]
// 7c) vocabulary	all_vocabularies[]
//----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif
#endif /* IncLxcnLexiconImpl */

