// Scanner class definition for the old style .dat files.
//
// Copyright 2001, KUN.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

// $Id: scanner.h,v 1.2 2003/10/17 12:36:29 pspiertz Exp $


#ifndef scanner_h
#define scanner_h

using namespace std;
#include <stdlib.h>
#include <FlexLexer.h>


//------------------------------------------------------------------------------
// Class Scanner
//
// Note:
//	Compound tokens have a mnemonic value >= 256, other tokens are
//	refered to with their ascii value.
//
//	On syntax error, the current token is discarded and
//	an error token is pushed in front of the input.
//------------------------------------------------------------------------------

class Scanner: private yyFlexLexer
{
    public:
        enum
        {
            MODULE = 256,
            ROOT,
            DEFINES,
            USES,
            PENALTY,
            SKIP,
            MATCH,
            PREV,
            AHEAD,
            PROMPT,
            DOUBLE_COLON,
            INT,
            TEXT,
            ID,
            STRING,
            LEXEME,
            NUMBER
        };

        enum { MAX_SYNTAX_ERRORS = 10 };
        enum { ERROR = -1, END_OF_FILE = 0 };

    private:
        int ahead;
        unsigned err_cnt;
        const char* fname;

    public:
        Scanner();
        Scanner(istream&, const char* = 0);

        void attach(istream&, const char* = 0);

        int is_token(int) const;
        int was_token(int);
        void should_be_token(int, char*);
        int look_ahead() const;
        void skip_token();
        int next_token();

        int eof() const;
        int error() const;
        unsigned length() const;
        unsigned line() const;
        char* text() const;
        char* number() const;

        void syntax_error(char*);
        unsigned errors() const;
};

inline Scanner::Scanner():
    yyFlexLexer()
{
    err_cnt = 0;
    fname = "stdin";
    ahead = yylex();
}

inline Scanner::Scanner(istream& ins, const char* fn):
    yyFlexLexer(&ins)
{
    err_cnt = 0;
    fname = fn;
    ahead = yylex();
}

inline void Scanner::attach(istream& ins, const char* fn)
{
    switch_streams(&ins, 0); 
    fname = fn;
    ahead = yylex();
    yylineno = 1;
}

inline int Scanner::is_token(int token) const
{
    return ahead == token;
}

inline int Scanner::was_token(int token)
{
    if (ahead == token) {
        ahead = yylex();
        return 1;
    } else {
        return 0;
    }
}

inline void Scanner::should_be_token(int token, char* error)
{
    if (ahead == token) {
        ahead = yylex();
    } else {
        syntax_error(error);
    }
}

inline int Scanner::look_ahead() const
{
    return ahead;
}

inline void Scanner::skip_token()
{
    ahead = yylex();
}

inline int Scanner::next_token()
{
    int token = ahead;
    ahead = yylex();
    return token;
}

inline int Scanner::eof() const
{
    return ahead == END_OF_FILE;
}

inline int Scanner::error() const
{
    return ahead == ERROR;
}

inline char* Scanner::text() const
{
    return yytext;
}

inline char* Scanner::number() const
{
    return yytext;
}

inline unsigned Scanner::length() const
{
    return yyleng;
}

inline unsigned Scanner::line() const
{
    return yylineno;
}

inline unsigned Scanner::errors() const
{
    return err_cnt;
}


#endif // scanner_h
