
/* rtsesc - handling of backslash escapes in strings
 *
 * Copyright 2000 KUN.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: rtsesc.c,v 1.4 2002/04/25 16:00:31 murphy Exp $ */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#else /* HAVE_MALLOC_H */
#include <stdlib.h>
#endif /* HAVE_MALLOC_H */

#include <stdio.h>
#include <errno.h>
#include "rtsio.h"	/* for rtsMessage */
#include "rtsesc.h"

static int
try_process_3_octal_digits(char const *src)
{
    if ((*src < '0') || (*src > '7') || (src[1] < '0') || (src[1] > '7')
					|| (src[2] < '0') || (src[2] > '7')) {
	return -EINVAL;
    }
    return (*src * 64) + (src[1] * 8) + src[2];
}

int
dupsubstr_unescaped(char const *src, char const *supto, char **retdst)
/* returns a copy of src with backslash escapes removed
**      e.g. "abc\"def\\ghi" becomes "abc"def\ghi"
*/
{
    /* this may alloc a little more than we really need: */
    char *ndst = (char *) malloc(supto - src + 1);

    if (!ndst) {
	rtsMessage("** Out of memory while copying '%s'\n", src);
	return -ENOMEM;
    }
    *retdst = ndst;
    while (src < supto) {
	if (*src == '\\') {
	    src++;
	    if ((*src >= 0) && (*src < ' ')) {
		rtsMessage("** Illegal \\ escape with char 0x%02x\n", *src);
		return -EINVAL;
	    } else if ((*src >= '0') && (*src <= '9')) {
		int bval = try_process_3_octal_digits(src);
		if ((bval < 0) || (bval > 255)) {
		    rtsMessage("** Expected 3 octal digits < 0377 at '%s'\n",
									src);
		    return -EINVAL;
		}
		*ndst = bval;
		src += 2; /* 3rd is skipped below */
	    } else if (((*src >= 'A') && (*src <= 'Z'))
		       || ((*src >= 'a') && (*src <= 'z'))) {
		switch (*src) {
		    case 'n': *ndst = '\n'; break;
		    case 'r': *ndst = '\r'; break;
		    case 't': *ndst = '\t'; break;
		    default:
		        rtsMessage("** Illegal \\ escape with letter '%c'\n",
									*src);
		        return -EINVAL;
		}
	    } else {
		*ndst = *src;
	    }
	} else {
	    *ndst = *src;
	}
	ndst++;
	src++;
    }
    *ndst = '\0';
    return 0;
}

int
dupstr_escaped(char const *src, char **retdst)
/* returns a copy of src with some backslash escapes added
**      e.g. "abc"def\ghi" becomes "abc\"def\\ghi"
** Currently used to format affix names, terminal names (unused?),
**	text affixes, regexp patterns (so rtstrelinp can read -G output).
** - escapes linefeed, return, tab as \n, \r, \t
** - escapes other control chars (1..31) as octal (e.g. ctrl-g becomes \007)
**	(char 0 is end of string)
** - inserts a \ before: \ "
** does _not_ insert a \ before: - [ ]	(makes regexps look misleadingly)
** - inserts a \ before a - at begin or end of string, other -'s left plain
*/
{
    char *ndst = (char *) malloc(strlen(src) * 4 + 1); /* probably overkill */

    if (!ndst) {
	rtsMessage("** Out of memory while copying '%s'\n", src);
	return -ENOMEM;
    }
    *retdst = ndst;
    if (*src == '-') {
	*ndst = '\\'; *++ndst = *src++; ndst++;
    }
    while (*src) {
	switch (*src) {
	    case '\n': *ndst = '\\'; *++ndst = 'n'; break;
	    case '\r': *ndst = '\\'; *++ndst = 'r'; break;
	    case '\t': *ndst = '\\'; *++ndst = 't'; break;
	    case '\\': *ndst = '\\'; *++ndst = *src; break;
	    case '"': *ndst = '\\'; *++ndst = *src; break;
	    case '-':
		if (!src[1]) {	/* this - is last char */
			*ndst = '\\'; *++ndst = *src; break;
		}
	    default:
		if ((*src > 0) && (*src < ' ')) {
		    sprintf(ndst, "\\%03o", *src);
		    ndst += 3; /* fourth added outside switch */
		} else {
		    *ndst = *src;
		}
	}
	ndst++;
	src++;
    }
    *ndst = '\0';
    return 0;
}
