// Implement parsing procedures for parsing old style .dat files.
//
// Copyright 2001, KUN.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

// $Id: parser.cc,v 1.2 2001/03/12 09:00:21 ejv Exp $


#include <iostream.h>

#include "parser.h"


//------------------------------------------------------------------------------
// Function:
//      static inline string to_lowercase(string& str)
//
// Description:
//      Converts str to lowercase and returns the result.
//
// Return value:
//      Lowercased str.
//------------------------------------------------------------------------------

static inline string to_lowercase(char* char_str)
{
    string res("");

    while (*char_str) {
        res += tolower(*(char_str++));
    }

    return res;
}


//------------------------------------------------------------------------------
// Function:
//	inline bool Parser::is_parameter(string& param)
//
// Description:
//	Try to parse parameter. The parameter, if any, is assigned to
//	param.
//
// Return value:
//	True if parameter was ahead, false otherwise.
//------------------------------------------------------------------------------

inline bool Parser::is_parameter(string& param)
{
    switch (look_ahead()) {
        case ID: {
            param = to_lowercase(text());
            skip_token();
            if (is_token('|')) {
                param += " | ";
                while (was_token('|')) {
                    if (is_token(ID)) {
                        param += to_lowercase(text());
                        skip_token();
                    } else {
                        syntax_error("identifier expected");
                    }
                }
            }
            return true;
        }

        case STRING: {
            param = text();
            skip_token();
            return true;
        }

        case NUMBER:
            param = number();
            skip_token();
            return true;

        case TEXT:
            param = "TEXT";
            skip_token();
            return true;

        case INT:
            param = "INT";
            skip_token();
            return true;

        default:
            return false;
    }
}


//------------------------------------------------------------------------------
// Function:
//	void Parser::should_be_parameters(string& params)
//
// Description:
//	Parses list of parameters. The resulting list is
//	assigned to params.
//
// Side effects:
//	Enters parameter lists into parameter table.
//
// Note:
//	Parameter lists should be created from right to left, in
//	order to maximize the sharing of tails of parameter lists.
//	Otherwise memory will be wasted due to unsharable tails of
//	text and integer parameters.
//------------------------------------------------------------------------------

void Parser::should_be_parameters(string& params)
{
    string param;

    if (is_parameter(param)) {
        if (was_token(',')) {
            param += ", ";
            should_be_parameters(params);
        }

        params = param + params;
    } else {
        syntax_error("parameter expected");
    }
}


//------------------------------------------------------------------------------
// Function:
//	inline void Parser::should_be_nonterm(string& entries)
//
// Description:
//	Parse nonterminal and add entry containing nonterminal to
//	entries.
//
// Side effects:
//	The nonterminal is entered into the nonterminal table.
//------------------------------------------------------------------------------

inline void Parser::should_be_nonterm(string& nonterm)
{
    if (is_token(ID)) {
        nonterm = text();
        skip_token();
        if (was_token('(')) {
            string params;
            should_be_parameters(params);
            should_be_token(')', "`)' expected");
            nonterm += " (" + params + ')';
        }
    } else {
        syntax_error("nonterminal expected");
    }
}

inline void Parser::should_be_lhs(StringList& nonterms)
{
    do {
        string nonterm;
        should_be_nonterm(nonterm);
        nonterms.push_back(nonterm);
    } while (was_token(','));
}


//------------------------------------------------------------------------------
// Function:
//	inline void Parser::should_be_rhs(EntryList& entries)
//
// Description:
//	Parse lexemes with their frequencies.
//
// Side effects:
//	For each lexeme, its frequency (if any) is assigned
//	to the frequency of the entries. Then, the lexeme
//	and its entries are entered into the rule table.
//	The rule table should copy each list of entries.
//------------------------------------------------------------------------------

inline void Parser::should_be_rhs(ostream& os, StringList& entries)
{
    do {
        string freq("");
        if (was_token('[')) {
            if (is_token(NUMBER)) {
                freq = '[';
                freq += number();
                freq += ']';
                skip_token();
            } else {
                syntax_error("number expected");
            }
            should_be_token(']', "`]' expected");
        }

        if (is_token(LEXEME)) {
            string term = text();
            skip_token();
            generate_new_format(os, entries, term, freq);
        } else {
            syntax_error("quoted string expected");
        }
    } while (was_token(';'));
}

void Parser::should_be_rules(ostream& os)
{
    do {
        StringList entries;
        should_be_lhs(entries);
        should_be_token(':', "`:' expected");
        should_be_rhs(os, entries);
        should_be_token('.', "`.' expected");
        if (error()) {
            while (!eof() && !was_token('.')) {
                skip_token();
            }
        }
    } while (!eof());
}

void Parser::generate_new_format(ostream& os, StringList& entries, string term,
                                 string freq)
{
    for (StringList::iterator i = entries.begin(); i != entries.end(); ++i) {
        os << term << "\t\t" << freq << "\t" << *i << endl;
    }
}

