#define MY
/* Compiler driver for agfl. Checks installation and configuration of system
 * and availability of auxiliary tools. Implements make-like functionality for
 * grammar and lexicon.
 *
 * Copyright 2003, KUN.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* "$Id: gen.c,v 1.38 2003/08/12 19:28:10 pspiertz Exp $" */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <glib.h>
#include <assert.h>

/*
//------------------------------------------------------------------------------
// Platform specific installation details. Define
//	WIN32		Windows NT
//	LINUX		Linux
//	SOLARIS		Solaris
//	SUNOS		SunOS
//	AIX		AIX
//	PARIX		Parix
//------------------------------------------------------------------------------
*/


/* set default target flags */
#ifndef TCPU
#define TCPU "unknown cpu"
#endif
#ifndef TOS
#define TOS "unknown OS"
#endif


#if defined(WIN32)
#  define DIR_SEP   "\\"
#  define RM        "del"
#  define EXE       ".exe"
#  define OUT_EXE   ".exe"
#  ifndef GENBASE
#    define GENBASE "\\agfl"
#  endif
#  ifndef GENGLIBFLAG
#    define GENGLIBFLAG "-L\\agfl\\lib -lglib -liberty"
#  endif
#else
#  define DIR_SEP   "/"
#  define RM        "rm"
#  define EXE       ""
#  define OUT_EXE   ""
#endif

/*
** GENAS is the assembly command used by the AGFL compiler.
** Note: take care that gen and agfl assume same default for GENAS!
*/

#ifndef GENAS
#  define GENAS	"gcc -c -x assembler-with-cpp -"
#endif

/*
** GENCC is the C compiler used for linking with the run-time library.
*/

#ifndef GENCC
#  define GENCC	"gcc"
#endif

/*
// Developers may use mygen instead of gen. It has more options
// available and does not pass optimization options to agfl by default.
// Furthermore, mygen passes debug options to the linker.
*/

#ifdef MY
#  define DIRECTORS_FLAG	0
#  define LEFTFAC_FLAG		0
#  define NEGMEMO_FLAG		0
#  define POSMEMO_FLAG		0
#  define STRIP_FLAG		0
#else
#  define DIRECTORS_FLAG	1
#  define LEFTFAC_FLAG		0
#  define NEGMEMO_FLAG		1
#  define POSMEMO_FLAG		1
#  define STRIP_FLAG		1
#endif

/*
// On aout platforms, agfl must prefix labels in the code with underscores.
*/

#if defined(SUNOS) || defined(WIN32) || (__OpenBSD__)
#  define USCORE_FLAG	1
#else
#  define USCORE_FLAG	0
#endif

void
output_error(const char* fmt, ...)
{
    va_list argp;

    fputs("gen: ", stderr);

    va_start(argp, fmt);
    vfprintf(stderr, fmt, argp);
    va_end(argp);

    fputs("\n", stderr);
}

void
output_log(const char* fmt, ...)
{
    va_list argp;

    va_start(argp, fmt);
    vfprintf(stderr, fmt, argp);
    va_end(argp);

    fputs("\n", stderr);
}

static void
show_version(int verbose)
{
    printf("%s version %s, Copyright 2003, KUN.\n", PACKAGE, VERSION);
    if (verbose) {
        char* tcpu;
	char* tos;
        tcpu = (char*) malloc(sizeof(TCPU) + 1);
	strcpy(tcpu, TCPU);
        tos = (char*) malloc(sizeof(TOS) + 1);
	strcpy(tos, TOS);
        printf("Compiled on %s %s\n",tcpu,tos);
    }
}

static char gen_help_mesg[] =
  "Usage: gen [ -a ] [ -c ] [ -f ] [ -G ] [ -h ] [ -m ] [ -p ] [ -v ] [ -w ] <grammar>\n"
  "\n"
  "Options are:\n"
  "    -a           analyze additional properties\n"
  "    -c           show counters after parsing\n"
  "    -f           force new generation\n"
  "    -G           generate generative grammar\n"
  "    -h           show this help, and exit\n"
  "    -m           prefix nonterminals with module name\n"
  "    -p           profile grammar\n"
  "    -V           show version\n"
  "    -v           be verbose\n"
  "    -w           suppress compiler warnings\n"
  "\n"
  "Arguments are:\n"
  "    <grammar>    name of grammar file  (.gra)\n"
  "\n";

#ifdef MY
static char mygen_help_mesg[] =
  "Additional options for mygen:\n"
  "    -e           link Electric Fence\n"
  "    -g           debug\n"
  "    -l           write log file\n"
  "    -s           strip executable\n"
  "    -t           show compile time\n"
  "    -A           suppress well-formedness analysis\n"
  "    -D           use director set lookahead\n"
  "    -F           left-factorize grammar\n"
  "    -P           use positive memoizing\n"
  "    -N           use negative memoizing\n"
  "    -T           trace\n"
  "    -_           toggle underscore flag\n"
  "\n";
#endif

void
show_help(FILE* file, int verbose)
{
    show_version(verbose);
    fputs(gen_help_mesg, stderr);

#ifdef MY
    fputs(mygen_help_mesg, stderr);
#endif
}

typedef struct
{
    int additional_analysis;            /* -a */
    int counters;                       /* -c */
    int electric_fence;                 /* -e */
    int force;                          /* -f */
    int debug;                          /* -g */
    int generative_grammar;             /* -G */
    int write_log;                      /* -l */
    int profile;                        /* -p */
    int strip;                          /* -s */
    int show_time;                      /* -t */
    int verbose;                        /* -v */
    int suppress_warnings;              /* -w */
    int suppress_analysis;              /* -A */
    int directors;                      /* -D */
    int left_factorize;                 /* -F */
    int positive_memo;                  /* -P */
    int negative_memo;                  /* -N */
    int trace;                          /* -T */
    int underscore;                     /* -_ */
    const char*	grammar_file;
    int module_names;                   /* -m */
} Options;

void init_options(Options* options)
{
    options->additional_analysis = 0;
    options->counters = 0;
    options->electric_fence = 0;
    options->force = 0;
    options->debug = 0;
    options->generative_grammar = 0;
    options->write_log = 0;
    options->profile = 0;
    options->strip = STRIP_FLAG;
    options->show_time = 0;
    options->verbose = 0;
    options->suppress_warnings = 0;
    options->suppress_analysis = 0;
    options->directors = DIRECTORS_FLAG;
    options->left_factorize = LEFTFAC_FLAG;
    options->positive_memo = POSMEMO_FLAG;
    options->negative_memo = NEGMEMO_FLAG;
    options->trace = 0;
    options->underscore = USCORE_FLAG;
    options->grammar_file = NULL;
    options->module_names = 0;
}

void
process_options(int argc, const char** argv, Options* options)
{
    const char*	opt;
    char c;

/*
** Check whether options specified
*/
    if (argc == 1) {
        output_error("compiler driver for %s version %s", PACKAGE, VERSION);
        exit(1);
    }

/*
** Process options
*/
    while((opt = *++argv)) {
        if ((opt[0] == '-') && (opt[1] != '\0')) {
            while((c = *++opt)) {
                switch (c) {
                    case 'a':
                        options->additional_analysis++;
                        break;
                    case 'c':
                        options->counters++;
                        break;
                    case 'f':
                        options->force++;
                        break;
                    case 'v':
                        options->verbose++;
                        break;
                    case 'G':
fprintf(stderr,"This version of AGFL does not yet support building generating grammars.\n"); exit(200);
                        options->generative_grammar = 1;
                        options->directors = 0;
                        options->positive_memo = 0;
                        options->negative_memo = 0;
                        break;
                    case 'h':
                        show_help(stderr,options->verbose);
                        exit(1);
                    case 'm':
                        options->module_names = 1;
                        break;
                    case 'p':
                        options->profile++;
                        break;
                    case 'V':
                        show_version(options->verbose);
                        exit(0);
                    case 'w':
                        options->suppress_warnings++;
                        break;
                    case 'T':
                        options->trace++;
                        break;
#ifdef MY
                    case 'e':
                        options->electric_fence++;
                        break;
                    case 'g':
                        options->debug++;
                        break;
                    case 'l':
                        options->write_log++;
                        break;
                    case 's':
                        options->strip++;
                        break;
                    case 't':
                        options->show_time++;
                        break;
                    case 'A':
                        options->suppress_analysis++;
                        break;
                    case 'D':
                        options->directors++;
                        break;
                    case 'F':
                        options->left_factorize++;
                        break;
                    case 'P':
                        options->positive_memo++;
                        break;
                    case 'N':
                        options->negative_memo++;
                        break;
                    case '_':
                        options->underscore = !options->underscore;
                        break;
#endif /* MY */
                    default:
                        output_error("fatal: unknown option `-%c'", c);
                        output_error("use `gen -h' for help");
                        exit(1);
                }
            }
        } else {
            if(options->grammar_file == NULL) {
                options->grammar_file = opt;
            } else {
                output_error("fatal: duplicate input file specified"
                            " (`%s' and `%s')", options->grammar_file, opt);
                exit(1);
            }
        }
    }

/*
** Check whether options valid
*/
    if (options->grammar_file == NULL) {
        output_error("fatal: no input file specified");
        exit(1);
    }

/*
 * 2002-03-19  PB  reporting of input options added
 */

    fprintf(stdout, "  %s %s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s %s\n",
#if !defined(MY)
        "gen",
#else
        "mygen",
#endif /* MY */
    options->additional_analysis ? " -a" : "",
    options->counters            ? " -c" : "",
    options->electric_fence      ? " -e" : "",
    options->force               ? " -f" : "",
    options->debug               ? " -g" : "",
    options->generative_grammar  ? " -G" : "",
    options->write_log           ? " -l" : "",
    options->profile             ? " -p" : "",
    options->strip               ? " -s" : "",
    options->show_time           ? " -t" : "",
    options->verbose             ? " -v" : "",
    options->suppress_warnings   ? " -w" : "",
    options->suppress_analysis   ? " -A" : "",
    options->directors           ? " -D" : "",
    options->left_factorize      ? " -F" : "",
    options->positive_memo       ? " -P" : "",
    options->negative_memo       ? " -N" : "",
    options->trace               ? " -T" : "",
    options->underscore          ? " -_" : "",
    options->module_names        ? " -m" : "",
    options->grammar_file        ? options->grammar_file : "");
}

void
strip_extension(char* path, const char* ext)
{
    int len1 = strlen(path);
    int len2 = strlen(ext);

    if (len1 > len2 && strcmp(path + len1 - len2, ext) == 0) {
        path[len1 - len2] = '\0';
    }
}

int
file_exists(const char* path)
{
    struct stat sbuf;
    return stat(path, &sbuf) >= 0;
}

/*
// S_ISREG should be in sys/stat.h
*/

#ifndef S_ISREG
#define S_ISREG(m) (((m)&S_IFMT) == S_IFREG)
#endif

int
is_regular_file(const char* path)
{
    struct stat sbuf;
    return stat(path, &sbuf) >= 0 && S_ISREG(sbuf.st_mode);
}

int
is_directory(const char* path)
{
  struct stat sbuf;
  return stat(path, &sbuf) >= 0 && S_ISDIR(sbuf.st_mode);
}

void
get_last_dir(char* name, const char *path)
{
    const char*	dir_sep = DIR_SEP;
    const char*	last_dir;
    const char*	p;

    last_dir = path;
    for (p = path; p[0] != '\0'; p++) {
        if (p[0] == *dir_sep && p[1] != '\0') {
            last_dir = p;
        }
    }

    strcpy(name, last_dir);
}

void
get_base_name(char* base, const char* name, const char* ext)
{
    char	path[PATH_MAX];

    strcpy(base, name);
    strip_extension(base, ext);
    sprintf(path, "%s%s", base, ext);
    if (!file_exists(path)) {
        if (is_directory(base)) {
            char name[PATH_MAX];
            get_last_dir(name, base);
            strcat(base, DIR_SEP);
            strcat(base, name);
            sprintf(path, "%s%s", base, ext);
        }

        if (!file_exists(path)) {
            output_error("fatal: cannot locate input file `%s'", path);
            exit(1);
        }
    }
    if (!is_regular_file(path)) {
        output_error("fatal: `%s' is not a regular file", path);
        exit(1);
    }
}

/*
** Paths to files of AGFL system, to be prefixed with base directory
*/

#define GENBIN		DIR_SEP "bin"
#define GENLIB		DIR_SEP "lib"
#define GENINCLUDE	DIR_SEP "include"

#define LEXGEN		GENBIN DIR_SEP "lexgen" EXE
#define AGFL		GENBIN DIR_SEP "gra2o" EXE
#define LIBNMRTS	GENLIB DIR_SEP "libnmrts.a"
#define LINKNMRTS       "-lnmrts"
#define LIBPMRTS	GENLIB DIR_SEP "libpmrts.a"
#define LINKPMRTS       "-lpmrts"
#define LIBRTSLOG	GENLIB DIR_SEP "librtslog.a"
#define LINKGENRTS      "-lgenrts"
#define LIBGENRTS       GENLIB DIR_SEP "libgenrts.a"
#define LINKDEBUGRTS    "-ldebugrts -lpthread -lgtk -lgdk -lgthread"
#define LIBDEBUGRTS     GENLIB DIR_SEP "libdebugrts.a"
#define MACHDEP		GENINCLUDE DIR_SEP "machdep.h"

typedef struct
{
    char lexgen[PATH_MAX];
    char gra2o[PATH_MAX];
    char include[PATH_MAX];
    char machdep[PATH_MAX];
    char lib[PATH_MAX];
    char librts[PATH_MAX];
    char librtslog[PATH_MAX];
    char libgenrts[PATH_MAX];
    char genas[PATH_MAX];
    char gencc[PATH_MAX];
} GenFiles;

void
check_file_installed(const char* path, const char* descr)
{
    if (!file_exists(path)) {
        output_error("installation problem: cannot find %s `%s'", descr, path);
        exit(1);
    }
}

void
check_installation(GenFiles* files)
{
    char* genbase;
    char* genas;
    char* gencc;

    genbase = getenv("GENBASE");
    if (genbase == NULL) {
        genbase = GENBASE;
    }

    sprintf(files->lexgen,    "%s%s", genbase, LEXGEN);
    sprintf(files->gra2o,     "%s%s", genbase, AGFL);
    sprintf(files->machdep,   "%s%s", genbase, MACHDEP);
    sprintf(files->include,   "%s%s", genbase, GENINCLUDE);
    sprintf(files->lib,       "%s%s", genbase, GENLIB);
    sprintf(files->librts,    "%s%s", genbase, LIBNMRTS);
    sprintf(files->librtslog, "%s%s", genbase, LIBRTSLOG);
    sprintf(files->libgenrts, "%s%s", genbase, LIBGENRTS);

    check_file_installed(files->gra2o, "agfl compiler");
    check_file_installed(files->lexgen, "lexgen command");
    check_file_installed(files->machdep, "include file");
    check_file_installed(files->librts, "run-time library");

// Not yet build by automake/conf
//  check_file_installed(files->librtslog, "run-time library");

    genas = getenv("GENAS");
    if (genas == NULL) {
        genas = GENAS;
    }

    strcpy(files->genas, genas);
    /* check_file_installed(files->genas, "assembler command"); */

    gencc = getenv("GENCC");
    if (gencc == NULL) {
        gencc = GENCC;
    }

    strcpy(files->gencc, gencc);

    /* check_file_installed(files->gencc, "C compiler"); */
}

int
file_ext_exists(const char* base, const char* ext)
{
    char path[PATH_MAX];

    sprintf(path, "%s%s", base, ext);

    return file_exists(path);
}

gboolean
have_lexicon(const char* base)
{
    return file_ext_exists(base, ".lif");
}

time_t
get_file_time(const char* path)
/*
// Return creation or update time of file with path,
// or 0 if file does not exist.
*/
{
    struct stat sbuf;

    if (stat(path, &sbuf) < 0) {
        return 0;
    } else {
        return sbuf.st_mtime;
    }
}

int
file_is_newer(const char* path1, const char* path2)
/*
// Check whether file with path1 is newer than file with path2.
// Return false if file with path1 does not exist.
*/
{
    time_t time1 = get_file_time(path1);
    time_t time2 = get_file_time(path2);

    return time1 > time2;
}

void
check_file_exists(const char* path)
{
    if (!file_exists(path)) {
        output_error("fatal: cannot locate input file `%s'", path);
        exit(1);
    }
}

void
delete_file(const char* path, int verbose)
{
    if (verbose) {
        output_log("%s %s", RM, path);
    }

    remove(path);
}

int
execute(const char* cmd, int verbose)
{
    int	error;

    if (verbose) {
        output_log(cmd);
    }
    error = system(cmd);

    return error;
}

void
make_lexicon(const char* base, const Options* opt, const GenFiles* files)
{
    char blf_file[PATH_MAX];
    char cmd[PATH_MAX];
    int error;

    sprintf(blf_file, "%s%s", base, ".blf");

    sprintf(cmd, "%s%s%s %s", files->lexgen,
            opt->verbose ? " -v" : "",
            opt->force ? " -f" : "",
            base);
    error = execute(cmd, opt->verbose);

    if (error) {
        delete_file(blf_file, opt->verbose);
        exit(1);
    }
}

void
make_parser(const char* base, const Options* opt, const GenFiles* files)
{
    char lex_file[PATH_MAX];
    char lif_file[PATH_MAX];
    char gra_file[PATH_MAX];
    char obj_file[PATH_MAX];
    char exe_file[PATH_MAX];
    char cmd[PATH_MAX];
    int error;
    int gra2o;
    int link;
    char* linklibrary;

    sprintf(lif_file, "%s%s", base, ".lif");
    sprintf(lex_file, "%s%s", base, ".lex");
    sprintf(gra_file, "%s%s", base, ".gra");
    sprintf(obj_file, "%s%s", base, ".o");
    sprintf(exe_file, "%s%s", base, OUT_EXE);

    check_file_exists(gra_file);
    gra2o = opt->force
            || !file_exists(exe_file)
            || file_is_newer(gra_file, exe_file)
            || file_is_newer(lex_file, exe_file)
            || file_is_newer(files->gra2o, exe_file);
    link = gra2o
           || file_is_newer(files->librts, exe_file)
           || file_is_newer(files->librtslog, exe_file)
           || file_is_newer(files->libgenrts, exe_file);

    if (link && !gra2o && !file_exists(obj_file)) {
        gra2o = 1;
    }

    if (gra2o) {
        sprintf(cmd, "%s %s -I%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
                files->gra2o, gra_file, files->include,
                opt->underscore ?          " -_" : "",
                opt->additional_analysis ? " -a" : "",
                opt->counters ?            " -c" : "",
                opt->debug ?               " -g" : "",
                opt->generative_grammar ?  " -G" : "",
                opt->write_log ?           " -l" : "",
                opt->module_names ?        " -m" : "",
                opt->profile ?             " -p" : "",
                opt->show_time ?           " -t" : "",
                opt->verbose ?             " -v" : "",
                opt->suppress_warnings ?   " -w" : "",
                opt->suppress_analysis ?   " -A" : "",
                opt->directors ?           " -D" : "",
                opt->left_factorize ?      " -F" : "",
                opt->positive_memo ?       " -P" : "",
                opt->negative_memo ?       " -N" : "",
                opt->trace ?               " -T" : "");

        error = execute(cmd, opt->verbose);

        if (error) {
            delete_file(obj_file, opt->verbose);
            exit(1);
        }
    }

    if (opt->counters) {
        assert(!"No counters support yet!");
        linklibrary = NULL;
    } else if (opt->generative_grammar) {
        linklibrary = LINKGENRTS;
    } else if (opt->positive_memo) {
        linklibrary = LINKPMRTS;
/*  } else if (opt->trace) {
        linklibrary = LINKDEBUGRTS;
 * stn 14/08/2003
 */
    } else {
        linklibrary = LINKNMRTS;
    }
    if (link) {
        check_file_exists(obj_file);
        sprintf(cmd, "%s %s -o %s%s%s -L%s %s%s %s -llexicon %s",
                files->gencc, obj_file, exe_file,
                opt->debug ?   " -g" : "",
                opt->strip ?   " -s" : "",
                files->lib,
                linklibrary,
                opt->electric_fence ? " -lefence" : "",
                GENGLIBFLAG,
#if defined(HAVE_LIBREADLINE)
                "-lreadline");
#else
                "");
#endif

        error = execute(cmd, opt->verbose);

        if (error) {
            output_error("fatal: link error, help!");
            exit(1);
        }
    } else {
        output_error("`%s' is up to date.", base);
    }
}

int
main(int argc, const char* argv[])
{
    GenFiles files;
    Options options;
    char base_name[PATH_MAX];

    init_options(&options);
    process_options(argc, argv, &options);
    check_installation(&files);

    get_base_name(base_name, options.grammar_file, ".gra");
    make_parser(base_name, &options, &files);

    /* lexgen decides if it needs to make a lexicon..: */
    make_lexicon(base_name, &options, &files);

    return 0;
}

