/*
 * Runtime system I/O routines.
 *
 * Copyright 2000 KUN.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: rtsio.c,v 1.26 2003/08/12 19:30:09 pspiertz Exp $ */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>

#include <assert.h>
#include <glib.h>

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#else /* HAVE_MALLOC_H */
#include <stdlib.h>
#endif /* HAVE_MALLOC_H */

#define MAX_STR_LEN 8192    /* for output_formatted() */

#ifndef WIN32
#include <termios.h>
#endif /* WIN32 */

#if !defined(MSDOS) && !defined(WIN32)
#include <sys/stat.h>
#include <fcntl.h>
#endif /* !MSDOS && !WIN32 */

#ifdef HAVE_LIBREADLINE
#ifdef READLINE_LIBRARY
#  include "readline.h"
#  include "history.h"
#else
#  include <readline/readline.h>
#  include <readline/history.h>
#endif
#endif


#include "rtsio.h"
#include "rtsopt.h"
#include "rtsutil.h"
#include "rtscode.h"
#include "rtsbuffer.h"

/*
** Private
*/

static Buffer*	output_buffer = NULL;
static FILE*	input_file = NULL;
static FILE*	profile_file = NULL;
static FILE*	tracelog_file = NULL;
static FILE*	output_file = NULL;
static char*	input_text = NULL;
static int	interactive = 0;


#ifdef PROFILING
static FILE*	profFile;		/* profile counters */
#endif


#ifdef DEBUG_IO
#define DB(x) x
#else
#define DB(x)
#endif


/*
**------------------------------------------------------------------------------
** Priority list routines:
**------------------------------------------------------------------------------
*/

typedef struct _PrioList
{
    long priority;
    unsigned length;
    StateIndicator state_node;
    GString* data;
    struct _PrioList* next;
} PrioList;

static long priolist_max_penalty;
static unsigned priolist_nr_parses;
static PrioList* ParseResults;
static PrioList* CurrentParse;

PrioList*
priolist_new()
{
    PrioList* pl = GetMem(sizeof(PrioList), "priolist_new");
    pl->next = NULL;

    pl->data = g_string_new("");

    return pl;
}

void
priolist_destroy(PrioList* l)
{
    if(l) {
        priolist_destroy(l->next);

        g_string_free(l->data, TRUE);

        FreeMem(l, "priolist_destroy");
    }
}

PrioList*
priolist_add_real(PrioList* l, PrioList* el)
{
    if(!l) {
        return el;
    }

    if ((l->length == el->length) && (l->priority < el->priority)) {
        l->next = priolist_add_real(l->next, el);

        return l;
    }

    el->next = l;
    return el;
}

void
priolist_add(PrioList* el)
{
    if(!ParseResults) {
        ParseResults = el;
    } else if (ParseResults->length < el->length) {
        priolist_destroy(ParseResults);
        ParseResults = el;
    } else if (ParseResults->length == el->length) {
        ParseResults = priolist_add_real(ParseResults, el);
    }
}

void
parse_results_start_new()
{
    CurrentParse = priolist_new();
    DB(printf("new parse started...\n");)
}

void
current_parse_set_prio_and_add(long prio, StateIndicator pos, unsigned length)
{
    CurrentParse = priolist_new();
    CurrentParse->priority = prio;
    CurrentParse->length = length;
    CurrentParse->state_node = pos;
    priolist_add(CurrentParse);
    priolist_nr_parses++;

    if((priolist_max_penalty == LONG_MAX)
     ||
     (priolist_max_penalty >= prio)) {
        if(priolist_nr_parses >= max_parses) {
            priolist_max_penalty = prio - 1; // only add better parsings
        } else {
            priolist_max_penalty = prio;
        }
    }

    DB(printf("Parsing with penalty %ld and length %u added.\n",
              prio, length);)
    DB(printf("\tpriolist_max_penalty = %ld\n", priolist_max_penalty);)
}

void
parse_results_init()
{
  priolist_nr_parses = 0;
  priolist_max_penalty = LONG_MAX;
  ParseResults = NULL;
  CurrentParse = NULL;
}

unsigned
parse_results_get_nr_parses()
{
  return priolist_nr_parses;
}

long
parse_results_get_max_penalty()
{
  return priolist_max_penalty;
}

void
parse_results_add(PrioList* el)
{
  priolist_add(el);
}

void
parse_results_destroy()
{
  priolist_destroy(ParseResults);
}

void
parse_results_dump()
{
    PrioList* l = ParseResults;
    unsigned nr = 0;

    DB(printf("Begin dump parse results:\n");)
    while(l && (nr < max_parses)) {
        printf("%s", l->data->str);

        l = l->next;

        nr++;
    }

    DB(printf("End dump parse results:\n"));
    fflush(stdout);
}

StateIndicator
get_statenode_from_first_parse()
{
    return ParseResults ? ParseResults->state_node : NULL;
}

unsigned
get_length_from_first_parse()
{
    return  ParseResults ? ParseResults->length : 0;
}

void
current_parse_add_int(const int i)
{
    g_string_sprintfa(CurrentParse->data, "%d", i);
}

void
current_parse_add_char(const char c)
{
    CurrentParse->data = g_string_append_c(CurrentParse->data, c);
}

void
current_parse_add_string(const char* s)
{
    CurrentParse->data = g_string_append(CurrentParse->data, s);
}

void
current_parse_add_space(int i)
{
    char* spaces = g_strnfill(i, ' ');
    current_parse_add_string(spaces);
    g_free(spaces);
}

void current_parse_printf(const char* fmt, ...)
{
  char* addition;

  va_list argp;
  va_start(argp, fmt);

  addition = g_strdup_vprintf(fmt, argp);
  current_parse_add_string(addition);

  va_end(argp);
  g_free(addition);
}

/*
** File IO
*/

static void
open_input_file()
{
  const char* fname = get_input_file_name();
  if (fname == NULL)
  {
    input_file = stdin;
  }
  else
  {
    input_file = fopen(fname, "r");
    if (input_file == NULL)
      rtsFatal("cannot open input file '%s'\n", fname);
  }
}

static void
open_profile_file()
{
   char* fname;
   if (profile_option) {
     fname = strdup(get_parser_name());
     /* strncpy(fname, get_parser_name(), 251); */
     strncat(fname, ".prd",4);
     profile_file = fopen(fname, "w");
     if (profile_file == NULL)
        rtsFatal("cannot open file '%s' to store profile results\n", fname);
   }
}

void
profile_printf(char* message, ...)
{
  va_list argp;
  va_start(argp, message);
  /* Hope this is still compatible with PA-RISC (PARIX define) */
  if (profile_option) {
    vfprintf(profile_file, message, argp);
    /* fflush(stderr); */
  }
  va_end(argp);
}

static void
close_profile_file()
{
  if (profile_option) {
    fclose(profile_file);
  }
}

static void
open_tracelog_file()
{
   char* fname;
   if (trace_option) {
     fname = strdup(get_parser_name());
     /* strncpy(fname, get_parser_name(), 251); */
     strncat(fname, ".trc",4);
     tracelog_file = fopen(fname, "w");
     if (tracelog_file == NULL)
        rtsFatal("cannot open file '%s' to store the parser's trace log\n", fname);
   }
}

void
tracelog_printf(char* message, ...)
{
  va_list argp;
  va_start(argp, message);
  /* Hope this is still compatible with PA-RISC (PARIX define) */
  if (trace_option) {
    vfprintf(tracelog_file, message, argp);
    /* fflush(stderr); */
  }
  va_end(argp);
}

static void
close_tracelog_file()
{
  if (trace_option) {
    fclose(tracelog_file);
  }
}

static void
close_input_file()
{
  fclose(input_file);
}

static void
open_output_file()
{
  const char* fname = get_output_file_name();
  if (fname == NULL)
  {
    output_file = stdout;
  }
  else
  {
    output_file = fopen(fname, "w");
    if (output_file == NULL)
      rtsFatal("cannot open output file '%s'\n", fname);
  }
}

static void
close_output_file()
{
  fclose(output_file);
}

/*
** Prelude and Postlude
*/

void InitIO()
{
  open_input_file();
  open_profile_file();
  open_tracelog_file();
  open_output_file();
  interactive = isatty(fileno(input_file)) && isatty(fileno(stderr)); 
  /* stn FIXME this is never given free */
  input_text = (char*)GetMem(MAXINPUT, "rtsio");
  input_text[0] = '\0';

  output_buffer = NewBuffer();
}

void EndIO()
{
  FreeMem(input_text, "rtsio");
  close_input_file();
  close_profile_file();
  close_tracelog_file();
  close_output_file();

  FreeBuffer(output_buffer);
}

/*
** Console output
*/

void rtsMessage(char *fmt, ...)
{
  va_list argp;
#ifdef PARIX
  char fName[] = "rtsMessage";
  char buf[1024];
#endif

  va_start(argp, fmt);
#ifdef PARIX
  vsprintf(buf, fmt, argp);
  LogError(EC_MESS, fName, "%s", buf);
#else
  vfprintf(stderr, fmt, argp);
  fflush(stderr); 
#endif
  va_end(argp);
}

/*
** Error handling
*/

void rtsBug(char *fName, char *fmt, ...)
{
  va_list argp;
#ifdef PARIX
  char buf[1024];
#endif

  va_start(argp, fmt);
#ifdef PARIX
  vsprintf(buf, fmt, argp);
  LogError(EC_ERROR, fName,
           "BUG: %s\n** Please contact agfl@cs.kun.nl\n", buf);
#else
  fprintf(stderr, "BUG (%s): ", fName);
  vfprintf(stderr, fmt, argp);
  fprintf(stderr, "\n** Please contact agfl@cs.kun.nl\n");
  fflush(stderr);
#endif
  va_end(argp);
  rtsExit(1);
}

void rtsAbort(char *fName, char *fmt, ...)
{
  va_list argp;
#ifdef PARIX
  char buf[1024];
#endif

  va_start(argp, fmt);
#ifdef PARIX
  vsprintf(buf, fmt, argp);
  LogError(EC_ERROR, fName, "ABORT: %s\n", buf);
#else
  fprintf(stderr, "ABORT (%s): ", fName);
  vfprintf(stderr, fmt, argp);
  fputc('\n', stderr);
  fflush(stderr);
#endif
  va_end(argp);
  assert(0); /* die here for GDB (Erik) */
  rtsExit(1);
}

void rtsFatal(char *fmt, ...)
{
  va_list argp;
#ifdef PARIX
  char fName[] = "rtsFatal";
  char buf[1024];
#endif

  va_start(argp, fmt);
#ifdef PARIX
  vsprintf(buf, fmt, argp);
  LogError(EC_ERROR, fName, "Fatal: %s\n", buf);
#else
  fprintf(stderr, "Fatal: ");
  vfprintf(stderr, fmt, argp);
  fputc('\n', stderr);
  fflush(stderr);
#endif
  va_end(argp);
  rtsExit(1);
}

void rtsError(char *fmt, ...)
{
  va_list argp;
#ifdef PARIX
  char fName[] = "rtsError";
  char buf[1024];
#endif

  va_start(argp, fmt);
#ifdef PARIX
  vsprintf(buf, fmt,argp);
  LogError(EC_ERROR, fName, "Error: %s\n", buf);
#else
  fprintf(stderr, "Error: ");
  vfprintf(stderr, fmt, argp);
  fputc('\n', stderr);
  fflush(stderr);
#endif
  va_end(argp);
}

void rtsWarning(char *fmt, ...)
{
  va_list argp;
#ifdef PARIX
  char fName[] = "rtsWarning";
  char buf[1024];
#endif

  va_start(argp, fmt);
#ifdef PARIX
  vsprintf(buf, fmt,argp);
  LogError(EC_ERROR, fName, "Warning: %s\n", buf);
#else
  fprintf(stderr, "Warning: ");
  vfprintf(stderr, fmt, argp);
  fputc('\n', stderr);
  fflush(stderr);
#endif
  va_end(argp);
}

void rtsExit(int error)
{
  exit(error);
}

/*
** Input
*/

const char*
get_input_text()
{
  return input_text;
}

int
read_input_line()
{
  static int eof = 0;
  unsigned input_len;

  if (eof)
    return 0;
  /* Input from terminal? Prompt user */
  /* Get input line */
#ifdef HAVE_LIBREADLINE
  if(interactive) {
    input_text = readline (">> ");
  }
  else {
    input_text = readline ("");
  }
  if (!input_text) 		/* test for EOF */
#else
  if(interactive)
     rtsMessage(">> ");
 
  if (fgets(input_text, MAXINPUT, input_file) == NULL)
#endif
  {
    eof = 1;
    return 0;
  };
 
#ifdef HAVE_LIBREADLINE
  /* If there is anything on the line, remember it */
  if (*input_text) {
    add_history (input_text);
  }
#endif

  /* Chop newline */
  input_len = strlen(input_text);
  if(input_text[input_len - 1] == '\n')
    input_text[--input_len] = '\0';

  /* Interactive users exit on pressing just enter */
  if (interactive && (input_len == 0))
    return 0;
  return 1;
}

int
read_input_block(char lf_subst)
{
    static int eof = 0;
    unsigned input_len;
    register int inchar;
    GString *tmp = g_string_sized_new(256);
    register gboolean seen_newline = FALSE;
    gboolean seen_end_paragraph = FALSE;

    if (eof) {
        return 0;
    }

    /* Input from terminal? Prompt user */
    if(interactive) {
        rtsMessage(">> ");
    }

    /* Get input block */
    while (!seen_end_paragraph && !feof(input_file)) {
        inchar = fgetc(input_file);
        if (inchar == EOF) {
            break;
        }

        if ((char) inchar == '\n') {
            if (seen_newline) {
                seen_end_paragraph = TRUE;
            } else {
                seen_newline = TRUE;
//                tmp = g_string_append_c(tmp, (char) inchar);
                tmp = g_string_append_c(tmp, lf_subst); /* usually ' ' */
            }
        } else if (io_sync_option && ((char) inchar == '\001')) {
            seen_end_paragraph = TRUE;
        } else if (strchr(invisible_chars, inchar)) {
            /* whitespace at beginning of new line. this may still end the
             * paragraph a \n is seen before a non-whitespace-char.
             */
            tmp = g_string_append_c(tmp, (char) inchar);
        } else {
            seen_newline = FALSE;
            tmp = g_string_append_c(tmp, (char) inchar);
        }
    };

    input_text = tmp->str;
    DB(fprintf(stderr,"input_text = \"%s\"\n", input_text);)
    g_string_free(tmp, FALSE);
    DB(fprintf(stderr,"after free: input_text = \"%s\"\n", input_text);)

    if (feof(input_file)) {
        eof = 1;
    }

    /* Chop newline */
    input_len = strlen(input_text);
    DB(fprintf(stderr, "input_len = %Zd\n", input_len);)
    if(input_text[input_len - 1] == '\n') {
        input_text[--input_len] = '\0';
    }

//    if (interactive && (input_len == 0)) {
//        return 0;
//    }
    return 1;
}

void maybe_output_sync()
{
    if (io_sync_option) {
        printf("\001");
        fflush(stdout);
    }
}

#ifdef GEN_RTS
typedef struct _GenOutputEl {
    const char* token;
    struct _GenOutputEl* next;
} GenOutputEl;

static GenOutputEl*
gen_output_push(GenOutputEl* l, const char* t)
{
    GenOutputEl* n = g_malloc(sizeof(GenOutputEl));

    n->next = l;
    n->token = t;

    return n;
}

static GenOutputEl*
gen_output_pop(GenOutputEl* l)
{
    GenOutputEl* res;

    assert(l);

    res = l->next;
    g_free(l);

    return res;
}

static void
gen_output_printf(GenOutputEl* l)
{
    if (!l) return;

    gen_output_printf(l->next);
    printf(" %s", l->token);
}

static GenOutputEl* generated_output;
static int gen_line_nr;

void
gen_output_init(unsigned line_nr)
{
    generated_output = NULL;
    gen_line_nr = line_nr;
}

void
gen_output_add_token(const char* txt)
{
    switch (*txt) {
        case '\1':
        case '\2':
        case '\3':
        case '\4':
            txt++;
            break;
        default:
    }

    generated_output = gen_output_push(generated_output, txt);
}

void
gen_output_remove_token()
{
    generated_output = gen_output_pop(generated_output);
}

void
gen_output_show()
{
    printf("%3d:", gen_line_nr);
    gen_output_printf(generated_output);
    printf("\n");
}

void
gen_output_free()
{
    while (generated_output) {
        generated_output = gen_output_pop(generated_output);
    }
}

#endif /* GEN_RTS */
