/*
 * Positive memoization routines.
 * 
 * Copyright 2000 KUN.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: posmemo.c,v 1.11 2003/07/22 07:38:17 paulj Exp $ */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#else /* HAVE_MALLOC_H */
#include <stdlib.h>
#endif /* HAVE_MALLOC_H */

#include <string.h>

#include "rtscode.h"
#include "rtsutil.h"
#include "rtslex.h"
#include "posmemo.h"

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_getmem  allocate memory for positive memoization
|*FDEF RES void*           pointer to memory allocated
|*FDEF ARG size_t          amount of memory to be allocated
|*FDEF USE malloc()
|*FDEF REM no check of success of malloc()
|*FDEF REM why static + inline?           
\*----------------------------------------------------------------------------------------------------*/

static inline void* posmemo_getmem(size_t s)
{
    return malloc(s);
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_freemem  free memory previouslt allocated for positive memoization
|*FDEF RES void
|*FDEF ARG void*            pointer to memory that was previously allocated
|*FDEF USE free()
|*FDEF REM no check of success of free()
|*FDEF REM why static + inline?           
\*----------------------------------------------------------------------------------------------------*/

static inline void posmemo_freemem(void* ptr)
{
    free(ptr);
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_init     initialize memory management for positive memoization
|*FDEF RES void             pointer to memory allocated
|*FDEF REM not implemented yet.
|*FDEF REM intention not clear.
\*----------------------------------------------------------------------------------------------------*/

void posmemo_init()
{
    /* init memory management, not implemented yet */
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_done     clean up memory management for positive memoization
|*FDEF RES void             pointer to memory allocated
|*FDEF REM not implemented yet. 
|*FDEF REM intention not clear.
\*----------------------------------------------------------------------------------------------------*/

void posmemo_done()
{
    /* stop memory management, not implemented yet */
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_fetch    locate and return previously generated psoitive memo structure
|*FDEF RES PosMemo*         pointer to memory allocated
|*FDEF ARG StateIndicator   location in input string being parsed
|*FDEF ARG unsigned         index of nonterminal that is looked for
|*FDEF USE assert()
|*FDEF EXC fails if 1st argument is zero
|*FDEF REM why check on StateIndicator, and why not check on validity of onoterminal index?
|*FDEF REM requires that all arrays of positive memoization structures are allocated immediately. Check!
\*----------------------------------------------------------------------------------------------------*/

static inline PosMemo* posmemo_fetch(StateIndicator i, unsigned n)
{
    assert(i);
    return &(i->pos_memos[n]);
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_init_table_entry  initialize value of new positive memo structure: let it point to itself
|*FDEF RES void
|*FDEF ARG PosMemo*                  pointer to positive memo stucture to be initialized
|*FDEF REM no check on validity of argument
\*----------------------------------------------------------------------------------------------------*/

void posmemo_init_table_entry(PosMemo* entry)
{
    (void*)*entry = (void*)entry;
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_is_unknown  check if pointer to positive memo structure has "unknown" status
|*FDEF RES int                 boolean test result: pointer is not null && points to itself
|*FDEF ARG StateIndicator      location in input string being parsed
|*FDEF ARG unsigned            index of nonterminal that is looked for
|*FDEF USE posmemo_fetch()
\*----------------------------------------------------------------------------------------------------*/

int posmemo_is_unknown(StateIndicator input_state, unsigned nont_nr)
{
    PosMemo* x = posmemo_fetch(input_state, nont_nr);
    return((x != NULL) && ((void*)x == (void*)(*x)));
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_is_known  check if pointer to positive memo structure has "known" status
|*FDEF RES int               boolean test result: pointer is NULL || does not point to itself
|*FDEF ARG StateIndicator    location in input string being parsed
|*FDEF ARG unsigned          index of nonterminal that is looked for
|*FDEF USE posmemo_fetch()
\*----------------------------------------------------------------------------------------------------*/

int posmemo_is_known(StateIndicator input_state, unsigned nont_nr)
{
    PosMemo* x = posmemo_fetch(input_state, nont_nr);
    return((*x == NULL) || ((void*)x != (void*)(*x)));
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_is_blocked  check if pointer to positive memo structure has "blocked" status
|*FDEF RES int                 boolean test result: pointer is NULL
|*FDEF ARG StateIndicator      location in input string being parsed
|*FDEF ARG unsigned            index of nonterminal that is looked for
|*FDEF USE posmemo_fetch()
\*----------------------------------------------------------------------------------------------------*/

int posmemo_is_blocked(StateIndicator input_state, unsigned nont_nr)
{
    PosMemo* x = posmemo_fetch(input_state, nont_nr);
    return(*x == NULL);
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_is_blocked_for_penlevel  check if pointer to positive memo structure has "known" status
|*FDEF RES int                              boolean test result: pointer is NULL || posmemo penalty is too high
|*FDEF ARG StateIndicator                   location in input string being parsed
|*FDEF ARG unsigned                         index of nonterminal that is looked for
|*FDEF ARG long                             maximum allowable penalty level
|*FDEF EXC                                  abort (!) if positive memo has "unknown" status
|*FDEF USE posmemo_fetch()
\*----------------------------------------------------------------------------------------------------*/

int posmemo_is_blocked_for_penlevel(StateIndicator input_state, unsigned nont_nr, long penlevel)
{
    PosMemo* x = posmemo_fetch(input_state, nont_nr);

    assert((void*)x != (void*)(*x)); /* if unknown, explode */

    if (*x == NULL) {
        return 1;
    }

    return (*x)->penalty > penlevel;
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_is_unblocked  check if pointer to positive memo structure has "unblocked" status
|*FDEF RES int                   boolean test result: pointer is not NULL && does not point to itself
|*FDEF ARG StateIndicator        location in input string being parsed
|*FDEF ARG unsigned              index of nonterminal that is looked for
|*FDEF USE posmemo_fetch()
|*FDEF REM                       !! does not only fail if blocked, but also if unknown !!
\*----------------------------------------------------------------------------------------------------*/

int posmemo_is_unblocked(StateIndicator input_state, unsigned nont_nr)
{
    PosMemo* x = posmemo_fetch(input_state, nont_nr);
    return((*x != NULL) && ((void*)x != (void*)(*x)));
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_set_unknown  set positive memo structure to "unknown" status (i.e., point to itself)
|*FDEF RES void
|*FDEF ARG StateIndicator       location in input string being parsed
|*FDEF ARG unsigned             index of nonterminal that is looked for
|*FDEF USE posmemo_fetch()
\*----------------------------------------------------------------------------------------------------*/

void posmemo_set_unknown(StateIndicator input_state, unsigned nont_nr)
{
    PosMemo* x = posmemo_fetch(input_state, nont_nr);
    (void*)*x = (void*)x;
}

/*----------------------------------------------------------------------------------------------------*\
|*FDEF NAM posmemo_set_blocked  set positive memo structure to "blocked" status (i.e., point to NULL)
|*FDEF RES void
|*FDEF ARG StateIndicator       location in input string being parsed
|*FDEF ARG unsigned             index of nonterminal that is looked for
|*FDEF USE posmemo_fetch()
|*FDEF EXC crashes if posmemo structure does not have "unknown" status.
\*----------------------------------------------------------------------------------------------------*/

void posmemo_set_blocked(StateIndicator input_state, unsigned nont_nr)
{
    PosMemo* x = posmemo_fetch(input_state, nont_nr);
    assert((void*)x == (void*)(*x));
    *x = NULL; 
}

int posmemo_next_equal_formals(PosMemo p1)
{
   return p1->next_equal_formals;
}

#if 0
static int equal = 0;
#endif

void posmemo_set_next_son(PosMemo p,int nr) 
{
    PosMemo *pms = (PosMemo *)&(p->variables[p->nr_variables+nr]);
    *pms = (*pms)->next;
}

void posmemo_reset_son(PosMemo p,int nr) 
{
    PosMemo *pms = (PosMemo *)&(p->variables[p->nr_variables+nr]);
    PosMemo curr = *pms;
    while (curr->prev!=0) {
        curr = curr->prev;
    };
    *pms = curr;
}


static void posmemo_add_sorted(PosMemo* pms, PosMemo new_prod)
{
    long nr_parses = 0;
    int inserted = 0;
    PosMemo last = NULL;
    PosMemo curr = *pms;


    for(;curr!=NULL;last = curr,curr = curr->next) {
        if (curr->penalty > new_prod->penalty) {
            /* no equal entries found at this pen. level */
            if (!inserted && (nr_parses<max_parses)) {
                new_prod->prev = NULL;
                new_prod->next = curr;
                new_prod->next_equal_formals = 0;
                if (last==NULL) 
                    *pms = new_prod;
                else 
                    last->next = new_prod;
                last = new_prod;
                inserted = 1;
                nr_parses++;
            };
        };

        /* check for equal entries */
        if ((curr->nr_formals == new_prod->nr_formals)
            && (curr->next_state == new_prod->next_state)
            && (memcmp(curr->variables, new_prod->variables,
                       sizeof(VALUE) * (new_prod->nr_formals))==0)) {
            /* check if we need to delete this entry */
            if (nr_parses==max_parses) {
                /* delete this entry from the list */
                if (last==NULL) {
                    *pms = curr->next;
                }
                else
                    last->next = curr->next;
                if (curr->variables) {
                    posmemo_freemem(curr->variables);
                }
                posmemo_freemem(curr);

                /* check if we need to delete the entry we wanted to add */
                if (!inserted) {
                    /* delete the new unused posmemo entry */
                    if (new_prod->variables) {
                        posmemo_freemem(new_prod->variables);
                    }
                    posmemo_freemem(new_prod);
                };

                /* we know we only need to delete at most 1 entry */
                return;
            };

            /* same entries */
            nr_parses++;

            /* now use next_equal_formals to speed up search */
            while ((nr_parses<max_parses) && curr->next_equal_formals) {
                nr_parses++;
                last = curr;
                curr = curr->next;
            };

            /* check we already have max_parses partial solutions */
            if (nr_parses==max_parses) {
                /* delete any next parse if it is equal */
                if (curr->next_equal_formals) {
                    /* delete it */
                    last->next = curr->next;
                    last->next_equal_formals = 0;

                    if (curr->variables) {
                        posmemo_freemem(curr->variables);
                    }
                    posmemo_freemem(curr);

                    /* check if we need to delete the entry we wanted to 
                     * add */
                    if (!inserted) {
                        /* delete the new unused posmemo entry */
                        if (new_prod->variables) {
                            posmemo_freemem(new_prod->variables);
                        }
                        posmemo_freemem(new_prod);
                    };

                    /* we know we only need to delete at most 1 entry */
                    return;
                };

                /* max_parses found and !inserted means we are done */
                if (!inserted) {
                    /* delete the new unused posmemo entry */
                    if (new_prod->variables) {
                        posmemo_freemem(new_prod->variables);
                    }
                    posmemo_freemem(new_prod);

                    /* done */
                    return;
                };
            };

            /* check for identical penalty levels */
            if (curr->penalty == new_prod->penalty) {

                /* add if nr_parses < max_parses */
                if (!inserted && nr_parses<max_parses) {
                    /* insert */
                    new_prod->next = curr->next;
                    new_prod->prev = curr;
                    new_prod->next_equal_formals = 0;
                    curr->next = new_prod;
                    curr->next_equal_formals = 1;
                    inserted = 1;
                    nr_parses++;

                    /* skip that entry for the next loop */
                    last = curr;
                    curr = curr->next;
#if 0
                    equal = 1;
#endif
                };

            }
        } else {
            /* now use next_equal_formals to speed up search */
            while ((nr_parses<max_parses) && curr->next_equal_formals) {
                nr_parses++;
                last = curr;
                curr = curr->next;
            };
        };
    }

    /* check if we still need to insert */
    if (!inserted) {
        if (nr_parses<max_parses) {
            /* insert */
            if (last!=NULL)
                last->next = new_prod;
            else 
                *pms = new_prod;

            new_prod->prev = NULL;
            new_prod->next = NULL;
            new_prod->next_equal_formals = 0;
        }
        else {
            /* delete the new unused posmemo entry */
            if (new_prod->variables) {
                posmemo_freemem(new_prod->variables);
            }
            posmemo_freemem(new_prod);
        }
    };
}

int pointsIntoStack(void *p);

void posmemo_add_production(StateIndicator input_state, unsigned nont_nr,
                            long penalty, unsigned nr_formals,
			    unsigned nr_locals, unsigned nr_sons,
                            VALUE* variables, StateIndicator target_state,
			    void *pass2)
{
    unsigned i;
    size_t variables_block_size = (nr_formals+nr_locals+nr_sons) * sizeof(VALUE);
    PosMemo* x = posmemo_fetch(input_state, nont_nr);
    PMPROD* new_memo = posmemo_getmem(sizeof(PMPROD));

    new_memo->nont_nr = nont_nr;
    new_memo->nr_formals = nr_formals;
    new_memo->nr_variables = nr_formals+nr_locals;
    new_memo->nr_sons = nr_sons;
    new_memo->pass2 = pass2;
    if (variables_block_size>0) {
        new_memo->variables = (VALUE*)posmemo_getmem(variables_block_size);
    } else {
        new_memo->variables = NULL;
    }

    /* copy formals + locals */
    /* NOTE: variables 1 is at -1, 2 at -2, etc! */
    for(i = 0;i < (nr_formals+nr_locals);i++) {
        new_memo->variables[i] = variables[-i];
    };
    /* copy sons by copying PMPROD entries from frame */
    for(i = 0;i < nr_sons;i++) {
	VALUE son = variables[-(nr_formals+nr_locals+i)];
	if (pointsIntoStack((VALUE *)(son.text_par)))
	    /* copy the PMPROD entry */
	    new_memo->variables[nr_formals+nr_locals+i] = 
		((VALUE *)(son.text_par))[-7];
	else
	    /* copy the transition entry */
	    new_memo->variables[nr_formals+nr_locals+i] = son;	
    }

    new_memo->penalty = penalty;
    new_memo->failcont = NULL;
    new_memo->next_state = target_state;

#if 0
    equal = 0;
#endif
    posmemo_add_sorted(x, new_memo);
#if 0
    if (equal)
        fprintf(stderr,"EQ\n");
    else
        fprintf(stderr,"NE\n");
#endif
}

void
posmemo_free_vec(PosMemo* entry)
{
    if ((*entry != NULL) && ((void*)entry != (void*)(*entry))) {
        PMPROD* prod = *entry;
        while (prod) {
            PMPROD* tail = prod->next;
            if (prod->variables) {
                posmemo_freemem(prod->variables);
            }
            posmemo_freemem(prod);
            prod = tail;
        }
    }
}

int posmemo_count_prod(StateIndicator input_state, unsigned nont_nr)
{
    int count = 0;
    PMPROD* x = *(posmemo_fetch(input_state, nont_nr));
    while (x) {
        count++;
        x = x->next;
    }

    return count;
}

void posmemo_set_failcont(PosMemo prod, void* pc)
{
    assert(prod);
    prod->failcont = pc;
}

void* posmemo_get_failcont(PosMemo prod)
{
    assert(prod);
    return prod->failcont;
}

void* posmemo_get_pass2(PosMemo prod)
{
    assert(prod);
    return prod->pass2;
}

PosMemo posmemo_get_prod_ptr(StateIndicator input_state, unsigned nont_nr)
{
    return *(posmemo_fetch(input_state, nont_nr));
}

void* posmemo_get_formal_ptr(PosMemo state)
{ 
    assert(state);
    return state->variables;
}

VALUE posmemo_get_variable(PosMemo state,int nr)
{ 
    assert(state);
    return state->variables[nr];
}

VALUE posmemo_get_local(PosMemo state,int nr)
{ 
    assert(state);
    return state->variables[state->nr_formals+nr];
}

PosMemo posmemo_get_son(PosMemo state,int nr)
{ 
    assert(state);
    return (PosMemo)(state->variables[state->nr_variables+nr].text_par);
}

long posmemo_get_penalty(PosMemo state)
{
    assert(state);
    return state->penalty;
}

PosMemo posmemo_get_next_prod(PosMemo curr)
{
    assert(curr);
    return curr->next;
}

StateIndicator posmemo_get_input_state(PosMemo curr)
{
    assert(curr);
    return curr->next_state;
}

void
posmemo_dump_pmprod(int node_nr, PMPROD* pmprod)
{
    int i;

    printf("%d: ", node_nr);
    printf("nont_nr = %u/p %ld/f %u -> %u (%p) %c ",
        pmprod->nont_nr, pmprod->penalty, pmprod->nr_formals, pmprod->next_state->pos, pmprod->failcont, pmprod->next_equal_formals ? 'E': 'N');
    for (i = 0; i < pmprod->nr_sons;++i) {
        PMPROD *son = (PMPROD *)(pmprod->variables[pmprod->nr_variables+i].text_par);
        if (son->next_state<0x100000)
            printf("?? ");
        else
            printf("%u:%u ",son->nont_nr,son->next_state->pos);
    };
    printf("\n");
    for (i = 0; i < pmprod->nr_formals; ++i) {
        printf("\tformal %d set value: %lu\n", i, (pmprod->variables)[i].set_par);
    }
}

void
posmemo_dump_pmprod_list(PMPROD* pmprod)
{
    int i;

    while (pmprod!=NULL) {
        fprintf(stderr,"nont_nr = %u, pen = %ld, nr_formals = %u, next_state = %u, failcont = %p\n",
            pmprod->nont_nr, pmprod->penalty, pmprod->nr_formals, pmprod->next_state->pos, pmprod->failcont);
        for (i = 0; i < pmprod->nr_variables; ++i) {
            fprintf(stderr,"\tformal %d set value: %lu\n", i, (pmprod->variables)[i].set_par);
        };
        pmprod = pmprod->next;
    }
}


void
posmemo_dump_table(Trellis* trellis)
{
    StateNode** state_row = GET_TRELLIS_STATE_ROW(trellis);
    int node_nr;
    int rule_nr;
    gboolean* empty_rule;
    gboolean* empty_node;

    int** overview = (int**) GetMem(sizeof(int*) * trellis->length, "overview[]");

    /* Build the table: */
    for(node_nr = 0; node_nr < trellis->length; node_nr++) {
        StateNode* state = *state_row++;

        if(!state) {
            overview[node_nr] = NULL;
        } else {
            PosMemo* pma = state->pos_memos;

            if(!pma) {
                overview[node_nr] = NULL;
printf("\t\t(posmemo %u skipped, NULL)\n", state->pos);
            } else {
                overview[node_nr] = (int*) GetMem(sizeof(int) * get_nr_syntax_nonterminals(), "overview[][]");

                for(rule_nr = 1; rule_nr < get_nr_syntax_nonterminals(); rule_nr++) {
                    if(posmemo_is_blocked(state, rule_nr)) {
                        overview[node_nr][rule_nr] = -2;
                    } else {
                        if(posmemo_is_unknown(state, rule_nr)) {
                            overview[node_nr][rule_nr] = -1;
// printf("\t\t(posmemo rule %d @ %u skipped, unknown)\n", rule_nr, state->pos);
                        } else {
                            PosMemo plijst = posmemo_get_prod_ptr(state, rule_nr);
                            int nr_ptrs = 0;

                            while(plijst) {
//                                posmemo_dump_pmprod(node_nr, plijst);
                                posmemo_dump_pmprod(node_nr, plijst);
                                nr_ptrs++;
                                plijst = plijst->next;
                            }

                            overview[node_nr][rule_nr] = nr_ptrs;
                        }
                    }
                }
            }
        }
    }

    /* printed table compression */
    empty_rule = (gboolean*) GetMem(sizeof(gboolean) * get_nr_syntax_nonterminals(), "empty_rule");
    for (rule_nr = 1; rule_nr < get_nr_syntax_nonterminals(); rule_nr++) {
        empty_rule[rule_nr] = TRUE;
        node_nr = 0;

        while ((node_nr < trellis->length) && (empty_rule[rule_nr])) {
            if (overview[node_nr]) {
                switch (overview[node_nr][rule_nr]) {
                    case -1:
                        break;
                    case -2:
                        empty_rule[rule_nr] = FALSE;
                        break;
                    default:
                        empty_rule[rule_nr] = !overview[node_nr][rule_nr];
                }
            }

            node_nr++;
        }
    }
    empty_node = (gboolean*) GetMem(sizeof(gboolean) * trellis->length, "empty_node");
    for (node_nr = 0; node_nr < trellis->length; node_nr++) {
        empty_node[node_nr] = TRUE;
        rule_nr = 1;

        while ((rule_nr < get_nr_syntax_nonterminals())
               && (empty_node[node_nr])
               && (overview[node_nr])) {
            switch (overview[node_nr][rule_nr]) {
                case -1:
                    break;
                case -2:
                    empty_node[node_nr] = FALSE;
                    break;
                default:
                    empty_node[node_nr] = !overview[node_nr][rule_nr];
            }

            rule_nr++;
        }
    }

    /* actually show it: */
    /* first the table */
    for (rule_nr = 1; rule_nr < get_nr_syntax_nonterminals(); rule_nr++) {
        if (!empty_rule[rule_nr]) {
            printf("%3d|", rule_nr);

            for (node_nr = 0; node_nr < trellis->length; node_nr++) {
                if (!empty_node[node_nr]) {
                    switch (overview[node_nr][rule_nr]) {
                        case -1:
                            printf("   u");
                            break;
                        case -2:
                            printf("   b");
                            break;
                        default:
                            printf(" %3d", overview[node_nr][rule_nr]);
                    }
                }
            }

            if (nonterm_names[rule_nr]) {
                printf(" | %s\n", nonterm_names[rule_nr]);
            } else {
                printf(" | ?\n");
            }
        }
    }
    /* then a neat line below it */
    printf("---+");
    for (node_nr = 0; node_nr < trellis->length; node_nr++) {
        if (!empty_node[node_nr]) {
            printf("----");
        }
    }
    /* and of course the numbers */
    printf("\n   |");
    for (node_nr = 0; node_nr < trellis->length; node_nr++) {
        if (!empty_node[node_nr]) {
            printf(" %3d", node_nr);
        }
    }
    printf("\n");

    /* free the space: */
    for (node_nr = 0; node_nr < trellis->length; node_nr++) {
        if (overview[node_nr]) {
            FreeMem(overview[node_nr], "overview[][]");
        }
    }
    FreeMem(overview, "overview[]");
    FreeMem(empty_rule, "empty_rule");
}
