/*
 * rts.c - agfl runtime system main routines
 *
 * Copyright 2000 KUN.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: rts.c,v 1.34 2003/08/12 19:30:09 pspiertz Exp $ */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/stat.h>
#include <time.h>

#include "rtsio.h"
#include "rtsopt.h"
#include "rtscode.h"
#include <lexicon.h>
#include "rtslex.h"
#include "rtstrelinp.h"
#include "rtsagfl.h"
#include "rtsbuffer.h"
#include "rtsutil.h"
#include "rtstime.h"

/*
** Private proto's
*/

#ifdef PROFILING
static void ShowProfile();
static void UpdateProfile();
#endif

/*
** install_signal_handlers():
** Give bug message in the event of crashing system
*/

void
CatchSignal(int sig)
{
    char *fName = "CatchSignal";

    switch(sig) {
        case SIGSEGV:
            rtsBug(fName, "Segmentation Fault");
            break;
#if !defined(MSDOS) && !defined(WIN32)
        case SIGBUS:
            rtsBug(fName, "Bus Error");
            break;
#endif
        default:
            rtsBug(fName, "Killed by signal %d", sig);
    };
}

static void install_signal_handlers(void)
{
#ifdef DIST
    char *fName = "install_signal_handlers";

    if (signal(SIGSEGV, CatchSignal) == SIG_ERR) {
        rtsAbort(fName, "signal failed");
    }

#if !defined(MSDOS) && !defined(WIN32)
    if (signal(SIGBUS, CatchSignal) == SIG_ERR) {
        rtsAbort(fName, "signal failed");
    }
#endif
#endif
}

static FILE *LexiconFile(void)
{
    char lexFName[256];
    FILE *lexFile;

    sprintf(lexFName, "%s.blf", get_parser_name());
    lexFile = fopen(lexFName, "rb");
    if(lexFile == NULL) {
        rtsFatal("cannot open binary lexicon file '%s'\n", lexFName);
    }

    return lexFile;
}

static void
log_unknown(unsigned pos, const char* word_form, unsigned len)
{
    if (lexer_stats_option) {
        char buf[1024];
        strncpy(buf, word_form, len);
        buf[len] = '\0';
        rtsWarning("skipped unknown token \"%s\" at pos %d\n", buf, pos);
    }
}

static void
init_lex_info(LexInfo* info, const Trie* trie, LEXICON* lexicon)
{
    info->trie = trie;
    info->lexicon = lexicon;
    /* lexnum  niet */
    /* eos_terminal? */
    info->terminals = term_names;
    info->matches = match_regexp_names;
    info->skips = skip_regexp_names;
    info->nr_terminals = nr_terminals;
    info->nr_skips = nr_skip_regexps;
    info->nr_matches = nr_match_regexps;
    info->nr_neg_memos = nr_memos;
    info->nr_syntax_nonterminals = nr_syntax_nonterminals;
    info->eos_terminal = eos_terminal;

    info->blanks = " \t\n\015";	/* this is constant */

    info->terminators = word_terminator_chars;	/* ",.\"';:!? \t\n" */
    info->invisibles = invisible_chars;		/* " \t\n\"'" */
    info->delimiters = list_delimiter_chars;	/* "[|]" */

    if (no_translate) {
        info->translate_src = "";
        info->translate_dst = "";
    } else {
        info->translate_src = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        info->translate_dst = "abcdefghijklmnopqrstuvwxyz";
    }

    /* log_unknown gekopieerd naar uit testlex.c */
    info->log_unknown = log_unknown;
}


LEXICON* Prelude(LexInfo *lex_info)
{
    Trie*	lex_trie = NULL;
    LEXICON* the_lexicon = NULL;

    install_signal_handlers();

    if (use_lexicon) {
        the_lexicon = lexicon_new(LexiconFile());
        lex_trie = get_lexicon_trie(the_lexicon);
    }

    InitBuffer();
    InitIO();
    init_lex_info(lex_info, lex_trie, the_lexicon);
    init_lexer(lex_info);
    init_parser();

    return the_lexicon;
}

static void Postlude(int nop, LEXICON *the_lexicon)
{
#ifdef PROFILING
    if ((prof_count > 0) && !nop) {
        if (verbose && !reset_profile) {
            ShowProfile("corpus");
        }

        if (update_profile) {
            UpdateProfile();
        }
    }
#endif

    end_parser();
    end_lexer();
    EndIO();
    EndBuffer();

    if(use_lexicon) {
        lexicon_free(the_lexicon);
    }
}

Trellis*
new_trellis(const char* input, LEXICON* the_lex)
{
    switch (lexing_method) {
	case TRELLIS_LEXING:
		 return make_trellis_from_prelexer_output(input, the_lex);
	default: return make_trellis_by_word_lexing(input, the_lex);
    }
}

/*
** Main
*/

#ifdef DEBUGRTS
int rts_main(int argc, char* const argv[])
#else /* DEBUGRTS */
int main(int argc, char* const argv[])
#endif /* DEBUGRTS */
{
    LexInfo lex_info; /* should exist all through the process */
    LEXICON* the_lexicon;
    int nop = 1;
#ifdef GEN_RTS
    int sent_nr = 0;
#else
    int read_by_paragraph = segment_mode;
    char newline_replacement = ' ';	/* '\n' if TRELLIS_LEXING, below */
#endif
    StateIndicator node = NULL;

    process_options(argc, argv);
    if (show_options_option) {
        show_options();
    }

    the_lexicon = Prelude(&lex_info);

#ifndef GEN_RTS
    /*
    * While we have input, lexicalize and parse it.
    */
    if (lexing_method == TRELLIS_LEXING) {
	read_by_paragraph = 1;
	newline_replacement = '\n';
    }
    while (read_by_paragraph ? read_input_block(newline_replacement)
			     : read_input_line()) {
        Trellis* trellis;
        const char* input_text;

        /*
        // Lexicalize input
        */
        input_text = get_input_text();
        reset_timers();
	if (!*input_text || !strcmp(input_text, "\n")) {
		continue;	/* skip empty lines silently */
	}
        start_scan_time();
        trellis = new_trellis(input_text, the_lexicon);
        stop_scan_time();
        /*
         * If we have non-empty input, parse it
         */
        if (!is_empty_trellis(trellis)) {
#ifndef STANDALONE_LEXER
            if (graph_option) {
                print_trellis(trellis);
            }
#ifdef COUNTERS
            show_neg_memo_blocks(trellis);
#endif
            if (max_parses > 0) {
                nop = 0;
                do {
                    node = parse_trellis(trellis, node);
#ifdef COUNTERS
                    show_neg_memo_blocks(trellis);
#endif
                } while (node);
            }
#else /* STANDALONE_LEXER */
            print_trellis(trellis);
#endif /* STANDALONE_LEXER */
        }

        /*
        * Clean-up trellis
        */
        delete_trellis(trellis);

        /* and synchronise with other programs */
        maybe_output_sync();
    }
#else /* GEN_RTS */
    if (!generate_option) {
        rtsAbort("main()",
                 "non-generative executable linked against generative rts!\n");
    }

    srand((unsigned int) time(NULL));

    while (sent_nr < max_parses) {
        gen_output_init(sent_nr + 1);
        reset_timers();
        node = parse_trellis(NULL, NULL);
        if (get_nr_parses() > 0) {
	    sent_nr++;
	}
        gen_output_free();
    }
#endif /* GEN_RTS */

    Postlude(nop, the_lexicon);

    rtsExit(0);
    return 1; /* not reached */
}

/*
//------------------------------------------------------------------------------
// Profiling
//
// Profiling code currently not generated. See version 1.7 for profiling
//------------------------------------------------------------------------------
*/

#ifdef PROFILING

static void ShowProfile(msg)
  char *msg;
{
  long i;
  char buf[256];
  int len;

  long **prof_exec_cnt = get_profile_exec();
  long **prof_succ_cnt = get_profile_succ();
  long **prof_parse_cnt = get_profile_parse();

  sprintf(buf, "Profile (%s)                              ", msg);
  sprintf(buf + 40 ,"|      Exec |      Succ |     Parse\n"
          "----------------------------------------"
                    "+-----------+-----------+-----------\n");
  rtsMessage(buf);

  for (i = 0; i < prof_count; i++)
  {
#if 0 /* not portable */
    if ((len = sprintf(buf, "%s", prof_table[i].id)) > 29)
      len = 29;
#else
    sprintf(buf, "%s", prof_table[i].id);
    len = strlen(buf);
    if (len > 29)
      len = 29;
#endif
    sprintf(buf + len, "/%-2d", prof_table[i].arity);
    rtsMessage("%-32s alt %2d | %9ld | %9ld | %9ld\n",
               buf, prof_table[i].altno,
               *(prof_exec_cnt[i]),
               *(prof_succ_cnt[i]),
               *(prof_parse_cnt[i]));
  };
  rtsMessage("\n");
}

#define MAXBUF 512

static char buf[MAXBUF];
static long line = 0;
static char *maxbuf = buf + MAXBUF;

static int ReadLine(char *buf, FILE *file, char *fname)
{
  if (fgets(buf, MAXBUF, file) == NULL)
  {
    if (feof(file))
      return 0;
    rtsError("error reading from file '%s'\n", fname);
    rtsAbort("ReadLine", "update aborted");
  };
  line++;
  return 1;
}

static void WriteFile(buf, file, fname)
  char *buf;
  FILE *file;
  char *fname;
{
  if (fputs(buf, file) == EOF)
  {
    fclose(file);
    remove(fname);
    rtsError("Error writing to file '%s'\n", fname);
    rtsAbort("WriteFile", "update aborted\n");
  };
}

static void WriteProfile(dst, dst_name)
  FILE *dst;
  char *dst_name;
{
  int i;

  long **prof_exec_cnt = get_profile_exec();
  long **prof_succ_cnt = get_profile_succ();
  long **prof_parse_cnt = get_profile_parse();

  sprintf(buf, "#%ld %s %s\n",
          prof_count, fingerprint, TimeToText(gra_version));
  WriteFile(buf, dst, dst_name);

  for (i = 0; i < prof_count; i++)
  {
    sprintf(buf, "%s/%d/%d %ld %ld %ld\n",
            prof_table[i].id, prof_table[i].arity, prof_table[i].altno,
            *(prof_exec_cnt[i]), *(prof_succ_cnt[i]), *(prof_parse_cnt[i]));
    WriteFile(buf, dst, dst_name);
  };
}

static void UpdateCounters(src, dst, src_name, dst_name)
  FILE *src;
  FILE *dst;
  char *src_name;
  char *dst_name;
{
  long **prof_exec_cnt = get_profile_exec();
  long **prof_succ_cnt = get_profile_succ();
  long **prof_parse_cnt = get_profile_parse();

  long i;

  for (i = 0; i < prof_count; i++)
  {
    char *p, *q, *r, *s;
    long exec, succ, parse;

    if (!ReadLine(buf, src, src_name))
    {
      fclose(dst);
      remove(dst_name);
      rtsError("Line %ld: missing line\n", line + 1);
      rtsAbort("UpdateCounters", "update aborted");
    };

    p = buf;
    while ((*p != '/') && (p < maxbuf))
      p++;
    while ((*p != ' ') && (p < maxbuf))
      p++;
    exec = strtol(p, &q, 10);
    succ = strtol(q, &r, 10);
    parse = strtol(r, &s, 10);

    if ((p == q) || (q == r) || (r == s))
    {
      fclose(dst);
      remove(dst_name);
      rtsError("Line %ld: invalid or missing counter\n", line);
      rtsAbort("UpdateCounters", "update aborted");
    };

    if (!new_profile)
    {
      *(prof_exec_cnt[i]) += exec;
      *(prof_succ_cnt[i]) += succ;
      *(prof_parse_cnt[i]) += parse;
    };
  };
}

static void CopyProfile(src, dst, src_name, dst_name)
  FILE *src;
  FILE *dst;
  char *src_name;
  char *dst_name;
{
  int idlen = strlen(fingerprint);

  while (ReadLine(buf, src, src_name))
  {
    if (buf[0] == '#')
    {
      char *id;

      long cnt = strtol(buf + 1, &id, 10);
      if (id == buf + 1)
      {
        fclose(dst);
        remove(dst_name);
        rtsError("Line %ld: number expected\n", line);
        rtsAbort("CopyProfile", "update aborted");
      };
      while (*id == ' ')
        id++;
      if (strncmp(id, fingerprint, idlen) == 0)
      {
        if (cnt != prof_count)
        {
          fclose(dst);
          remove(dst_name);
          rtsError("Line %ld: invalid id or number\n", line);
          rtsAbort("CopyProfile", "update aborted");
        }
        else
          UpdateCounters(src, dst, src_name, dst_name);
      }
      else
        WriteFile(buf, dst, dst_name);
    }
    else
      WriteFile(buf, dst, dst_name);
  };
}

static int FileExists(fname)
  char *fname;
{
  struct stat sbuf;

  return (stat(fname, &sbuf) >= 0);
}

static char *TempFName(dir, prefix)
  char *dir;
  char *prefix;
{
  char *fname;

  if ((fname = tempnam(dir, prefix)) == NULL)
    rtsAbort("TempFName", "cannot get name for temp file");
  return fname;
}

static FILE *OpenFile(fname, access)
  char *fname;
  char *access;
{
  FILE *file;

  if ((file = fopen(fname, access)) == NULL)
  {
    rtsError("Cannot open file '%s'\n", fname);
    rtsAbort("OpenFile", "update aborted");
  };
  return file;
}

static void CloseFile(file, fname)
  FILE *file;
  char *fname;
{
  if (fclose(file) != 0)
  {
    rtsError("Cannot close file '%s'\n", fname);
    rtsAbort("CloseFile", "update aborted");
  };
}

#if defined(MSDOS) || defined(WIN32)
#  define MV "REN"
#else
#  define MV "mv"
#endif

static void MoveFile(src, dst)
  char *src;
  char *dst;
{
  char cmd[512];

  sprintf(cmd, MV " %s %s", src, dst);
  if (system(cmd) == -1)
  {
    rtsError("Cannot move file '%s' to '%s'\n", src, dst);
    rtsAbort("MoveFile", "update aborted");
  };
}

static void UpdateProfile()
{
  char profname[256];

  sprintf(profname, "%s.prof", get_parser_name());

  rtsMessage("Updating profile '%s'...\n", profname);

  if (FileExists(profname))
  {
    char *new_name = TempFName("./", "prof");
    FILE *old = OpenFile(profname, "r");
    FILE *new = OpenFile(new_name, "w");

    CopyProfile(old, new, profname, new_name);
    WriteProfile(new, new_name);
    CloseFile(new, new_name);
    fclose(old);
    MoveFile(new_name, profname);
    free(new_name);
  }
  else
  {
    FILE *profile = OpenFile(profname, "w");

    WriteProfile(profile, profname);
    CloseFile(profile, profname);
  };
  rtsMessage("Done\n");
}

#endif /* PROFILING */
