//
// Copyright (C) 1999, 2000 Marco Kesseler
//

implementation module dct

import StdReal, StdInt, StdBool, StdMisc, StdList, StdEnum, StdArray
import basic, matrix

/*********************************************************************************
 *
 * This module contains three implementations of the inverse DCT (Discrete
 * Cosine Transform).
 * - A straightforward, but slow one that uses floating point arithmetic. It is
 *   based on a straightforward implementaton by Jeroen Fokker, Department of
 *   Computer Science, Utrecht, the Netherlands.
 * - A fast floating point version, based on jidctflt.c by Thomas G. Lane.,
 *   part of the Idependent JPEG Group's software, Release 6b.
 * - A fast fixed point version, based on jidctint.c by Thomas G. Lane.,
 *   part of the Idependent JPEG Group's software, Release 6b.There are
 *   faster fixed-point versions around, but these are less accurate.
 *
 * All three implementations expect a block of 8x8 coefficients and a block
 * of 8x8 scaling factors. Both blocks are passed as a list of columns. The
 * result is an 8x8 block, that is returned as a list of rows. Passing in the
 * arguments as a list of columns may seem odd, but it allows us to avoid
 * a transpose of these arguments.
 *
 * The comments below have been partly taken over from the implementation of
 * Thomas Lane, simply because they apply to these implementations equally well.
 *
 ************************************************************************************/
 
/**************************************************************************************
 *
 * Descale
 *
 *************************************************************************************/

/* Descale and correctly round an INT32 value that's scaled by N bits.
 * We assume '>>' rounds towards minus infinity, so adding
 * the fudge factor is correct for either sign of X.
 */
//descale :: Int Int -> Int
descale n x :== (x + (1 << (n-1))) >> n

/**************************************************************************************
 *
 * This is a straightforward, but slow, inverse DCT.
 *
 *************************************************************************************/

IDCT64SlowScaleFactors :: [[Int]] -> [[Int]]
IDCT64SlowScaleFactors factors = factors

cosinuses =: map f [1,3..15]
where
	f x = map g [0..7]
	where
		g 0 = 0.5 / sqrt 2.0
		g u = 0.5 * cos(toReal(x * u) * (pi/16.0))

IDCT64Slow :: [[Int]] [[Int]] (Int -> Int) -> [[Int]]
IDCT64Slow mat factors range
	= matMap truncate (map inverseDCT8 (transpose (map inverseDCT8 (matMap2 dequantize mat factors))))
where
	inverseDCT8 vec = matrixVectorProduct cosinuses vec
	
	dequantize coef factor = toReal (coef * factor)
	
	truncate :: Real -> Int
	truncate r = range (round r) 

/*************************************************************************************
 *
 * This floating point implementation should be more accurate than any of the
 * integer IDCT implementations.  However, it may not give the same results on all
 * machines because of differences in roundoff behavior. Speed will depend
 * on the hardware's floating point capacity.
 *
 * A 2-D IDCT can be done by 1-D IDCT on each column followed by 1-D IDCT
 * on each row (or vice versa, but it's more convenient to emit a row at
 * a time).  Direct algorithms are also available, but they are much more
 * complex and seem not to be any faster when reduced to code.
 *
 * This implementation is based on Arai, Agui, and Nakajima's algorithm for
 * scaled DCT.  Their original paper (Trans. IEICE E-71(11):1095) is in
 * Japanese, but the algorithm is described in the Pennebaker & Mitchell
 * JPEG textbook.
 *
 * While an 8-point DCT cannot be done in less than 11 multiplies, it is
 * possible to arrange the computation so that many of the multiplies are
 * simple scalings of the final outputs.  These multiplies can then be
 * folded into the multiplications or divisions by the JPEG quantization
 * table entries.  The AA&N method leaves only 5 multiplies and 29 adds
 * to be done in the DCT itself.
 * The primary disadvantage of this method is that with a fixed-point
 * implementation, accuracy is lost due to imprecise representation of the
 * scaled quantization values.  However, that problem does not arise if
 * we use floating point arithmetic.
 *
 ************************************************************************************/

/* For float AA&N IDCT method, multipliers are equal to quantization
 * coefficients scaled by scalefactor[row]*scalefactor[col], where
 *   scalefactor[0] = 1
 *   scalefactor[k] = cos(k*PI/16) * sqrt(2)    for k=1..7
 */

IDCT64RealScaleFactors :: [[Int]] -> [[Real]]
IDCT64RealScaleFactors quaTab = [row q r \\ q <- quaTab & r <- [0..7]]
where
	row quaTab r = [(toReal q) * factor.[r] * factor.[c] \\ q <- quaTab & c <- [0..7]]
	
	factor = {scaleFactor i \\ i <- [0..7]}

	scaleFactor 0 = 1.0
	scaleFactor k = (cos ((toReal k) * pi / 16.0)) * root
	
	root = sqrt 2.0
	  
IDCT64Real :: [[Int]] [[Real]] (Int -> Int) -> [[Int]]
IDCT64Real block factors range
	= map (IDCT8row range) (transpose (map2 IDCT8col block factors))
where
	/*
	 * Perform dequantization and inverse DCT on one block of coefficients.
	 */
	IDCT8col :: [Int] [Real] -> [Real]
	IDCT8col [i0, i1, i2, i3, i4, i5, i6, i7] fs=:[f0, f1, f2, f3, f4, f5, f6, f7]
	| i1 == 0 && i2 == 0 && i3 == 0 && i4 == 0 && i5 == 0 && i6 == 0 && i7 == 0
		= [dcval, dcval, dcval, dcval, dcval, dcval, dcval, dcval]
		with
			dcval = dequantize i0 f0
		= IDCT8
			(dequantize i0 f0)
			(dequantize i1 f1)
			(dequantize i2 f2)
			(dequantize i3 f3)
			(dequantize i4 f4)
			(dequantize i5 f5)
			(dequantize i6 f6)
			(dequantize i7 f7)
		where
			dequantize :: !Int !Real -> Real
			dequantize coef quantval = (toReal coef) * quantval

	/* Note that we must descale the results by a factor of 8 == 2**3. */

	IDCT8row :: (Int -> Int) [Real] -> [Int]
	IDCT8row range [i0, i1, i2, i3, i4, i5, i6, i7]
		= map truncate (IDCT8 i0 i1 i2 i3 i4 i5 i6 i7)
	where
		truncate :: !Real -> Int
		truncate r = range (descale 3 (round r))

	IDCT8 :: Real Real Real Real Real Real Real Real -> [Real]
	IDCT8 i0 i1 i2 i3 i4 i5 i6 i7
		= let!
		
    		phase1_a = i0 + i4
   			phase1_b = i0 - i4
    		phase1_c = i2 + i6
    		phase1_d = (i2 - i6) * 1.414213562 - phase1_c
    		phase1_e = i5 + i3
    		phase1_f = i5 - i3
    		phase1_g = i1 + i7
    		phase1_h = i1 - i7

    		phase2_a = phase1_a + phase1_c
    		phase2_b = phase1_a - phase1_c
    		phase2_c = phase1_b + phase1_d
    		phase2_d = phase1_b - phase1_d
    		phase2_e = phase1_g + phase1_e
    		phase2_f = (phase1_g - phase1_e) * 1.414213562
    		phase2_g = (phase1_f + phase1_h) * 1.847759065
    		phase2_h = 1.082392200 * phase1_h - phase2_g
    		phase2_i = -2.613125930 * phase1_f + phase2_g

    		phase3_a = phase2_i - phase2_e
    		phase3_b = phase2_f - phase3_a
    		phase3_c = phase2_h + phase3_b
    		
    		o0 = phase2_a + phase2_e
    		o1 = phase2_c + phase3_a
    		o2 = phase2_d + phase3_b
    		o3 = phase2_b - phase3_c
	  	  	o4 = phase2_b + phase3_c
	  	  	o5 = phase2_d - phase3_b
	  	  	o6 = phase2_c - phase3_a
	  	  	o7 = phase2_a - phase2_e
	  	in
	  		[o0, o1, o2, o3, o4, o5, o6, o7]

/*********************************************************************************
 *
 * This file contains a slow-but-accurate integer implementation of the
 * inverse DCT (Discrete Cosine Transform).  In the IJG code, this routine
 * must also perform dequantization of the input coefficients.
 *
 * A 2-D IDCT can be done by 1-D IDCT on each column followed by 1-D IDCT
 * on each row (or vice versa, but it's more convenient to emit a row at
 * a time).  Direct algorithms are also available, but they are much more
 * complex and seem not to be any faster when reduced to code.
 *
 * This implementation is based on an algorithm described in
 *   C. Loeffler, A. Ligtenberg and G. Moschytz, "Practical Fast 1-D DCT
 *   Algorithms with 11 Multiplications", Proc. Int'l. Conf. on Acoustics,
 *   Speech, and Signal Processing 1989 (ICASSP '89), pp. 988-991.
 * The primary algorithm described there uses 11 multiplies and 29 adds.
 * We use their alternate method with 12 multiplies and 32 adds.
 * The advantage of this method is that no data path contains more than one
 * multiplication; this allows a very simple and accurate implementation in
 * scaled fixed-point arithmetic, with a minimal number of shifts.
 *
 * Each 1-D IDCT step produces outputs which are a factor of sqrt(N)
 * larger than the true IDCT outputs.  The final outputs are therefore
 * a factor of N larger than desired; since N=8 this can be cured by
 * a simple right shift at the end of the algorithm.  The advantage of
 * this arrangement is that we save two multiplications per 1-D IDCT,
 * because the y0 and y4 inputs need not be divided by sqrt(N).
 *
 * We have to do addition and subtraction of the integer inputs, which
 * is no problem, and multiplication by fractional constants, which is
 * a problem to do in integer arithmetic.  We multiply all the constants
 * by CONST_SCALE and convert them to integer constants (thus retaining
 * CONST_BITS bits of precision in the constants).  After doing a
 * multiplication we have to divide the product by CONST_SCALE, with proper
 * rounding, to produce the correct output.  This division can be done
 * cheaply as a right shift of CONST_BITS bits.  We postpone shifting
 * as long as possible so that partial sums can be added together with
 * full fractional precision.
 *
 * The outputs of the first pass are scaled up by PASS1_BITS bits so that
 * they are represented to better-than-integral precision.  These outputs
 * require BITS_IN_JSAMPLE + PASS1_BITS + 3 bits; this fits in a 16-bit word
 * with the recommended scaling.  (To scale up 12-bit sample data further, an
 * intermediate INT32 array would be needed.)
 *
 * To avoid overflow of the 32-bit intermediate results in pass 2, we must
 * have BITS_IN_JSAMPLE + CONST_BITS + PASS1_BITS <= 26.  Error analysis
 * shows that the values given below are the most effective.
 *
 ********************************************************************************/

/* If sample precision is 8 */
CONST_BITS :== 13
PASS1_BITS :== 2

/* If sample precision is larger than 8 */
//CONST_BITS :== 13
//PASS1_BITS :== 1		/* lose a little precision to avoid overflow */

CONST_SCALE :== toReal (1 << CONST_BITS)

/* Convert a positive real constant to an integer scaled by CONST_SCALE. */
FIX x :== round (x * CONST_SCALE)

/* The following will only work if CONST_BITS = 13 */
FIX_0_298631336 :== 2446	/* FIX(0.298631336) */
FIX_0_390180644 :== 3196	/* FIX(0.390180644) */
FIX_0_541196100 :== 4433	/* FIX(0.541196100) */
FIX_0_765366865 :== 6270	/* FIX(0.765366865) */
FIX_0_899976223 :== 7373	/* FIX(0.899976223) */
FIX_1_175875602 :== 9633	/* FIX(1.175875602) */
FIX_1_501321110 :== 12299	/* FIX(1.501321110) */
FIX_1_847759065 :== 15137	/* FIX(1.847759065) */
FIX_1_961570560 :== 16069	/* FIX(1.961570560) */
FIX_2_053119869 :== 16819	/* FIX(2.053119869) */
FIX_2_562915447 :== 20995	/* FIX(2.562915447) */
FIX_3_072711026 :== 25172	/* FIX(3.072711026) */

/* The following will always work, but unfortunately the Clean
 * compiler does not evaluate constant expressions, so this is
 * much slower than using the solution above. Defining these
 * constants as constant graphs is even slower than defining them
 * as macros, so we do not do that either...
 */
/*
FIX_0_298631336 :== FIX 0.298631336
FIX_0_390180644 :== FIX 0.390180644
FIX_0_541196100 :== FIX 0.541196100
FIX_0_765366865 :== FIX 0.765366865
FIX_0_899976223 :== FIX 0.899976223
FIX_1_175875602 :== FIX 1.175875602
FIX_1_501321110 :== FIX 1.501321110
FIX_1_847759065 :== FIX 1.847759065
FIX_1_961570560 :== FIX 1.961570560
FIX_2_053119869 :== FIX 2.053119869
FIX_2_562915447 :== FIX 2.562915447
FIX_3_072711026 :== FIX 3.072711026
*/

/* Dequantize a coefficient by multiplying it by the multiplier-table
 * entry; produce an int result.  In this module, both inputs and result
 * are 16 bits or less, so either int or short multiply will work.
 */

/* For LL&M IDCT method, multipliers are equal to raw quantization
 * coefficients, but are stored as ints to ensure access efficiency.
 */
IDCT64IntScaleFactors :: [[Int]] -> [[Int]]
IDCT64IntScaleFactors factors = factors

/*
 * Perform dequantization and inverse DCT on one block of coefficients.
 */

IDCT64Int :: [[Int]] [[Int]] (Int -> Int) -> [[Int]]
IDCT64Int block factors range
	= map (IDCT8row range) (transpose (map2 IDCT8col block factors))
where
	/* Pass 1: process columns from input. */
	/* Note results are scaled up by sqrt(8) compared to a true IDCT; */
	/* furthermore, we scale the results by 2**PASS1_BITS. */
	
	IDCT8col :: [Int] [Int] -> [Int]
	IDCT8col [i0, i1, i2, i3, i4, i5, i6, i7] fs=:[f0, f1, f2, f3, f4, f5, f6, f7]
	| i1 == 0 && i2 == 0 && i3 == 0 && i4 == 0 && i5 == 0 && i6 == 0 && i7 == 0
		= [dcval, dcval, dcval, dcval, dcval, dcval, dcval, dcval]
		with
			dcval = (dequantize i0 f0) << PASS1_BITS
		= map (descale (CONST_BITS-PASS1_BITS)) 
			(IDCT8
				(dequantize i0 f0)
				(dequantize i1 f1)
				(dequantize i2 f2)
				(dequantize i3 f3)
				(dequantize i4 f4)
				(dequantize i5 f5)
				(dequantize i6 f6)
				(dequantize i7 f7))
	where
		dequantize :: !Int !Int -> Int
		dequantize coef quantval = coef * quantval

  	/* Pass 2: Note that we must descale the results by a factor of 8 == 2**3, */
  	/* and also undo the PASS1_BITS scaling. */

	IDCT8row :: (Int -> Int) [Int] -> [Int]
	IDCT8row range [i0, i1, i2, i3, i4, i5, i6, i7]
		= map truncate (IDCT8 i0 i1 i2 i3 i4 i5 i6 i7)
  	where
  		truncate :: !Int -> Int
		truncate r = range (descale (CONST_BITS+PASS1_BITS+3) r)

	IDCT8 :: Int Int Int Int Int Int Int Int -> [Int]
	IDCT8 i0 i1 i2 i3 i4 i5 i6 i7
		= let!
		
			/* Even part: reverse the even part of the forward DCT. */
    		/* The rotator is sqrt(2)*c(-6). */
        
    		phase1_a = (i2 + i6) * FIX_0_541196100
    		phase1_b = phase1_a + (i6 * (~FIX_1_847759065))
    		phase1_c = phase1_a + (i2 * FIX_0_765366865)
    		phase1_d = (i0 + i4) << CONST_BITS
    		phase1_e = (i0 - i4) << CONST_BITS
        	phase1_f = i7 + i1
    		phase1_g = i5 + i3
    		phase1_h = i7 + i3
    		phase1_i = i5 + i1
    		phase1_j = i7 * FIX_0_298631336
    		phase1_k = i5 * FIX_2_053119869
    		phase1_l = i3 * FIX_3_072711026
    		phase1_m = i1 * FIX_1_501321110
    		
    		phase2_a = phase1_d + phase1_c
    		phase2_b = phase1_d - phase1_c
    		phase2_c = phase1_e + phase1_b
    		phase2_d = phase1_e - phase1_b
    		phase2_e = (phase1_h + phase1_i) * FIX_1_175875602
    		phase2_f = phase1_f * (~FIX_0_899976223)
    		phase2_g = phase1_g * (~FIX_2_562915447)
    		phase2_h = phase1_h * (~FIX_1_961570560)
    		phase2_i = phase1_i * (~FIX_0_390180644)
    
    		phase3_a = phase2_h + phase2_e
    		phase3_b = phase2_i + phase2_e
    
    		phase4_a = phase1_j + phase2_f + phase3_a
    		phase4_b = phase1_k + phase2_g + phase3_b
    		phase4_c = phase1_l + phase2_g + phase3_a
    		phase4_d = phase1_m + phase2_f + phase3_b        	

			o0 = phase2_a + phase4_d
			o1 = phase2_c + phase4_c
			o2 = phase2_d + phase4_b
			o3 = phase2_b + phase4_a
    		o4 = phase2_b - phase4_a
    		o5 = phase2_d - phase4_b
    		o6 = phase2_c - phase4_c
    		o7 = phase2_a - phase4_d
    	in
    		[o0, o1, o2, o3, o4, o5, o6, o7]
