//
// Copyright (C) 1999, 2000, Marco Kesseler
//

implementation module bitstream2

import StdInt, StdChar, StdArray, StdClass, StdMisc

import bits

next8 [x : xs] = (toInt x, xs)
next8 [] = (0, [])

:: BitStream	:== (!Int, !Int, [Char])	// (buffer, available, chars)

/****************************************************************
 *
 * Accesssing a stream of bits
 *
 ***************************************************************/

:: LEBitStream :== BitStream

toLEBitStream :: [Char] -> LEBitStream
toLEBitStream chars = (0, 0, chars)

fromLEBitStream :: LEBitStream -> [Char]
fromLEBitStream (bits, available, stream)
| available > 8	= fromLEBitStream (bits >> 8, available - 8, [toChar (lowBits bits 8) : stream])
				= stream

peekLEBits :: LEBitStream Int -> (Int, LEBitStream)
peekLEBits bitstream=:(bits, available, stream) nbits
| available >= nbits
	= (bits >> keep, bitstream)
	with
		keep = available - nbits
	= peekLEBits (bits << 8 bitor hd, available + 8, tl) nbits
	with
		(hd, tl) = next8 stream

getLEBits :: LEBitStream Int -> (Int, LEBitStream)
getLEBits (bits, available, stream) nbits
| available >= nbits
	= (bits >> keep, (lowBits bits keep, keep, stream))
	with
		keep = available - nbits
	= getLEBits (bits << 8 bitor hd, available + 8, tl) nbits
	with
		(hd, tl) = next8 stream

skipLEBits :: LEBitStream Int -> LEBitStream
skipLEBits (bits, available, stream) nbits
| available >= nbits
	= (lowBits bits keep, keep, stream)
	with
		keep = available - nbits
	= skipLEBits (hd, 8, tl) (nbits - available)
	with
		(hd, tl) = next8 stream

//**************************************************************
//
//	This module implements functions on bitstreams, assuming
//	that each unit of bits ends with the most significant bits.
//
//**************************************************************

//**************************************************************
//
//	Having a big-endian bitstream means that each new byte should
//	be put BEFORE the bits that have been read so far, and not
//	after it.
//
//	The ReadableBEBitStream data structure consists of a tuple
//	with:
//	- a 32-bit buffer
//	- the number of bits in the buffer
//	- a list of chars, containing unread bits
//
//***************************************************************

:: BEBitStream :== BitStream

toBEBitStream :: [Char] -> BEBitStream
toBEBitStream chars = (0, 0, chars)

fromBEBitStream :: BEBitStream -> [Char]
fromBEBitStream (bits, available, stream)
| available > 8	= fromBEBitStream (lowBits bits remain, remain, newstream)
				with
					newstream = [toChar (bits >> remain) : stream]
					remain = available - 8
				= stream

peekBEBits :: BEBitStream Int -> (Int, BEBitStream)
peekBEBits bitstream=:(bits, available, stream) nbits
| available >= nbits
	= (lowBits bits nbits, bitstream)
	= peekBEBits (bits bitor (hd << available), available + 8, tl) nbits
	with
		(hd, tl) = next8 stream

getBEBits :: BEBitStream Int -> (Int, BEBitStream)
getBEBits (bits, available, stream) nbits
| available >= nbits
	= (lowBits bits nbits, (bits >> nbits, available - nbits, stream))
	= getBEBits (bits bitor (hd << available), available + 8, tl) nbits
	with
		(hd, tl) = next8 stream

skipBEBits :: BEBitStream Int -> BEBitStream
skipBEBits (bits, available, stream) nbits
| available >= nbits
	= (bits >> nbits, available - nbits, stream)
	= skipBEBits (hd, 8, tl) (nbits - available)
	with
		(hd, tl) = next8 stream
		
getBEBitList :: BEBitStream [Int] -> ([Int], BEBitStream)
getBEBitList (bits, available, stream) nbits = getNext bits available stream nbits
where

	getNext bits available stream [] = ([], (bits, available, stream))
	getNext bits available stream [nbits : nbit_list] = get bits available stream nbits nbit_list
	
	get bits available stream=:[b : bs] nbits nbit_list
	| available < nbits	= get (bits bitor ((toInt b) << available)) (available + 8) bs nbits nbit_list
						= ([lowBits bits nbits : more_bits], new_stream)
						with
							(more_bits, new_stream) = getNext (bits >> nbits) (available - nbits) stream nbit_list	
	get bits available [] nbits nbit_list
	| available < nbits	= abort "getBEBitList: premature end of bitstream"
						= ([lowBits bits nbits : more_bits], new_stream)
						with
							(more_bits, new_stream) = getNext (bits >> nbits) (available - nbits) [] nbit_list
