//
// Copyright (C) 1999, 2000, Marco Kesseler
//

definition module jpeg

import StdString, imagePlanes

// This module exports 3 functions:
// - JPEGDecode1: takes a JPEG stream as input and delivers the first scan of the
//   first frame that does not have any application-specific 'markers' (see below).
//   Ususally this is the 'main' image. In theory, a baseline JPEG stream may contain only
//	 a single image (or frame), but in practice they sometimes contain more. Frames with
//   application-specific markers commonly represent application-specific previews
//   and the like. JPEGDecode1 ignores these. Furthermore, each frame may contain
//	 one or more scans, each scan consisting of one or more components. In practice
//	 however, a frame commonly contains only a single scan. JPEGDecode1 delivers
//	 the first scan of the chosen frame.
// - JPEGDecode: takes a JPEG stream as input and delivers all scans of all frames
//   in the stream. It deliverers these as a list of JPEGPart elements. The start
//   of a frame is indicated by a JPEGFRAME element, which contains the comments
//	 of the frame, and its application markers. Scans are delivered as JPEGSCAN
//	 elements. Apart from the comments and application markers of the scan, these
//	 elements contain the image itself.
// - DCTDecode: implements the DCTDecode functionality of PDF/PostScript.
// All versions are based on the same decoder functions and implement
// baseline JPEG decoding only.
//
// Notes on the JPEG format:
//
// In theory, a baseline JPEG stream may contain only a single image (or frame),
// but in practice they sometimes contain more. Frames with application-specific
// markers commonly represent application-specific previews and the like.
//
// A single image may contain multiple scans, each having one or more components.
// If more than one scan is present, they represent different 'planes' of the same
// image. For example, the first scan (with one component) stores the red
// component of the image, the second the green component, and the third the
// blue component. In practice however, a frame commonly contains only a single
// scan with multiple components. In this case, the components are stored
// 'interleaved'. For example: first the R, G and B component of the first
// pixels, than the R, G and B component of the second, etc.
//
// Unfortunately, the JPEG standard does not say anything about colour spaces.
// In practice, it boils down to the following:
// - images with a single component are greyscale images
// - images with three components are RGB images
// - images with four components are CMYK images.
//
// But there is more: RGB and CMYK images generally do not compress as well as
// images that use a 'YCbCr' or a 'YCbCrK' colour space. To accommodate for
// such colour spaces, several parties have defined their own application
// markers which allow them to specify and detect JPEG images with other
// colour spaces than RGB and YMCK. The follong two are rather common: 
// - The JFIF (JPEG File Interchange Format) 'APP0' marker: If this marker
//	 is present, the image should have 1 or 3 components, having respectively
//   greyscale or YCbCr colour spaces.
// - Adobe's 'APPE' marker: This marker distinguishes greyscale, RGB, CMYK,
//   YCbCr and YCbCrK colour spaces for images with 1, 3, or 4 components.
// As these markers are so common, we do not treat them as ordinary application-
// specific markers. Instead, we interpret these markers and transform the image
// so that it comes out as an ordinary greyscale, RGB, or CMYK image. To reflect
// these transformations JPEGDecode NOT return these markers as part of its result
//
// Memory Requirements:
// 
// These largely depend on the order in which image data is accessed. DCTDecode
// is safest in this respect, provided that its output is consumed in the order
// that it gets delivered (there is normally little reason to do otherwise).
// 
// JPEGDecode1 is a bit less safe. It is best to access all rows from top to bottom
// and all pixels within a row from left to right (see imageToCharList). If one
// starts with accessing the rightmost pixel at the bottom, memory requirements
// will probably be huge, as this pixel is stored towards the end of the stream.
// I do not know exactly to what extend preceding pixels will have to be evaluated in
// order to access this last pixel, but at least all preceding characters of the
// stream will get loaded into memory.
//
// For JPEGDecode the same holds as for JPEGDecode1, but in addition one must also
// be careful in which order one accesses the scans in the resulting list. If
// you use the standard 'last' function to access the last image first, you will
// notice that some images quickly lead to considerable memory use. The reason for
// this is simple: in order to determine whether an image is the last one in the
// stream, one has to check whether there is a next one. This implies skipping
// - and remembering - all source bytes of the current image.
//
// Some of these problems can be avoided by reading directly from file, but
// I have not yet any plans in this direction.
//
// Aborts:
//
// There are a number of situations that the decoder may abort. Notably:
// - The image is not a baseline image
// - The input is incorrect
// - JPEGDecode1 cannot find an image without application-specific markers.
//
// Colours:
//
// If colours come out 'wrong' the image probably has some application-specific
// markers that specify some colour transformation.
//

:: JPEGResult :== [JPEGPart]
:: JPEGPart
	= JPEGFRAME [String] [String]		// Comments and application markers
	| JPEGSCAN [String] [String] Image	// Comments, application markers and a scan
	
JPEGDecode :: [!Char] -> JPEGResult
JPEGDecode1 :: [!Char] -> Image

// The DCTDecode function is a specialised JPEG decoder that implements
// the DCTDecode functionality as specified in the PDF reference
// manual. It flattens the resulting image(s) into a list of characters.
// The colour space and precision are not delivered, as these can
// be found elsewhere in a PDF file.
 
DCTDecode :: [!Char] -> [!Char]
