//
// Copyright (C) 1999, 2000, Marco Kesseler
//

implementation module bitstream

import StdMisc, StdInt, StdChar

import bitstream2

class BitStream a
where
	getBits	:: a Int -> (Int, a)
	peekBits :: a Int -> (Int, a)
	skipBits :: a Int -> a
	getBitList	:: a [Int]	-> ([Int], a)

	toBitStream :: [Char] -> a
	fromBitStream :: a -> [Char]

instance BitStream BEBitStream
where
	getBits s n = getBEBits s n
	peekBits s n = peekBEBits s n
	skipBits s n = skipBEBits s n
	getBitList s n = getBEBitList s n

	toBitStream s = toBEBitStream s
	fromBitStream s = fromBEBitStream s

instance BitStream LEBitStream
where
	getBits s n = getLEBits s n
	peekBits s n = peekLEBits s n
	skipBits s n = skipLEBits s n
	getBitList s n = abort "<ReadableLEBitStream> getBitList: not implemented"

	toBitStream s = toLEBitStream s
	fromBitStream s = fromLEBitStream s


:: WritableBitStream :== [(Int, Int)]	// list of (nbits, bits)

putLEBits :: WritableBitStream -> [Char]
putLEBits [(nbits, bits) : stream]	= putMiddle nbits bits stream
where

	// putFirst is used when there are bits left from a previous word.
	// If no next word is available these final bits will be delivered. Otherwise,
	// the previous bits will be combined with the first bits of the next
	// word, and putMiddle will be called for the rest.
	
	putFirst nprev prev [(nbits, bits) : stream]
		= [toChar (prev bitor (bits >> remain)) : putMiddle remain bits stream]
		where
			remain = nbits - 8 + nprev
	putFirst nprev prev []
		= [toChar prev]
	
	// putMiddle is used when there are no bits left from a previous word.
	// It will extract 8 bits from a word for as long as this is possible.
	// If less than 8 bits remain, it will pass them to putFirst.
	// If zero bits remain, putBits can be called recursively.
	
	putMiddle nbits bits stream
	| nbits >= 8	= [toChar (bits >> remain) : putMiddle remain bits stream]
					with remain = nbits - 8
	| nbits > 0		= putFirst nbits (bits << (8 - nbits)) stream
					= putLEBits stream
					
putLEBits [] = []


