/*
 * Csizmazia Balazs
 *
 * Clean-CORBA extension
 *
 * It can parse a file containing interface specifications in an IDL-like
 * language.
 *
 * It generates the following output files for each interface defined in
 * the source interface description file:
 *
 *  - A header file (.h), that can be used with htoclean to produce the
 *    neccessary Clean mapping helper functions.
 *  - An adapter file, that acts as a converter between the functions
 *    called by Clean applications and the ORBit functions.
 *  - OPTIONALLY it can generate an IDL file, in case the syntax of the
 *    two files are different (or will be different)
 *
 */

%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "nodeType.h"

#define MAXNROFPARAMS 8192         /* Maxmimum number of parameters in a declaration */
#define MAXNROFINTERFACENAMES 8192

nodeType *construct_id(char *s);
nodeType *construct_onep(char *p_name,char *p_type); /* Its content shouldn't be deallocated! */
nodeType *construct_new_pali(nodeType *newparameter);
nodeType *merge_palis(nodeType *lista1, nodeType *lista2);
char *content_of_id(nodeType *p);
nodeType *construct_typenode(char *s);
char *content_of_typenode(nodeType *p);

void freeNode(nodeType *p);

/* CODE GENERATION PROTOTYPES */

void cg_if_init(void);
void cg_if_exit(void);
void cg_h_init(void);
void cg_h_exit(void);
void cg_module_init(char *modulnev);
void cg_module_exit(void);
void cg_interface_init(char *ifnev);
void cg_interface_exit(void);
void cg_functions_emit(char *nev,nodeType *params,char *rettypename);

/* Type conversion routines */

char *myidltype_to_c(char *name) ;
char *myidltype_to_c_CLEANTYPE(char *name,int returntype) ;
char *myidltype_to_c1(char *name) ;

/* Interface name aliases */

void make_new_interface_name_alias(char *name);
int is_interface_name_alias(char *name);

%}

%union {
  int  iValue;    /* Integer value -- not used in this application */
  char *sValue;   /* String value  on the parsing stack */
  nodeType *nPtr; /* Complex abstract syntax tree node */
};

%token MODULEUTASITAS
%token INTERFACEUTASITAS
%token EXCEPTIONUTASITAS
%token RAISESUTASITAS
%token FUNCTIONUTASITAS
%token INUTASITAS
%token RETURNSUTASITAS
%token STRINGUTASITAS
%token INT32UTASITAS
%token BOOLEANUTASITAS
%token IORUTASITAS
%token SYM_LPAREN
%token SYM_RPAREN
%token SYM_LBRACKET 
%token SYM_RBRACKET
%token SYM_COMMA
%token SYM_SEMICOLON
%token <iValue> SYM_NUMBER
%token SYM_ISMERETLEN
%token SYM_EOF
%token <sValue> SYM_IDENTIFIER

%type <nPtr> identifier
%type <nPtr> parameterek
%type <nPtr> parameter
%type <nPtr> typename

%%
/* The following part contains the syntax rules for the processed IDL file */

/* S <program> ::= <modul> <interfeszek> '}' ';' */
program: modul interfeszek SYM_RBRACKET SYM_SEMICOLON

/* S <modul> ::= 'module' <identifier> '{' */
modul: MODULEUTASITAS identifier SYM_LBRACKET {
         cg_module_init(content_of_id($2));
         freeNode($2);
       }

/* S <interfesz> ::= <interfesz1> <interfesz2> | <interfesz1a> */
interfesz: interfesz1 interfesz2
           | interfesz1a           /* Forward declaration */

/* S <interfesz1> ::= 'interface' <identifier> '{' */
interfesz1: INTERFACEUTASITAS identifier SYM_LBRACKET {
         cg_interface_init(content_of_id($2));
         if (!is_interface_name_alias(content_of_id($2))) {
           make_new_interface_name_alias(content_of_id($2));
         }
         cg_if_init();
         freeNode($2);
      }

/* S <interfesz1a> ::= 'interface' <identifier> ';' */
interfesz1a:  INTERFACEUTASITAS identifier SYM_SEMICOLON  { /* FORWARD */
         make_new_interface_name_alias(content_of_id($2));
         freeNode($2);
      }

/* S <interfesz2> ::= <definiciok> '}' ';' */
interfesz2: definiciok SYM_RBRACKET SYM_SEMICOLON {
         cg_if_exit();
         cg_interface_exit();
      }

/* S <interfeszek> ::= <interfesz> | <exception> | <interfesz> <interfeszek>
                       | <exception> <interfeszek> */
interfeszek: interfesz
          |  exception
          |  interfesz interfeszek
          |  exception interfeszek
          ;

/* S <exception> ::= 'exception' <identifier> '{' <exceptiontagok> '}' ';' */
exception: EXCEPTIONUTASITAS identifier SYM_LBRACKET exceptiontagok SYM_RBRACKET SYM_SEMICOLON {
           /* The content is ignored, only small syntax checks are done */
         }

/* S <exception> ::= 'exception' <identifier> '{' '}' ';' */
exception: EXCEPTIONUTASITAS identifier SYM_LBRACKET SYM_RBRACKET SYM_SEMICOLON {
           /* The content is ignored, only small syntax checks are done */
         }

/* S <exceptiontagok> ::= <exceptiontag> | <exceptiontag> <exceptiontagok> */
exceptiontagok: exceptiontag
           |  exceptiontag exceptiontagok
           ;

/* S <exceptiontag> ::= <typename> <identifier> ';' */
exceptiontag: typename identifier SYM_SEMICOLON {
               fprintf(stderr,"Exception member ignored.\n");
            } 
        ;

/* S <definiciok> ::= <definicio> | <definicio> <definiciok> */
definiciok: definicio 
          |  definicio definiciok
          ;

/* S <definicio:> ::= <typename> <identifier> '(' <parameterek> ')' <raisedexceptions> ';' */
/* S                | <typename> <identifier> '(' ')' <raisedexceptions> ';' */ 
definicio: typename identifier SYM_LPAREN parameterek SYM_RPAREN raisedexceptions SYM_SEMICOLON { 
               /* printf(" function name: %s  / return value type: STRING \n", content_of_id($2)); */
               sprintf(CURRENT_ID,"%s_%s_%s",modulename,interfacename,content_of_id($2));
               cg_functions_emit(CURRENT_ID,$4,content_of_typenode($1));
               freeNode($2);
             }
           | typename identifier SYM_LPAREN SYM_RPAREN raisedexceptions SYM_SEMICOLON { 
               sprintf(CURRENT_ID,"%s_%s_%s",modulename,interfacename,content_of_id($2));
               cg_functions_emit(CURRENT_ID,NULL,content_of_typenode($1));
               freeNode($2);
             }
   ;

/* S <raisedexceptions> ::= 'raises' '(' <exceptionnames> ')' */ 
raisedexceptions: RAISESUTASITAS SYM_LPAREN exceptionnames SYM_RPAREN  { 
               fprintf(stderr,"Exception names ignored.\n");
            } /* XXX */
               | { } ;
   ;


/* S <exceptionnames> ::= <identifier> | <identifier> ',' <exceptionnames> */
exceptionnames: identifier
            | identifier SYM_COMMA exceptionnames
           ;

/* S <parameterek> ::= <parameter> | <parameter> ',' <parameterek> */
parameterek: parameter { 
                         $$=construct_new_pali($1);
                         /*$$=construct_new_pali(construct_onep("alma","STRING"));*/
                         /*freeNode($1); */
                       }
   |  parameter SYM_COMMA parameterek {
                         $$=merge_palis(construct_new_pali($1),$3);
                         /* freeNode ?? XXX see merge_palis (wrapper needs to be freed) */
                         /*freeNode($1); */
                       }
   ;

/* S <parameter> ::= 'in' <typename> <identifier> */
parameter: INUTASITAS typename identifier { 
               /* printf(" parameter neve: %s \n", content_of_id($3)); */
               $$=construct_onep(content_of_id($3),content_of_typenode($2));
               freeNode($3);
               freeNode($2);
             }

/* S <typename> ::= 'string' | 'long' | 'ior' | 'boolean' | <identifier> */
typename: STRINGUTASITAS  { $$=construct_typenode("string"); }
        | INT32UTASITAS   { $$=construct_typenode("long"); }
        | IORUTASITAS     { $$=construct_typenode("ior"); }
        | BOOLEANUTASITAS { $$=construct_typenode("boolean"); }
        | identifier       { 
                            if (is_interface_name_alias(content_of_id($1)))
                            {
                              $$=construct_typenode("ior"); 
                            } else {
                              fprintf(stderr,"Identifier is not an interface name: %s\n",content_of_id($1));
                              $$=construct_typenode("unknown"); 
                            }
                          }

identifier: SYM_IDENTIFIER { $$=construct_id($1); }
   ;

%%
/* Here comes the main processing part */

#include <stdio.h>

extern FILE *yyin;
extern int num_lines;
char OUTPUT_FILE_NAME[1024]; /* should be POSIX_MAX_FILENAMELEN */
char OUTPUT_FILE_NAME2[1024]; /* should be POSIX_MAX_FILENAMELEN */
char CURRENT_ID[1024]; /* long enough to hold a module+interface ID */

/* Prints out an error message with line number, where the error occurred */
void yyerror(char *szoveg)
{
    printf("%s (line:%d)\n", szoveg, num_lines);
    fflush(stdout);
    exit(1);
}

/* main program */
main(int argc, char **argv)
{

   if (argc>=2)
   {
     if ((yyin=fopen(argv[1], "r"))==NULL)
     {
       yyin=stdin;
     }
   }
   else
   {
      yyin=stdin;
   }

   if (argc>=3)
   {
     fprintf(stderr,"Too many arguments\n");
     exit(-1);
   }
   
   yyparse(); /* Call the parser */

   cg_module_exit(); /* finish emitting the last module */
}

static nodeType *construct_con(int i)  /* Not used currently */
{
  nodeType *p;

  /* allocate node */
  if ((p = malloc(sizeof(conNodeType))) == NULL) 
       yyerror("out of memory");

  /* fill in the informations */
  p->type = typeCon;
  p->con.value = i;

  return p;
}

/* Constructs a node in the abstract syntax tree, which represents a data type */
nodeType *construct_typenode(char *s) 
{
  nodeType *p;

  /* allocate node */
  if ((p = malloc(sizeof(typeNodeType))) == NULL) 
       yyerror("out of memory");

  /* fill in the informations */
  p->type = typeTypeName;
  p->tp.name = malloc(strlen(s)+1);
  strcpy(p->tp.name,s);

  return p;
}

/* Returns the content of a type node which is a data type */
char *content_of_typenode(nodeType *p) 
{
  if (!p) return NULL; /* something is wring */
  if (p->type == typeTypeName) {
    return strdup(p->tp.name); 
  } else return NULL;

}

/* Constructs a node in the abstract syntax tree, which represents an identifier */
nodeType *construct_id(char *s) 
{
  nodeType *p;

  /* allocate node */
  if ((p = malloc(sizeof(idNodeType))) == NULL) 
       yyerror("out of memory");

  /* fill in the informations */
  p->type = typeId;
  p->id.name = malloc(strlen(s)+1);
  strcpy(p->id.name,s);

  return p;
}

/* Returns the content of a type node which is an identifier */
char *content_of_id(nodeType *p) 
{
  if (!p) return; /* valami disznosag van */
  if (p->type == typeId) {
    return strdup(p->id.name); 
  } else return NULL;

}

/* Constructs a node in the abstract syntax tree, which represents one parameter */
/* Later these can be merged into longer parameter lists */
nodeType *construct_onep(char *p_name,char *p_type) 
{
  nodeType *p;

  /* allocate node */
  if ((p = malloc(sizeof(onepNodeType))) == NULL) 
       yyerror("out of memory");

  /* fill in the informations */
  p->type = typeOneParam;
  p->onep.parname = malloc(strlen(p_name)+1);
  p->onep.partype = malloc(strlen(p_type)+1);
  strcpy(p->onep.parname,p_name);
  strcpy(p->onep.partype,p_type);

  return p;
}

/* Constructs a node in the abstract syntax tree, which represents a parameter list */
nodeType *construct_new_pali(nodeType *newparameter)
{
  nodeType *p;

  if (newparameter->type != typeOneParam) {
	fprintf(stderr,"ASSERTION FAILED: construct_new_pali got a parameter, which is not the type with one parameter.\n");
	exit(-1);
  } /* ASSERT */

  /* allocate node */
  if ((p = malloc(sizeof(paliNodeType))) == NULL) 
       yyerror("out of memory");

  /* fill in the informations */
  p->type = typeParamList;
  p->pali.howmanyparameters = 1;
  p->pali.palist = malloc(sizeof(onepNodeType)*1); /* XXX */
  p->pali.palist[0]=(onepNodeType*) newparameter;

  return p;
}

/* Merge two parameter lists, creates a new list object */
nodeType *merge_palis(nodeType *lista1, nodeType *lista2)
{
  nodeType *p;
  onepNodeType **o;
  int i;

  if ((lista1->type != typeParamList) || (lista2->type != typeParamList)) {
    fprintf(stderr,"ASSERTION FAILED: merge_palis got a parameter, which is not a parameter list itself\n");
    exit(-1);
  } /* ASSERT */

  /* allocate node */
  if ((p = malloc(sizeof(paliNodeType))) == NULL) 
       yyerror("out of memory");

  /* fill in the informations */
  p->type = typeParamList;
  p->pali.howmanyparameters = lista1->pali.howmanyparameters + lista2->pali.howmanyparameters;
  p->pali.palist = malloc(sizeof(onepNodeType*)*p->pali.howmanyparameters); /* XXX */
  o = p->pali.palist ;
#ifdef DEBUG
  /* First print out everything */
  for (i=0;i<lista1->pali.howmanyparameters;i++)
  {
    fprintf(stderr,".%d:[%d]",i,lista1->pali.palist[i]->type);
    fprintf(stderr,"%s.",((onepNodeType *)(lista1->pali.palist[i]))->parname);
  }
  for (i=0;i<lista2->pali.howmanyparameters;i++)
  {
    fprintf(stderr,".%d(%d):[%d].",i+lista1->pali.howmanyparameters,i,lista2->pali.palist[i]->type);
    fprintf(stderr,"%s.",((onepNodeType *)(lista2->pali.palist[i]))->parname);
  }
#endif
  /* Now concatenate the lists */
  for (i=0;i<lista1->pali.howmanyparameters;i++)
    o[i]=lista1->pali.palist[i];
  for (i=0;i<lista2->pali.howmanyparameters;i++)
    o[i+lista1->pali.howmanyparameters]=lista2->pali.palist[i];
  /* XXX !!! lista1 and lista2 should be freed, but not the pointers (elements) */
  /* , only the "wrapper" */

  return p;
}

/* Free the memory allocated by the given node */
void freeNode(nodeType *p)
{
  int i;

  /* do the work according to the node's type */
  if (!p) return; /* something is wring */
  if (p->type == typeId) {
    free(p->id.name); /* this was malloc'd separately */
  }
  if (p->type == typeOneParam) {
    free(p->onep.parname); /* this was malloc'd separately */
    free(p->onep.partype); /* this was malloc'd separately */
  }
  if (p->type == typeParamList) {
    for (i=0;i<p->pali.howmanyparameters;i++)
      {
	if (p->pali.palist[i]->type != typeOneParam) {
		fprintf(stderr,"ASSERTION FAILED: in freeNode, while freeing parameter lists");
		exit(-1);
	} /* ASSERT */
        freeNode((nodeType *)(p->pali.palist[i]));
      }
    free(p->pali.palist);  /* this was malloc'd separately */
  }
  free(p);
}

/* ------------------------------------------- CODE GENERATION --------------------------------- */
/* This part is responsible for generating the CORBA IDL and other stuff */

#ifdef GENERATE_IDL
FILE *idlfile; /* IDL file produced for orbit-idl */
#endif
FILE *hfile;   /* header file produced for htoclean */
FILE *adapterfile; /* adapter file produced for htoclean */
#define MAXNAMELEN 8192
char modulename[MAXNAMELEN];
char interfacename[MAXNAMELEN];

#define NEST2 all_spaces(2)
#define NEST4 all_spaces(4)

/* returns 'n' spaces */
char *all_spaces(int n) /* The returned value is STATIC!!! */
{
  int i;
  char *s;

  s=malloc(n+1);
  for (i=0;i<n;i++) {
    s[i]=' ';
  }
  s[n]='\0';
  return(s); /* XXX needs to be freed */
}

/* Initialize code generator for the 'h' file */
void cg_h_init(void)
{
  strcpy(OUTPUT_FILE_NAME,OUTPUT_FILE_NAME2);
  hfile=fopen(strcat(OUTPUT_FILE_NAME,".h"),"w");
}

/* Initialize code generator for the by-interface file */
/* It initializes the 'h' file (calls the corresp. func.) and the adapter file */
void cg_if_init(void)
{
  /* Create file name from module name + interface name */
  sprintf(OUTPUT_FILE_NAME,"%s_%s",modulename,interfacename);
  /* Now do the processing */
  strcpy(OUTPUT_FILE_NAME2,OUTPUT_FILE_NAME);
#ifdef GENERATE_IDL
  idlfile=fopen(strcat(OUTPUT_FILE_NAME,".idl"),/*"output.idl",*/ "w");
  fprintf(idlfile,"module %s {\n",modulename); /* ! Needs end of module mark */
  fprintf(idlfile,"%sinterface %s {\n",NEST2,interfacename); /* ! Needs end of interface mark */
#endif
  cg_h_init();
  strcpy(OUTPUT_FILE_NAME,OUTPUT_FILE_NAME2);
  adapterfile=fopen(strcat(OUTPUT_FILE_NAME,"_adapter.c"),"w");
  fprintf(adapterfile," #include %cClean.h%c  \n",'<','>'); 
  fprintf(adapterfile," #include <stdlib.h>  \n"); 
  fprintf(adapterfile," #include <stdio.h>  \n"); 
  fprintf(adapterfile," #include <orb/orbit.h>  \n"); 
  fprintf(adapterfile," #include %c%s.h%c  \n",'"',modulename,'"'); 
  fprintf(adapterfile," \n"); 
  fprintf(adapterfile," static CleanString common_clean_return_value_string;  \n"); 
  fprintf(adapterfile," static int ccrvs_is_mallocated = 0;  \n"); 
  fprintf(adapterfile," static int common_clean_return_value_integer;  \n"); 
  fprintf(adapterfile," static char* NYEMA = %cNYEMA%c;  \n",'"','"'); 
  fprintf(adapterfile," "); 
  fprintf(adapterfile," extern void * __nl_kun_cs_my_orb;\n"); 
  fprintf(adapterfile," extern CORBA_exception_type __nl_kun_cs_lastop_exception;\n"); 
  fprintf(adapterfile," extern char *__nl_kun_cs_lastop_excname;"); 
  fprintf(adapterfile," "); 
  /*fprintf(hfile," #include %cClean.h%c  \n",'"','"'); */
  /*fprintf(hfile," Clean ( :: *State :== Int )   \n"); */
}

/* Flush and exit code generator for the 'h' file */
void cg_h_exit(void)
{
  fclose(hfile);
}

/* Flush ans exit the by-interface files ('h' and adapter) */
void cg_if_exit(void)
{
  fclose(adapterfile);
  cg_h_exit();
#ifdef GENERATE_IDL
  fprintf(idlfile,"%s};\n",NEST2);
  fprintf(idlfile,"};\n"); 
  fclose(idlfile);
#endif
}

/* Code generation, module, initialization */
void cg_module_init(char *modulnev)
{
  strcpy(modulename,modulnev);
}

/* Code generation, module, finalization */
void cg_module_exit(void)
{
}

/* Code generation, interface, initialization */
void cg_interface_init(char *ifnev)
{
  strcpy(interfacename,ifnev);
}

/* Code generation, interface, finalization */
void cg_interface_exit(void)
{
}

/* Generates the code (into then h,adapter,idl file) or a function */
void cg_functions_emit(char *nev,nodeType *params, char *rettypename) 
{
  onepNodeType *o;
  int i;
  int returntypeisstring;
  char CleanTypeString[8192];
  char TEMPCleanTypeString[8192];
 
  if (params != NULL) {
			if (params->type != typeParamList) {
				fprintf(stderr,"ASSERTION FAILED: cg_functions_emit got a parameter, which is not a parameter list itself");
				exit(-1);
				} /* ASSERT */
  }

#ifdef GENERATE_IDL
  fprintf(idlfile,"%s",NEST4); 
#endif

#ifdef GENERATE_IDL
  if (strcmp(rettypename,"ior") == 0) {
    fprintf(idlfile,"  string %s ( ", nev); 
  } else { 
    fprintf(idlfile,"  %s %s ( ", rettypename, nev); 
  }
#endif


  if (params != NULL) {
    for (i=0;i<params->pali.howmanyparameters;i++)
    {
#ifdef GENERATE_IDL
      if (i != 0) fprintf(idlfile," , "); 
#endif
		  if (params->pali.palist[i]->type != typeOneParam) {
			  fprintf(stderr,"ASSERTION FAILED: got a parameter which is not a parameter\n");
                  }
      o=params->pali.palist[i];
#ifdef GENERATE_IDL
      fprintf(idlfile," in %s %s ",((onepNodeType *)o)->partype,((onepNodeType *)o)->parname); 
#endif
    }
  }

#ifdef GENERATE_IDL
  fprintf(idlfile," ) ;"); 
  fprintf(idlfile,"\n"); 
#endif

  /* Now generate the C header file  */

  sprintf(CleanTypeString,"Clean( CORBA_%s :: ",nev);
  returntypeisstring=0;
  if ((strcmp(rettypename,"string") != 0) && (strcmp(rettypename,"ior") != 0)) {
    fprintf(hfile,"  %s ", myidltype_to_c(rettypename)); 
  } else {
    fprintf(hfile,"  void "); 
    returntypeisstring=1;
  }
  fprintf(hfile,"  CORBA_%s ( ", nev); 

  /* -- Now the object reference -- */
  /* According to the htoclean specs, its compatible with an int */
  fprintf(hfile," int CORBA_IOR "); 
  sprintf(TEMPCleanTypeString," Int ");
  strcat(CleanTypeString,TEMPCleanTypeString);
  /* -- */
 
  if (params != NULL) {
    for (i=0;i<params->pali.howmanyparameters;i++)
    {
      fprintf(hfile," , "); 
      if (params->pali.palist[i]->type != typeOneParam) {
        fprintf(stderr,"ASSERTION FAILED: got a parameter which is not a parameter (1)\n");
      }
      o=params->pali.palist[i];
      fprintf(hfile," %s %s ",myidltype_to_c(((onepNodeType *)o)->partype),((onepNodeType *)o)->parname); 
      sprintf(TEMPCleanTypeString," %s ",myidltype_to_c_CLEANTYPE( ((onepNodeType *)o)->partype  , 0  ));
      strcat(CleanTypeString,TEMPCleanTypeString);
    }
  }

  /* Doesn't changes the variable i !!! */
  if (returntypeisstring==1) {
    if (i != 0) fprintf(hfile," , "); 
    fprintf(hfile," CleanString * ret_str_val "); 
  }

  fprintf(hfile," ) ;"); 
  fprintf(hfile,"\n"); 

  sprintf(TEMPCleanTypeString," Int -> ( %s , Int ) ) \n",myidltype_to_c_CLEANTYPE(rettypename,1));
  strcat(CleanTypeString,TEMPCleanTypeString);
  fprintf(hfile,"%s\n",CleanTypeString); 


  /* And now generate the adapter file */
  /* the role of an adapter file is to be callable by Clean applications,
   * and then it should call the CORBA client stubs
   */


  /*fprintf(adapterfile,"  %s CORBA_%s ( ", myidltype_to_c(rettypename), nev); */
  if ((strcmp(rettypename,"string") != 0) && (strcmp(rettypename,"ior") != 0)) {
    fprintf(adapterfile,"  %s ", myidltype_to_c(rettypename)); 
  } else {
    fprintf(adapterfile,"  void "); 
  }
  fprintf(adapterfile,"  CORBA_%s ( ", nev); 

  /* -- Now the object reference -- */
  /* According to the htoclean specs, its compatible with an int */
  fprintf(adapterfile," int CORBA_IOR ");  
  /*fprintf(adapterfile," void* CORBA_IOR ");  */ /* XXX */
  /* -- */

  if (params != NULL) {
    for (i=0;i<params->pali.howmanyparameters;i++)
    {
      /* if (i != 0) */ fprintf(adapterfile," , "); 
		  if (params->pali.palist[i]->type != typeOneParam) {
			  fprintf(stderr,"ASSERTION FAILED: got a parameter which is not a parameter (1)\n");
                  }
      o=params->pali.palist[i];
      fprintf(adapterfile," %s %s ",myidltype_to_c(((onepNodeType *)o)->partype),((onepNodeType *)o)->parname); 
    }
  }
  /* Dont touch the variable i !!! */
  if (returntypeisstring==1) {
    if (i != 0) fprintf(adapterfile," , "); 
    fprintf(adapterfile," CleanString * ret_str_val "); 
  }

  fprintf(adapterfile," ) "); 
  fprintf(adapterfile,"\n"); 
  fprintf(adapterfile,"{\n"); 

  if (params != NULL) {
    for (i=0;i<params->pali.howmanyparameters;i++)
    {
      o=params->pali.palist[i];
      fprintf(adapterfile," %s par_%d;\n",myidltype_to_c1(((onepNodeType *)o)->partype),i); 
    }
  }
  fprintf(adapterfile," CORBA_Environment ev,evc;\n"); 
  fprintf(adapterfile," CORBA_char *rcc=NULL; /* to be able to free the resulting objref */\n"); 


  if ((strcmp(rettypename,"string") == 0) || (strcmp(rettypename,"ior") == 0)) {
    fprintf(adapterfile," char* string_return_value; unsigned int string_v_e_len ;\n");
  } else {
    fprintf(adapterfile," int common_clean_return_value_integer ; \n");
  }

  /* Clean to C Conversion */

  if (params != NULL) {
    for (i=0;i<params->pali.howmanyparameters;i++)
    {
      char parname[8192];

      o=params->pali.palist[i];
      sprintf(parname,"par_%d",i); 
      if (strcmp( (((onepNodeType *)o)->partype) , "string" ) == 0) {
        fprintf(adapterfile," %s = calloc(1,CleanStringLength(",parname); 
        fprintf(adapterfile," %s)+1); \n",((onepNodeType *)o)->parname); 
        fprintf(adapterfile," strncpy(%s,CleanStringCharacters(%s),CleanStringLength(%s)); \n", parname,
                                   ((onepNodeType *)o)->parname,
                                   ((onepNodeType *)o)->parname);
#ifdef DEBUG_VALUES
      fprintf(adapterfile," printf(%c%c%c%c,%s); \n",'"','%','s','"',parname); 
#endif
      }
      if (strcmp( (((onepNodeType *)o)->partype) , "long" ) == 0) {
        fprintf(adapterfile," %s = %s; \n",parname,((onepNodeType *)o)->parname); 
      }
      if (strcmp( (((onepNodeType *)o)->partype) , "boolean" ) == 0) {
        fprintf(adapterfile," %s = %s; \n",parname,((onepNodeType *)o)->parname); 
      }
      if (strcmp( (((onepNodeType *)o)->partype) , "ior" ) == 0) {
        fprintf(adapterfile," %s = %s; \n",parname,((onepNodeType *)o)->parname); 
      }
    }
  }

    fprintf(adapterfile," CORBA_exception_init(&ev);\n ");
    fprintf(adapterfile," CORBA_exception_init(&evc);\n ");

   /* Call of a CORBA method */
  
   /* Beginning of an assignment (generated) */
 
   if ((strcmp(rettypename,"string") == 0) || (strcmp(rettypename,"ior") == 0)) {
    fprintf(adapterfile," string_return_value = ");
   } else {
    fprintf(adapterfile," common_clean_return_value_integer = ");
   }
   if ((strcmp(rettypename,"ior") == 0)) {
    fprintf(adapterfile,"rcc=CORBA_ORB_object_to_string(__nl_kun_cs_my_orb,");
   }
 
   /* Right side of an assignment */

   fprintf(adapterfile,"  %s ( (void *) CORBA_IOR ",  nev); 

  if (params != NULL) {
    for (i=0;i<params->pali.howmanyparameters;i++)
    {
      fprintf(adapterfile,", par_%d ",i); 
    }
  }
  fprintf(adapterfile,", &ev) "); 

  if ((strcmp(rettypename,"ior") == 0)) {
    fprintf(adapterfile,", &evc)");
  }
  fprintf(adapterfile,";\n");

  fprintf(adapterfile,"  __nl_kun_cs_lastop_exception = ev._major ;\n");
  /* If there wasm't an exception, then the exception string variable is UNDEFINED */
   fprintf(adapterfile,"  if (ev._major != CORBA_NO_EXCEPTION) {\n"); 
   fprintf(adapterfile,"    if (__nl_kun_cs_lastop_excname != NULL) {\n"); 
   fprintf(adapterfile,"      CORBA_free(__nl_kun_cs_lastop_excname);\n"); 
   fprintf(adapterfile,"    }\n"); 
   fprintf(adapterfile,"    __nl_kun_cs_lastop_excname=CORBA_exception_id(&ev);;\n"); 
   fprintf(adapterfile,"  }\n"); 

   fprintf(adapterfile,"if (ev._major == CORBA_NO_EXCEPTION) {\n"); /* If ther wasn't any exception */
     if ((strcmp(rettypename,"string") == 0) || (strcmp(rettypename,"ior") == 0)) {
       fprintf(adapterfile," string_v_e_len = strlen(string_return_value);\n");
     } else {
     }
   fprintf(adapterfile,"}\n"); /* If ther wasn't any exception */

   /* RETURN VALUE */

   if ((strcmp(rettypename,"string") == 0) || (strcmp(rettypename,"ior") == 0)) {
     fprintf(adapterfile,"if (ev._major != CORBA_NO_EXCEPTION) {\n"); /* If ther wasn't any exception */
     fprintf(adapterfile,"  string_return_value=NYEMA; /* Failure */ \n");
     fprintf(adapterfile,"  string_v_e_len = strlen(string_return_value);\n");
     fprintf(adapterfile,"  /* THOU SHALT NOT FREE NYEMA! */ \n");
     fprintf(adapterfile,"}\n"); /* If ther wasn't any exception */
    /* Now generate the return of the string value */
    fprintf(adapterfile," /* result is in : string_return_value, string_v_e_len */ \n");
    fprintf(adapterfile,"if (ccrvs_is_mallocated) {\n free(common_clean_return_value_string); 
}\n");
    fprintf(adapterfile," common_clean_return_value_string=malloc(sizeof(int)+1+string_v_e_len);\n"); 

    fprintf(adapterfile," memcpy((char*)(1+(unsigned int*)common_clean_return_value_string),string_return_value,string_v_e_len);");
    fprintf(adapterfile," ccrvs_is_mallocated = 1;  \n"); 
    fprintf(adapterfile," *((unsigned int *)(common_clean_return_value_string)) = string_v_e_len;  \n"); 
    fprintf(adapterfile," *ret_str_val = common_clean_return_value_string;  \n"); 
   } else {
    fprintf(adapterfile," /* result is in : common_clean_return_value_integer */ \n");
    fprintf(adapterfile,"if (ev._major != CORBA_NO_EXCEPTION) {\n"); /* If ther wasn't any exception */
    fprintf(adapterfile," common_clean_return_value_integer=0;  /* Failure */ \n"); 
    fprintf(adapterfile,"}\n"); /* If ther wasn't any exception */
    fprintf(adapterfile," return common_clean_return_value_integer;  \n"); 
   }
  

   if ((strcmp(rettypename,"ior") == 0)) {
    fprintf(adapterfile,"if (rcc != NULL) { CORBA_free(rcc); rcc=NULL; }");
   }
  fprintf(adapterfile,"}\n"); 
  fprintf(adapterfile,"\n"); 

}

/* ----------------------------------------------------------- */
/* Type conversion routines                                    */

char *myidltype_to_c(char *name) 
{
  if (strcmp(name,"string")==0) return("CleanString"); /* * for the mode "in" */
  if (strcmp(name,"ior")==0) return("int"); /* * compatible with CORBA_Object */
  if (strcmp(name,"long")==0) return("int");
  if (strcmp(name,"boolean")==0) return("int");
  return("????UNKNOWN TYPE???");
}

char *myidltype_to_c_CLEANTYPE(char *name, int returntype) 
{
  if (strcmp(name,"string")==0) return("String"); /* * for the mode "in" */
  if (strcmp(name,"ior")==0) 
  {
    if (returntype) {
      return("String"); /* * for the mode "in"  -- was "String" csb 99jun13*/
    } else {
      return("Int"); /* * for the mode "in" */
    }
  }
  if (strcmp(name,"long")==0) return("Int");
  if (strcmp(name,"boolean")==0) return("Int");
  return("????UNKNOWN TYPE???");
}

char *myidltype_to_c1(char *name) 
{
  if (strcmp(name,"string")==0) return("char *"); 
  if (strcmp(name,"ior")==0) return("char *"); 
  if (strcmp(name,"long")==0) return("int");
  if (strcmp(name,"boolean")==0) return("int");
  return("????UNKNOWN TYPE???");
}

/* It can store names of interfaces, so that these can be recognised */
/* later as IOR (interoperable reference) objects                    */

char **interface_name_aliases;
static int nr_of_interface_aliases=0;
static int is_iftable_initialised=0;

void make_new_interface_name_alias(char *name)
{
  if (!is_iftable_initialised) {
    interface_name_aliases=malloc(sizeof(char *) * MAXNROFINTERFACENAMES);
    is_iftable_initialised=1;
  }
  interface_name_aliases[nr_of_interface_aliases]=malloc(strlen(name)+1);
  strcpy(interface_name_aliases[nr_of_interface_aliases],name);
  nr_of_interface_aliases++;
}

int is_interface_name_alias(char *name)
{
  int i;

  for (i=0;i<nr_of_interface_aliases;i++) 
    if (strcmp(interface_name_aliases[i],name)==0) return 1;
  return 0;
}
