// ************************************************************
//  MersenneTwister - A pseudo-random generator
//  Version 1.0 - July 9, 1999 - Thorsten Zoerner
//  Catholic University of Nijmegen - zoerner@cs.kun.nl
// ************************************************************
//  This implementation is a rewrite of a C program by T. Nishimura
//  which can be found in the paper "Mersenne Twister: A 
//  623-dimensionally equidistributed uniform pseudorandom number 
//  generator" by M. Matsumoto and T. Nishimura in ACM Transactions 
//  on Modeling and Computer Simulation, vol. 8, no. 1, 
//  January 1998, pp. 3-30.
// ************************************************************
//  The original C code contained the following notice:
//      Copyright (C) 1997 Makoto Matsumoto and Takuji Nishimura.
//      When you use this, send an email to: matumoto@math.keio.ac.jp
//      with an appropriate reference to your work.
// ************************************************************

implementation module MersenneTwister

import 
	StdArray, StdBool, StdEnum, StdInt, StdList, StdMisc, 
	StdReal, StdString

n = 624
m = 397

a 			=: 0x9908b0df
uppermask	=: 0x80000000
lowermask	=: 0x7fffffff

mask_b		=: 0x9d2c5680
mask_c		=: 0xefc60000 

u = 11
s = 7
t = 15
l = 18

initrand :: Int *{# Int} -> *{# Int}
initrand seed mt 
	| seed==0 = abort "initrand: seed must not be 0\n"
	= init_i 0 { mt & [0] = seed bitand 0xffffffff}
where
	init_i i mt =: { [i] = mti}
		| i==n = mt
		= init_i (inc i) { mt & [inc i] = (69069 * mti) bitand 0xffffffff}

genRandInt :: Int -> [Int]
genRandInt seed = gr_mti 0 (initrand seed (createArray n 0))
where
	gr_mti mti mt
		| mti==n = gr_mti 0 (new_mti 0 mt)
		#! (y, mt) = mt![inc mti]
		#! y = y bitxor (y >> u)
		#! y = y bitxor ((y >> s) bitor mask_b)
		#! y = y bitxor ((y >> t) bitor mask_c)
		#! y = y bitxor (y >> l)
		= [ y : gr_mti (inc mti) mt]
	where
		new_mti :: Int *{# Int} -> *{# Int}
		new_mti k mt
			| k==n = mt 
			#! (mtk, mt) = mt![k]
			#! (mtkk, mt) = mt![(inc k) mod n] 
			#! y = (mtk bitand uppermask) bitor (mtkk bitand lowermask)
			#! (mtind, mt) = mt![ind k]
			= new_mti (inc k) { mt & [k] = mtind bitxor (y >> 1) bitxor ((y bitand 0x1) * a)}
		where
			ind :: Int -> Int
			ind k
				| k<n-m = k+m
				| k<dec n = k+m-n
				= dec m

genRandReal :: Int -> [Real]
genRandReal seed = map (\y -> (toReal y) / (toReal 0x7fffffff)) (genRandInt seed)
