/* This file is written by Martin Wierich (martinw@cs.kun.nl) on 15.5.2000.
   It is "work based on the GMP library" (under terms of the GNU Library General
   Public License) because it contains source code from that library, so:

This file is work based on the GNU MP Library.

The GNU MP Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Library General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

The GNU MP Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
License for more details.

You should have received a copy of the GNU Library General Public License
along with the GNU MP Library; see the file COPYING.LIB.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA. */

#include "gmp.h"
#include "gmp-impl.h"
#include "longlong.h"

int get_max_string_sizeC(unsigned int base, unsigned int msize) {
      return (int) ( msize * BITS_PER_MP_LIMB
		 * __mp_bases[base].chars_per_bit_exactly) + 3;
	}

void add_1C (register mp_ptr res_ptr,
	   register mp_srcptr s1_ptr,
	   register mp_size_t s1_size,
	   register mp_limb_t s2_limb)
{
  register mp_limb_t x;
  mp_ptr res_ptr_save;
  
  res_ptr_save = res_ptr;
  x = *s1_ptr++;
  s2_limb = x + s2_limb;
  *res_ptr++ = s2_limb;
  if (s2_limb < x)
    {
      while (--s1_size != 0)
	{
	  x = *s1_ptr++ + 1;
	  *res_ptr++ = x;
	  if (x != 0)
	    goto fin;
	}
		*res_ptr = 1;
		return;
    }

 fin:
  if (res_ptr != s1_ptr)
    {
      mp_size_t i;
      for (i = 0; i < s1_size - 1; i++)
	res_ptr[i] = s1_ptr[i];
    }
  (res_ptr_save[-2])--;
}

///////////////////////////////////////////////////////////////////////////////

#include "Clean.h"

#define true (!0)
#define false 0

#define CleanArraySize(x) x[-2]
#define errorMessage "Error in module BigInt: a Clean function allocated unsufficient memory before calling GMP"

#define get_mp_size(sign, limbs, size)	((size==1 && limbs[0]==0) ? 0 : (sign>0 ? size : (-size)))

int allocationIsDefault=true;

void setSizeC(unsigned int* array, int newSize)
{
 	CleanArraySize(array)	= newSize;
}

void setStringSizeC(CleanString cs, int newSize)
{
 	CleanStringLength(cs)	= newSize;
}

void countLeadingZeros(unsigned int word, int *pResult)
// flip arguments for interfacing with Clean
{
        count_leading_zeros(*pResult, word);
}

int set_strC(mp_limb_t *R1P, CleanString digits, int base)
{
	return mpn_set_str(R1P, (const unsigned char *) CleanStringCharacters(digits),
	                   CleanStringLength(digits),base);
}

void init_set_limbs(mpz_ptr p_gmp_var, int sign, mp_limb_t *limbs, int size)
{
	p_gmp_var->_mp_alloc	= size;
	p_gmp_var->_mp_size		= get_mp_size(sign, limbs, size);
	p_gmp_var->_mp_d		= limbs;
}

void *abort1(size_t x)
{
	perror(errorMessage);
	abort();
}

void *abort2(void *x, size_t y, size_t z)
{
	perror(errorMessage);
	abort();
}

void abort3(void *x, size_t y)
{
	perror(errorMessage);
	abort();
}


inline void disableDefaultAllocation()
{
	if (allocationIsDefault) {
		mp_set_memory_functions(abort1,abort2,abort3);
		allocationIsDefault	= false;
		};
}

inline void enableDefaultAllocation()
{
	if (!allocationIsDefault) {
		mp_set_memory_functions(NULL,NULL,NULL);
		allocationIsDefault	= true;
		};
}

void remainderC(mp_limb_t *nomLimbs, int nomSize, mp_limb_t *denomLimbs, int denomSize)
{
	mpn_divrem(nomLimbs+denomSize, 0, nomLimbs, nomSize, denomLimbs, denomSize);
}

int powmC(mp_limb_t *destLimbs	, int baseSign		, mp_limb_t *baseLimbs
									, int exponentSign	, mp_limb_t *exponentLimbs
									, int modulusSign	, mp_limb_t *modulusLimbs)
{
	mpz_t	dest, base, exponent, modulus;
	init_set_limbs(dest		, true			, destLimbs		, CleanArraySize(destLimbs));
	init_set_limbs(base		, baseSign		, baseLimbs		, CleanArraySize(baseLimbs));
	init_set_limbs(exponent	, exponentSign	, exponentLimbs	, CleanArraySize(exponentLimbs));
	init_set_limbs(modulus	, modulusSign	, modulusLimbs	, CleanArraySize(modulusLimbs));
	disableDefaultAllocation();
	mpz_powm(dest, base, exponent, modulus);
	return dest->_mp_size;
}

int gmp_gcdC(mp_limb_t *destLimbs	, int uSign	, mp_limb_t *uLimbs
									, int vSign	, mp_limb_t *vLimbs)
{
	mpz_t	dest, u, v;
	init_set_limbs(dest		, true			, destLimbs		, CleanArraySize(destLimbs));
	init_set_limbs(u		, uSign			, uLimbs		, CleanArraySize(uLimbs));
	init_set_limbs(v		, vSign			, vLimbs		, CleanArraySize(vLimbs));
	disableDefaultAllocation();
	mpz_gcd(dest, u, v);
	return dest->_mp_size;
}

void pow_uiC(int baseSign, mp_limb_t *baseLimbs, unsigned long int exp,
			int *pGMPSize, mpz_t **pResultAdr)
{
	static mpz_t power_result;
	mpz_t	base;
	init_set_limbs(base		, baseSign		, baseLimbs		, CleanArraySize(baseLimbs));
	enableDefaultAllocation();
	mpz_init(power_result);
	mpz_pow_ui(power_result, base, exp);
	*pGMPSize	= power_result->_mp_size;
	*pResultAdr	= &power_result;
}

void transferGMPNumberToCleanC(mp_limb_t *dest, mpz_t *mpz_src, int nr_of_words)
{
	mp_limb_t *src;
	src = (*mpz_src)->_mp_d;
	my_memcpy(dest, src, nr_of_words<<2);
	mpz_clear(*mpz_src);
}

size_t
call_mpn_get_str (int offset, CleanString str, int base, mp_ptr mptr, mp_size_t msize)
{
	return mpn_get_str (CleanStringCharacters(str)+offset, base, mptr, msize);
}

double bigIntToRealC(int sign	, mp_limb_t *limbs) {
	mpz_t	i;

	init_set_limbs(i, sign, limbs, CleanArraySize(limbs));
	disableDefaultAllocation();
	return mpz_get_d(i);
	}

double ratioToRealC(int nomSign	, mp_limb_t *nomLimbs, int denomSign	, mp_limb_t *denomLimbs) {
	mpq_t	q;

	init_set_limbs(&(q->_mp_num), nomSign		, nomLimbs		, CleanArraySize(nomLimbs));
	init_set_limbs(&(q->_mp_den), denomSign		, denomLimbs	, CleanArraySize(denomLimbs));
	disableDefaultAllocation();
	return mpq_get_d(q);
	}
