implementation module basicDB

import StdEnv
import deltaEventIO, deltaDialog, deltaIOSystem, deltaMenu, deltaWindow, deltaFont
import deltaPicture, deltaIOState, deltaFileSelect, deltaControls, deltaSystem
import listextensions

::	InfoFont	=	{ font		:: Font					// The font which is used
					, width		:: Int					// Its widest character
					, height	:: Int					// Its line height
					}

MinDbDomainSize	::	(Int,Int)
MinDbDomainSize =	(100,1)								// Minimal size of recordwindow
CharsInInputBox :== 20									// Input width (number of characters)
InputBoxWidth	::	Measure
InputBoxWidth   =	Pixel (CharsInInputBox*DfFont.width)// Width of boxes in fields, queries and field names

DontCareId		::	Int
DontCareId      =	0
editinfoid		::	Int
editinfoid		=	5773
RecordWindowId	::	Int
RecordWindowId	=	0									// Id of window in which the records are shown 
EdDialogId		::	Int
EdDialogId		=	0
FieldDialogId	::	Int
FieldDialogId	=	1									// Ids of main dialogs used
FirstEditField	::	Int
FirstEditField	=	0

Replace			:== True								// Replace current selection when adding new record
New				:== not Replace
Separator		:==	": "								// Separates field names and contents

DbFont :: InfoFont
DbFont =: {font = f, width = maxwidth, height = ascent+descent+leading} 
where													// Global graph def: font used in this database
	(ascent,descent,maxwidth,leading)	= FontMetrics f
	(_,f)								= SelectFont "courier" [] 10

DfFont :: InfoFont
DfFont =: {font = f, width = maxwidth, height = ascent+descent+leading}
where													// Global graph def: default font (in dialogs)
	(ascent,descent,maxwidth,leading)	= FontMetrics f
	(_,f)								= SelectFont name styles length
	(name,styles,length)				= DefaultFont

::	*IO				:==	IOState *DataBase		 		// Synonym for IOState (see deltaEventIO)
::	DataBase		=	{old::State,current::State}		// State contains all relevant info
::  Descriptor    	:==	[AttDescriptor]
::	AttDescriptor	=	AttDesc AttributeName TypeCode
::	AttributeName	:==	String
::  TypeCode        =	STRING
::	Table  		  	:==	[ Record ]
::	Record		  	:==	[ AttributeValue ]
::	AttributeValue  =	AS String

::	State		=	{ records		:: Table			// All records
					, descriptor	:: Descriptor		// All fieldnames
					, selection		:: Int				// Indicating current record selected
					, query			:: Record			// Record to look for
					, name			:: String			// Name of table
					, fw			:: Int				// Max width of field contents
					, dw			:: Int				// Max width of descriptor fields
					}

ShowEditDialog :: *DataBase IO -> (*DataBase, IO)
ShowEditDialog db=:{current={descriptor,records,selection}} io 
 #	io	= OpenDialog editDialog	io
	io	= SetTextFields infostring descriptor (if (isEmpty records) [] (records!!selection)) io
 =	(db,io)
where
	infostring	= "Current Record Number: "+++toString selection
	editDialog	= CommandDialog EdDialogId "Edit & Query" [] addId dialogitems
	dialogitems
	 =	[ DynamicText editinfoid Left InputBoxWidth ""
		: [	 text
		  \\ attDesc	<- descriptor
		  &	 eid		<- [FirstEditField..]
		  &	 sid		<- [FirstEditField+nrOfDescr..]
		  ,	 text		<- inputfield sid eid (toString attDesc)
		  ]
		] ++
		[ DialogButton upId	   (Below lastSid)	 "Next"		  Able Up
		, DialogButton downId  (Below upId)		 "Prev"		  Able Down
		, DialogButton dispQId (RightTo upId)	 " DisplQ "	  Able DisplQuery
		, DialogButton setQId  (RightTo dispQId) " SetQ "	  Able SetQuery
		, DialogButton srchQId (RightTo setQId)	 "SearchQ" 	  Able Search
		, DialogButton slctQId (RightTo srchQId) "SelectAllQ" Able SelectAll
		, DialogButton replId  (Below dispQId)	 "Replace"	  Able (AddRecord Replace)
		, DialogButton delId   (Below setQId)	 "Delete" 	  Able DeleteRecord
		, DialogButton addId   (Below srchQId)	 "Add" 		  Able (AddRecord New)
		, DialogButton sortId  (Below slctQId)	 "Sort"		  Able Sort
		]
	
	inputfield sid eid field
	 =	[StaticText sid Left field, EditText eid pos InputBoxWidth 1 ""]
	where 
		pos		= if (eid==FirstEditField) (XOffset sid offset) (Below (dec eid))
		offset	= Pixel (DfFont.width + MaxWidthD - MaxWidth DfFont.font [field])
	MaxWidthD	= MaxWidth DfFont.font [toString attDesc \\ attDesc <- descriptor]
	nrOfDescr	= length descriptor
	lastSid		= FirstEditField+2*nrOfDescr - 1
	[upId,downId,dispQId,setQId,srchQId,slctQId,replId,delId,addId,sortId:_] = [lastSid+1..]

//	Handling the edit dialog

DisplQuery :: DialogInfo *DataBase IO -> (*DataBase, IO)
DisplQuery info db=:{current={descriptor,query}} io 
 =	(db,SetTextFields "Query:" descriptor query io)

SetQuery :: DialogInfo *DataBase IO -> (*DataBase, IO)
SetQuery info db=:{current=state} io
 #	(nquery,io) = GetTextFields state.descriptor io
 =	({db & old=state,current={state & query = nquery}}, io)

Up :: DialogInfo *DataBase IO -> (*DataBase, IO)
Up _ db=:{current=state=:{records,query,selection=sel}} io
 |	isEmpty records	= (db, Beep io)
 |	otherwise		= MakeSelectionVisible {db & old=state,current={state & selection=nsel}} (ChangeSelection state sel nsel io)
where nsel = (sel+1) rem length records

Down :: DialogInfo *DataBase IO -> (*DataBase, IO)
Down _ db=:{current=state=:{records,query,selection=sel}} io
 |	isEmpty records	= (db, Beep io)
 |	otherwise		= MakeSelectionVisible {db & old=state,current={state & selection=nsel}} (ChangeSelection state sel nsel io)
where nsel	= (sel-1+n) rem n
	  n		= length records

Search :: DialogInfo *DataBase IO -> (*DataBase, IO)
Search info db=:{current=state=:{records,query,selection=sel}} io
 |	isEmpty found	= (db, Beep io)
 |	otherwise		= MakeSelectionVisible {db & old=state,current={state & selection=nsel}} (ChangeSelection state sel nsel io)
where
	nsel			= hd found
	found			= [i \\ e <- el ++ bl & i <- [sel+1 .. length records - 1] ++ [0..] | QueryRecord query e]
	(bl,el)			= splitAt (sel+1) records

QueryRecord :: Record Record -> Bool
QueryRecord query e
 =	and [ EqPref qf f \\ AS f <- e & AS qf <- query ]
where
	EqPref pref name
	 |	size pref > size name	= False
	 |	otherwise				= pref == name%(0,size pref - 1)

SelectAll :: DialogInfo *DataBase IO -> (*DataBase, IO)
SelectAll info db=:{current=state=:{records,query,selection,descriptor}} io
 |	isEmpty recs	= (db, Beep io)
 #	io				= ChangeSelection state selection 0 io
	io				= ChangeWindowTitle RecordWindowId selname io
 |	otherwise		= UpdateDbDomain {db & old=state,current=nstate} io
where
	recs			= filter (QueryRecord query) records
	nstate			= { state
					  & selection = 0
					  , records	  = recs
					  , name	  = selname
					  , fw		  = MaxWidth DbFont.font [s \\ rec <- recs, AS s <- rec] 
					  }
	selname			= "Select"

MakeSelectionVisible :: *DataBase IO -> (*DataBase,IO)
MakeSelectionVisible db=:{current={records,selection,descriptor}} io
 |	isEmpty records		= (db,io)
 #	(((_,visibletop),(_,visiblebot)), io)
						= WindowGetFrame RecordWindowId io
	selection_invisible = selthumb < visibletop || selthumb >= visiblebot
 |	selection_invisible	= ChangeScrollBar RecordWindowId (ChangeVThumb selthumb) db io
 |	otherwise			= (db,io)
where
	selthumb 			= toPicCo descriptor selection

DeleteRecord :: DialogInfo *DataBase IO -> (*DataBase, IO)
DeleteRecord dialogInfo db=:{current=state=:{records,selection,descriptor,fw}} io
 |	isEmpty records	= (db,Beep io)
 |	otherwise		= UpdateDbDomain {db & old=state,current=nstate} io
where
	newrecs		= removeAt selection records
	fieldwidth	= if recalcwidth (MaxWidth DbFont.font (map toString (flatten newrecs))) fw
	recalcwidth	= fw == MaxWidth DbFont.font (map toString (records !! selection))
	nselection	= if (isEmpty newrecs) 0 (selection mod length newrecs) 
	nstate		= {state & records = newrecs, selection = nselection, fw = fieldwidth}

AddRecord :: Bool DialogInfo *DataBase IO -> (*DataBase, IO)
AddRecord replace dialogInfo db=:{current=state=:{descriptor,selection,records,fw}} io
 |	isEmpty records && replace = (db,Beep io)
 =	UpdateDbDomain {db & old=state,current=nstate} io1
where
	(newrec,io1)	= GetTextFields descriptor io
	(index,newrecs)	= insertindex (<=) newrec (if replace (removeAt selection records) records)
	fieldwidth | recalc	= MaxWidth DbFont.font (map toString (flatten newrecs))
						= max (MaxWidth DbFont.font (map toString newrec)) fw
	recalc			= replace && MaxWidth DbFont.font (map toString (records !! selection)) == fw
	nstate			= {state & records=newrecs, selection=index, fw=fieldwidth}

Sort :: DialogInfo *DataBase IO -> (*DataBase, IO)
Sort dialogInfo db=:{current=state=:{records}} io
 =	UpdateDbDomain {db & old=state,current={state & records = sort records}} io

GetTextFields :: Descriptor IO -> (Record,IO)
GetTextFields descr io
 =	( [	 AS (GetEditText id dialogInfo)
	  \\ id <- [FirstEditField..FirstEditField+length descr-1]
	  ]
	, nio
	)
where (_,dialogInfo,nio) = GetDialogInfo EdDialogId io

SetTextFields :: String Descriptor Record IO ->IO
SetTextFields s d rec io
 =	ChangeDialog EdDialogId dialogchanges io
where
	dialogchanges
	 =	[ ChangeDynamicText editinfoid s
		: [   ChangeEditText id f
		  \\ id <- [FirstEditField..FirstEditField+length d-1]
		  &  AS f <- rec
		  ]
		]

MaxWidth :: Font [String] -> Int;
MaxWidth font []   = 0
MaxWidth font list = maxList (FontStringWidths list font)

ChangeSelection :: State Int Int IO -> IO
ChangeSelection state=:{descriptor=descr,records} old new io
 #	io	= DrawInWindow RecordWindowId (HiliteSelection state old ++ HiliteSelection state new) io
 = SetTextFields infostring descr (records!!new) io
where
	infostring = "Current Record Number: "+++toString new

UpdateDbDomain :: *DataBase IO -> (*DataBase,IO)
UpdateDbDomain db=:{current} io
 #	(db,io)	= ChangePictureDomain RecordWindowId (DbPictureDomain current 0 (max (length current.records) 1)) db io
	(db,io)	= DrawInWindowFrame   RecordWindowId UpdateRecordWindow db io
	(db,io)	= MakeSelectionVisible db io
 =	(db,io)

HiliteSelection :: State Int -> [Picture -> Picture]
HiliteSelection s i
 =	[ SetPenMode HiliteMode
	, FillRectangle (DbPictureDomain s i (inc i))
	, SetPenNormal
	, SetPenColour BlackColour
	]

UpdateRecordWindow :: UpdateArea u:DataBase -> (u:DataBase, [DrawFunction])
UpdateRecordWindow domains db=:{current=state=:{records,descriptor,selection}}
 |	isEmpty records	= (db, [EraseRectangle domain \\ domain <- domains])
 =	( db
	, [ SetFont DbFont.font
	  : [	drawFun
		\\	domain=:((_,top),(_,bottom)) <- domains
		,	toprec	<- [toRecCo descriptor top]
		,	botrec	<- [toRecCo descriptor (dec bottom)]
	 	,	drawFun
	 		 <-	[ EraseRectangle domain
	 			: [  drawAtt
				  \\ rec	 <- records % (toprec,botrec)
	 			  &  recNo	 <- [toprec..]
	 			  ,	 attName <- attNames
	 			  &	 attVal	 <- rec
	 			  &	 attNo	 <- [1..]
	 			  ,	 drawAtt
	 			  	 <-	[ MovePenTo (0,toPicCo descriptor recNo + attNo*DbFont.height)
	 					, DrawString (attName +++ Separator +++ toString attVal)
	  	]		] ]		]
	  ] ++ HiliteSelection state selection
	)
where
	attNames	= [s +++ toString (spaces (nameLength - size s)) \\ s <- strings]
	nameLength	= maxList [size s \\ s <- strings]
	strings		= [toString ad \\ ad <- descriptor]

//	Drawing utilities

DbPictureDomain :: State Int Int -> PictureDomain
DbPictureDomain state=:{descriptor,records,dw,fw} fr to 
 = ((~whiteMargin,toPicCo descriptor fr), (right,bottom))
where
	(minWidth,minHeight) = MinDbDomainSize
	whiteMargin	= DbFont.width
	right		= max (dw + MaxWidth DbFont.font [Separator] + fw + whiteMargin) minWidth
	bottom		= max (toPicCo descriptor to) minHeight

//	Switching between picture coordinates and indices in the list of records ('record coordinates')

toPicCo :: Descriptor Int -> Int
toPicCo descr n = n * (inc (length descr) * DbFont.height)

toRecCo :: Descriptor Int -> Int
toRecCo descr n = n / (inc (length descr) * DbFont.height)

// ---

instance < AttributeValue
where (<) (AS a) (AS b) = a < b

instance toString AttributeValue
where toString (AS s) = s

instance toString AttDescriptor
where toString (AttDesc name type) = name

