implementation module StdFile

// ********************************************************
//	Concurrent Clean Standard Library Module Version 1.4
//	Copyright 1997 University of Nijmegen
// ********************************************************

class (<<<) infixl a :: !*File !a -> *File

instance <<< Int where
  (<<<) file i = fwritei i file
  
instance <<< Char where
  (<<<) file c = fwritec c file
  
instance <<< {#Char} where
  (<<<) file s = fwrites s file

instance <<< Real where
  (<<<) file r = fwriter r file

//	File modes synonyms

FReadText	:== 0	//	Read from a text file
FWriteText	:== 1	//	Write to a text file
FAppendText	:== 2	//	Append to an existing text file
FReadData	:== 3	//	Read from a data file
FWriteData	:== 4	//	Write to a data file
FAppendData	:== 5	//	Append to an existing data file

//	Seek modes synonyms

FSeekSet	:== 0	//	New position is the seek offset
FSeekCur	:== 1	//	New position is the current position plus the seek offset
FSeekEnd	:== 2	//	New position is the size of the file plus the seek offset

::	Files
	:==	World

//	Access to the file system:

class FileEnv env where
	accFiles :: !.(*Files -> (.x,*Files)) !*env -> (!.x,!*env)
	appFiles :: !.(*Files -> *Files)      !*env -> *env

instance accFiles World where
	accFiles :: !.(*Files -> (.x,*Files)) !*World -> (!.x,!*World)
	accFiles accfun world = accfun world
	
	appFiles :: !.(*Files -> *Files) !*World -> *World
	appFiles appfun world = appfun world


//	Opening and Closing files:

fopen :: !{#Char} !Int !*Files -> (!(!Bool,!*File),!*Files)
fopen s i world
	# (b,f,world)	= fopen` s i world
	= ((b,f),world)
where
	fopen` :: !{#Char} !Int !*Files -> (!Bool,!*File,!*Files)
	fopen` s i f 
		= code {
			.d 1 1 i
			jsr	openF
			.o 0 3 b f
				}

fclose :: !*File !*Files -> (!Bool,!*Files)
fclose file world
	= code {
		.d 0 2 f
			jsr	closeF
		.o 0 1 b
			}

stdio :: !*Files -> (!*File,!*Files)
stdio f
	= code {
		.d 0 0
			jsr	stdioF
		.o 0 2 f
			}

sfopen :: !{#Char} !Int !*Files -> (!(!Bool,!File),!*Files)
sfopen s i world
	# (b,f,world) = sfopen` s i world
	= ((b,f),world)
where
	sfopen` :: !{#Char} !Int !*Files -> (!Bool,!File,!*Files)
	sfopen` s i f 
		= code {
			.d 1 1 i
				jsr	openSF
			.o 0 3 b f
				}

freopen::!*File !Int -> (!Bool,!*File)
/*	Re-opens an open file in a possibly different mode.
	The boolean indicates whether the file was successfully closed before reopening. */
freopen f m 
	= code {
		.inline freopen
		.d 0 3 f i
			jsr reopenF
		.o 0 3 b f
		.end
	}

//	Input. The boolean output parameter reports success or failure of the operations.

freadc::!*File -> (!Bool,!Char,!*File)
/*	Reads a character from a text file or a byte from a datafile. */
freadc f
	= code {
		.inline freadc
		.d 0 2 f
			jsr	readFC
		.o 0 4 b c f
		.end
	}

freadi::!*File -> (!Bool,!Int,!*File)
/*	Reads an integer from a textfile by skipping spaces, tabs and newlines and
	then reading digits, which may be preceeded by a plus or minus sign.
	From a datafile freadi will just read four bytes (a Clean Int). */
freadi f
	= code {
		.inline freadi
		.d 0 2 f
			jsr	readFI
		.o 0 4 b i f
		.end
	}

freadr::!*File -> (!Bool,!Real,!*File)
/*	Reads a real from a textfile by skipping spaces, tabs and newlines and then
	reading a character representation of a real number.
	From a datafile freadr will just read eight bytes (a Clean Real). */
freadr f
	= code {
		.inline freadr
		.d 0 2 f
			jsr	readFR
		.o 0 5 b r f
		.end
	}

freads::!*File !Int -> (!*{#Char},!*File)
/*	Reads n characters from a text or data file, which are returned as a {#Char}.
	If the file doesn't contain n characters the file will be read to the end
	of the file. An empty {#Char} is returned if no characters can be read. */
freads f l
	= code {
		.inline freads
		.d 0 3 f i
			jsr readFS
		.o 1 2 f
		.end
	}

freadline::!*File -> (!{#Char},!*File)
/*	Reads a line from a textfile. (including a newline character, except for the last
	line) freadline cannot be used on data files. */
freadline f
	= code  {
		.inline freadline
		.d 0 2 f
			jsr readLineF
		.o 1 2 f
		.end
	}

//	Output. Use FError to check for write errors.

fwritec::!Char !*File -> *File
/*	Writes a character to a textfile.
	To a datafile fwritec writes one byte (a Clean Char). */
fwritec c f
	= code {
		.inline fwritec
		.d 0 3 c f
			jsr writeFC
		.o 0 2 f
		.end
	}

fwritei::!Int !*File -> *File
/*	Writes an integer (its textual representation) to a text file.
	To a datafile fwritec writes four bytes (a Clean Int). */
fwritei i f 
	= code {
		.inline fwritei
		.d 0 3 i f
			jsr writeFI
		.o 0 2 f
		.end
	}

fwriter::!Real !*File -> *File
/*	Writes a real (its textual representation) to a text file.
	To a datafile fwriter writes eight bytes (a Clean Real). */
fwriter r f
	= code {
		.inline fwriter
		.d 0 4 r f
			jsr writeFR
		.o 0 2 f
		.end
	}

fwrites::!{#Char} !*File -> *File
/*	Writes a {#Char} to a text or data file. */
fwrites s f 
	= code {
		.inline fwrites
		.d 1 2 f
			jsr writeFS
		.o 0 2 f
		.end
	}


//	Tests

fend::!*File -> (!Bool,!*File)
/*	Tests for end-of-file. */
fend f 
	= code {
		.inline fend
		.d 0 2 f
			jsr endF
		.o 0 3 b f
		.end
	}

ferror::!*File -> (!Bool,!*File)
/*	Has an error occurred during previous file I/O operations? */
ferror f 
	= code {
		.inline ferror
		.d 0 2 f
			jsr errorF
		.o 0 3 b f
		.end
	}

fposition::!*File -> (!Int,!*File)
/*	returns the current position of the file pointer as an integer.
	This position can be used later on for the fseek function. */
fposition f 
	= code {
		.inline fposition
		.d 0 2 f
			jsr positionF
		.o 0 3 i f
		.end
	}

fseek::!*File !Int !Int -> (!Bool,!*File)
/*	Move to a different position in the file, the first integer argument is the offset,
	the second argument is a seek mode. (see above). True is returned if successful. */
fseek f p m 
	= code {
		.inline fseek
		.d 0 4 f i i
			jsr seekF
		.o 0 3 b f
		.end
	}


//	Predefined files.


stderr::   *File
/*	Open the 'Errors' file for writing only. May be opened more than once. */
stderr 
	= code {
		.inline stderr
		.d 0 0
			jsr	stderrF
		.o 0 2 f
		.end
	}

sfreadc::!File -> (!Bool,!Char,!File)
sfreadc f 
	= code {
		.inline sfreadc
		.d 0 2 f
			jsr	readSFC
		.o 0 4 b c f
		.end
	}

sfreadi::!File -> (!Bool,!Int,!File)
sfreadi f
	= code {
		.inline sfreadi
		.d 0 2 f
			jsr	readSFI
		.o 0 4 b i f
		.end
	}

sfreadr::!File -> (!Bool,!Real,!File)
sfreadr f 
	= code {
		.inline sfreadr
		.d 0 2 f
			jsr	readSFR
		.o 0 5 b r f
		.end
	}


sfreads::!File !Int -> (!{#Char},!File)
sfreads f i 
	= code {
		.inline sfreads
		.d 0 3 f i
			jsr readSFS
		.o 1 2 f
		.end
	}

sfreadline::!File -> (!{#Char},!File)
sfreadline f 
	= code {
		.inline sfreadline
		.d 0 2 f
			jsr readLineSF
		.o 1 2 f
		.end
	}

sfseek::!File !Int !Int -> (!Bool,!File)
sfseek f i1 i2 
	= code {
		.inline sfseek
		.d 0 4 f i i
			jsr seekSF
		.o 0 3 b f
		.end
	}

/*	Change a file so that from now it can only be used with sfF... operations. */
fshare::!*File -> File
fshare f 
	= code {
		.inline fshare
		.d 0 2 f
			jsr shareF
		.o 0 2 f
		.end
	}

/*	The functions sfend and sfposition work like fend and fposition, but don't return a
	new file on which other operations can continue. They can be used for files opened
	with sfopen or after fshare, and in guards for files opened with fopen or freopen. */
sfend::!File -> Bool
sfend f
	= code {
		.inline sfend
		.d 0 2 f
			jsr endSF
		.o 0 1 b
		.end
	}

sfposition::!File -> Int
sfposition f
	= code {
		.inline sfposition
		.d 0 2 f
			jsr positionSF
		.o 0 1 i
		.end

		.inline fopen
		.end
		.inline fclose
		.end
		.inline stdio
		.end
		.inline sfopen
		.end
		.inline <<<;i
		.end
		.inline <<<;c
		.end
		.inline <<<;r
		.end
		.inline <<<;#
		.end
		.inline accFiles;w
		.end
		.inline appFiles;w
		.end
	}
