module worm;

/*
	The famous Unix game 'worm' (or 'snake') in Concurrent Clean.
	
	Run the program using the "No Console" option (Application options).

	To generate an application for this program the memory of the Clean
	0.8 application should be set to at least 2000K.
	The linker needs an additional 700K of free memory inside or outside
	the Clean 0.8 application.
	To launch the generated application 650K of free memory is needed
	outside the Clean 0.8 application.
*/

import	StdClass;
import StdBool, StdChar, StdString, StdFile, StdArray;
import deltaDialog, deltaEventIO;
import deltaWindow, deltaMenu, deltaTimer, deltaSystem;
import wormfile, wormshow;


    
::	*IO		:==  IOState State;
::	*TState	:== (State, IO);


     

	FileID			:== 1;
		PlayID		:== 11;
		InterruptID	:== 12;
			HaltID	:== 121;
			ContID	:== 122;
		QuitID		:== 13;
	LevelID			:== 2;
		EasyID		:== 21;
		MediumID	:== 22;
		HardID		:== 23;
		HiScoreID	:== 24;
	
	HighDlogID		:== 1000;
		HighOKID	:== 10001;
		HighTitleID	:== 10002;
	OverDlogID		:== 2000;
		OverOKID	:== 20001;

	WindowID		:== 1;
	WdCorner		:== (0,0);
	WdPicSize		:== ((0,0),WdMinSize);
	WdMinSize		:== (488,303);
	ScrBar			:== ScrollBar (Thumb 0) (Scroll 8);
	
	TimerID			:== 1;

	EasySpeed		:== TicksPerSecond / 10;
	MediumSpeed		:== TicksPerSecond / 15;
	HardSpeed		:== TicksPerSecond / 20;
	
	HelpFile		:== "WormHelp";
	HiScoresFile	:== "wormhi";

	NrOfLevels		:== 8;
	NrOfHiScores	:== 8;
	PointsPerLevel	:== 500;
	StartLevel		:== 1;
	NrOfWorms		:== 4;


InitialiseRandomSeed :: State IO -> TState;
InitialiseRandomSeed (lv,fd,g,p,d,w,(files,hs),ls,_) io
	=  ((lv,fd,g,p,d,w,(files,hs),ls,seed), io`);
	where {
		((hours,minutes,seconds), io`)
			= GetCurrentTime io;
		seed
			= seconds + minutes * 60 + hours * 3600 + 1
	};

Start	:: * World -> * World;
Start world =  CloseEvents events` (closefiles files` world``);
	where {
	files`                 =: WriteHiScores hifile highs`;
	(l,f,g,p,d,w,highs`,ls,speed)=: state;
	(state,events`)        =: PlayWorm highs events;
	(hifile, highs)        =: ReadHiScores HiScoresFile files;
	(files,world``)        =: openfiles world`;
	(events,world`)        =: OpenEvents world;
	};

PlayWorm	:: HiScores EVENTS -> (State,EVENTS);
PlayWorm (files,highs) events
	=  StartIO [about, menu, window, timer] (InitState files` highs) [InitialiseRandomSeed] events;
	
		where {
		about	=: DialogSystem [about_dialog];
		(about_dialog,files`)=: MakeAboutDialog "Worm" HelpFile files Help;

		menu	=: MenuSystem [file, options];
		file	=: PullDownMenu FileID "File" Able
					[MenuItem PlayID "Play"     (Key 'R') Able Play,
					 MenuItemGroup InterruptID [MenuItem HaltID "Halt" (Key '.') Unable Halt],
					 MenuSeparator,
					 MenuItem QuitID "Quit"     (Key 'Q') Able Quit
					];
		options	=: PullDownMenu LevelID "Options" Able
					[MenuRadioItems EasyID 
						[MenuRadioItem EasyID   "Slow"   (Key '1') Able (SetSpeed EasySpeed),
						 MenuRadioItem MediumID "Medium" (Key '2') Able (SetSpeed MediumSpeed),
						 MenuRadioItem HardID   "Fast"   (Key '3') Able (SetSpeed HardSpeed)
				  		],
					 MenuSeparator,
					 MenuItem HiScoreID "High Scores" (Key 'H') Able ShowHiScores
					];
						
		window=:		WindowSystem [playfield];
		playfield=:	FixedWindow WindowID WdCorner "Worm" WdPicSize UpdateWindow
								[GoAway Quit, Keyboard Able MakeTurn];
								      
		timer=:		TimerSystem [Timer TimerID Unable EasySpeed OneStep];
		};

InitState	:: Files Highs -> State;
InitState files highs
	=  (initlevel,newFood,0,0,RightKey,initworm,(files,highs),NrOfWorms,0);
	where {
	(newFood,_) = NewFood initworm initlevel 0;
	initworm =: NewWorm initlevel;
	initlevel=: InitLevel EasySpeed StartLevel;
	};

/*	The update function for the playfield window.
*/

UpdateWindow	:: UpdateArea State -> (State, [DrawFunction]);
UpdateWindow update_area state=:(level,food,g,points,d,worm,h,lifes,seed)
	= 	(state, DrawGame level food points worm lifes);

/*	The function for the Help button of the about-dialog
*/

Help	:: State IO -> TState;
Help (lv,fd,g,p,d,w,(files,hs),ls,seed) io =  ((lv,fd,g,p,d,w,(files`,hs),ls,seed), io`);
	where {
	(files`,io`)=: ShowHelp HelpFile files io;
	};

/*	The function for the Play command.
*/

Play	:: State IO -> TState;
Play ((fix,speed,l,o1,o2),food,g,p,d,w,high,ls,seed) io
	= 	(initstate, io1);
	where {
	io1=: ChangeIOState [ActivateWindow WindowID,
						DisableMenus [LevelID],
						DisableMenuItems [PlayID, QuitID],
						EnableMenuItems [HaltID],
						SetTimerInterval TimerID speed,
						EnableTimer TimerID,
						DrawInWindow WindowID [EraseGame : drawgame],
						ChangeWindowCursor WindowID HiddenCursor] io;
	initstate=:  (initlevel, initfood, 0, initpoints, RightKey, initworm, high, initlifes, seed`);
	initlevel=:  InitLevel fix StartLevel;
	(initfood, seed`)=:   NewFood initworm initlevel seed;
	initpoints=: 0;
	initworm =:  NewWorm initlevel;
	initlifes=:  NrOfWorms;
	drawgame=:   DrawGame initlevel initfood initpoints initworm initlifes;
	};


/*	The functions for the Halt/Continue command(s).
*/

Halt	:: State IO -> TState;
Halt state io =  (state, io1);
	where {
	io1=: ChangeIOState [DisableTimer TimerID,
						EnableMenuItems [QuitID],
						RemoveMenuItems [HaltID],
						InsertMenuItems InterruptID 1 [continue],
						ChangeWindowCursor WindowID StandardCursor] io;
	continue=: MenuItem ContID "Continue" (Key '.') Able Continue;
	};


Continue	:: State IO -> TState;
Continue state io =  (state, io1);
	where {
	io1=: ChangeIOState [ActivateWindow WindowID,
						DisableMenuItems [QuitID],
						RemoveMenuItems [ContID],
						InsertMenuItems InterruptID 1 [halt],
						EnableTimer TimerID,
						ChangeWindowCursor WindowID HiddenCursor] io;
	halt=: MenuItem HaltID "Halt" (Key '.') Able Halt;
	};

/*	The function for the Quit command: stop the interaction (the program).
*/

Quit	:: State IO -> TState;
Quit state io =  (state, QuitIO io);


/*	Set a new speed (called when one of the Options commands is chosen).
*/

SetSpeed	:: Int State IO -> TState;
SetSpeed fix ((x,s,l,o1,o2),f,g,p,d,w,h,ls,seed) io
	= 	(((fix,fix,l,o1,o2),f,g,p,d,w,h,ls,seed), io);


/*	Show the high scores.
*/

ShowHiScores	:: State IO -> TState;
ShowHiScores state=:(l,f,g,p,d,w,(files,[]),ls,seed) io
	=  (state, Alert "No high scores available." io);
ShowHiScores state=:(l,f,g,p,d,w,(files,highs),ls,seed) io
	= 	OpenModalDialog dialog state io;
		where {
		dialog=: CommandDialog HighDlogID "High Scores" [] HighOKID [title : scores];
		title=:	StaticText HighTitleID Center "Worm High Scores:";
		scores=: MakeScores 1 highs;
		};

MakeScores	:: Int Highs -> [DialogItem State IO];
MakeScores id [] =  [DialogButton HighOKID Center "OK" Able HighOK];
MakeScores id [(name, hi) : scores]
	= 	[dt, st : MakeScores (inc id) scores];
		where {
		dt=: DynamicText id (YOffset (dec id) (Pixel 2)) (MM 65.0) ( toString id  +++  ". " +++ name );
		st=: StaticText (id + 20) (XOffset id (MM 0.0)) (toString hi);
		};

HighOK	:: DialogInfo State IO -> TState;
HighOK dialog state io =  (state, CloseActiveDialog io);

/*	The MakeTurn function is called when a key is pressed.
*/

MakeTurn	:: KeyboardState State IO -> TState;
MakeTurn (key,KeyDown,mod_new) state=:(l,f,g,p,UpKey,w,h,ls,seed) io
	| key == LeftKey || key == RightKey =  OneStep 1 (l,f,g,p,key,w,h,ls,seed) io;
	=  (state,io);
MakeTurn (key,KeyDown,mod_new) state=:(l,f,g,p,DownKey,w,h,ls,seed) io
	| key == LeftKey || key == RightKey =  OneStep 1 (l,f,g,p,key,w,h,ls,seed) io;
	=  (state,io);
MakeTurn (key,KeyDown,mod_new) state=:(l,f,g,p,LeftKey,w,h,ls,seed) io
	| key == UpKey || key == DownKey =  OneStep 1 (l,f,g,p,key,w,h,ls,seed) io;
	=  (state,io);
MakeTurn (key,KeyDown,mod_new) state=:(l,f,g,p,RightKey,w,h,ls,seed) io
	| key == UpKey || key == DownKey =  OneStep 1 (l,f,g,p,key,w,h,ls,seed) io;
	=  (state,io);
MakeTurn keyboard_state state io
	=  (state,io);


/*	The function for the Timer device: do one step of the worm game.
*/

OneStep	:: TimerState State IO -> TState;
OneStep nul state=:(l,f,g,p,k,w,h,lifes,seed) io
	| end && (lifes == 0) = 	GameOver state1 io2;
	| end = 	NextLife state1 io1;
	= 	(state1, io1);
		where {
		(end, state1, io1)=:	Step state io;
		io2=: ChangeIOState [EnableMenus [LevelID],
							EnableMenuItems [PlayID, QuitID],
							DisableMenuItems [HaltID],
							DisableTimer TimerID] io1;
		};

Step	:: State IO -> (Bool, State, IO);
Step (level=:(f,speed,lvl,o1,o2), food, grow, pts, dir, worm=:[oldh:rest], high, lifes,seed) io
	|  pts mod PointsPerLevel  >  pts` mod PointsPerLevel  = 	ToNextLevel level pts` high lifes` seed` io;
	| scored = 	(end, (level, food`, grow``, pts`, dir, worm`, high, lifes,seed`), Beep io1);
	= 	(end, (level, food`, grow``, pts`, dir, worm`, high, lifes,seed`), io1);
		where {
		(head, tail, grow`, worm`)   =: StepWorm dir grow worm;
		end                          =: Collision level worm head;
		(scored, food`, pts`, grow``, seed`)=: Dinner worm` level food pts grow` head seed;
		lifes`=: PossibleExtraLife pts pts` lifes;
		io1=: DrawInWindow WindowID [DrawStep scored food food` pts` oldh head tail] io;
		}; 

ToNextLevel	:: Level Points HiScores Lifes RandomSeed IO -> (Bool, State, IO);
ToNextLevel (fix,speed,level,o1,o2) points high lifes seed io
	= 	(False, state, io`);
		where {
		state		=: (newlevel, newfood, 0, points, RightKey, initworm, high, lifes,seed`);
		(newfood, seed`)     =: NewFood initworm newlevel seed;
		initworm	=: NewWorm newlevel;
		newlevel	=: NewLevel fix speed level;
		io`			=: NextLevelAnimation io;
		};

NextLevelAnimation	:: IO -> IO;
NextLevelAnimation io
	= 	ChangeIOState [ChangeTimerFunction TimerID (BetweenLevels 40 (-1)),
		               SetTimerInterval TimerID speed] io;
		where {
		speed=: TicksPerSecond / 30;
		};

BetweenLevels	:: Int Int TimerState State IO -> TState;
BetweenLevels 41 s ts state=:(level,food,grow,points,dir,worm,high,lifes,seed) io =  (state,io`);
		where {
		io`=: ChangeIOState [DrawInWindow WindowID [EraseGame : DrawGame level food points worm lifes],
							SetTimerInterval TimerID speed, ChangeTimerFunction TimerID OneStep] io;
		(f,speed,l,o1,o2)=: level;
		};
BetweenLevels 1 s ts state io =  (state, ChangeTimerFunction TimerID (BetweenLevels 2 1) io);
BetweenLevels n s ts state io =  (state, ChangeTimerFunction TimerID (BetweenLevels (n + s) s) io`);
		where {
		io`=: DrawInWindow WindowID [DrawAnimation n s] io;
		};

StepWorm	:: Direction Grow Worm -> (Segment, Segment, Grow, Worm);
StepWorm dir 0 worm
	= 	(head, tail, 0, [head : worm`]);
		where {
		(tail, worm`)=: Get_and_RemoveLast worm;
		head=: NewHead dir worm;
		};
StepWorm dir grow worm
	= 	(head, (0,0), dec grow, [head : worm]);
	where {
	head=: NewHead dir worm;
		
	};

NewHead	:: Direction Worm -> Segment;
NewHead UpKey    [(x,y) : rest] =  (x, dec y);
NewHead DownKey  [(x,y) : rest] =  (x, inc y);
NewHead LeftKey  [(x,y) : rest] =  (dec x, y);
NewHead RightKey [(x,y) : rest] =  (inc x, y);

Get_and_RemoveLast	:: Worm         -> (Segment, Worm);
Get_and_RemoveLast [segment]    =  (segment, []);
Get_and_RemoveLast [seg : rest] =  (segment, [seg : rest`]);
	where {
	(segment, rest`)=: Get_and_RemoveLast rest;
	};

Collision	:: Level Worm Segment -> Bool;
Collision (f,s,l,obs1,obs2) worm head=:(x, y)
	| (x < 1) || (y < 1) || (x > SizeX) || (y > SizeY) = 	True;
	| (InRectangle x y obs1) || (InRectangle x y obs2) = 	True;
	= 	Contains head worm;

Contains	:: Segment Worm -> Bool;
Contains segment [] =  False;
Contains segment=:(x,y) [(h,v) : rest]
	| (x == h) && (y == v) = 	True;
	= 	Contains segment rest;

Dinner	:: Worm Level Food Points Grow Segment RandomSeed -> (Bool, Food, Points, Grow, RandomSeed);
Dinner worm level=:(fix,s,l,o1,o2) food=:(p,fx,fy) pts grow (x,y) seed
	| (x == fx) && (y == fy) = 	(True, newFood, AddPoints pts p fix, NewGrow grow p, newSeed);
	= 	(False, food, pts, grow, seed);
	where {
		(newFood, newSeed)
			= NewFood worm level seed;
	}

AddPoints	:: Points Points Int -> Points;
AddPoints pts add speed
	=  pts +  (add * (9 -  (60 * speed) / TicksPerSecond )) << 1 ;

NewGrow	:: Grow Points -> Grow;
NewGrow g p =  g +  (p * 3) / 2 ;

PossibleExtraLife	:: Points Points Lifes -> Lifes;
PossibleExtraLife oldp pts lifes
	|  oldp mod bound  >  pts mod bound  = 	inc lifes;
	=  	lifes;
	where {
	bound=: NrOfLevels * PointsPerLevel;
	};
	
/*	Fortunately there are worms left: use them when one has passed away.
*/

NextLife	:: State IO -> TState;
NextLife state=:(level,food,grow,points,key,worm,h,lifes,seed) io =  (state`, io`);
	where {
	state`   =: (level, newfood, 0, newpoints, RightKey, newworm, h, newlifes,seed`);
	(newfood, seed`)  =: NewFood newworm level seed;
	newpoints=: NewPoints points;
	newlifes =: dec lifes;
	newworm  =: NewWorm level;
	io`      =: DeadWormAlert worm io;
	};

NewPoints	:: Points -> Points;
NewPoints points
	| points <= 100 = 	0;
	= 	points - 100;

DeadWormAlert	:: Worm IO -> IO;
DeadWormAlert worm io 
	= 	ChangeIOState [ChangeTimerFunction TimerID (DeadWorm worm),
		               SetTimerInterval TimerID speed] io;
		where {
		speed=: TicksPerSecond / 30;
		};

DeadWorm	:: Worm TimerState State IO -> TState;
DeadWorm [segment:rest] ts state io =  (state, ChangeTimerFunction TimerID (DeadWorm rest) io`);
		where {
		io`=: DrawInWindow WindowID [EraseSegment segment] io;
		};
DeadWorm [] ts state=:(level,food,grow,points,key,worm,h,lifes,seed) io
	= 	(state, ChangeIOState [drawgame, ChangeTimerFunction TimerID OneStep,
		                         SetTimerInterval TimerID speed] io);
		where {
		drawgame=: DrawInWindow WindowID [EraseGame : DrawGame level food points worm lifes];
		(f,speed,l,o1,o2)=: level;
		};

/*	Game Over, check for a new high score.
*/

GameOver	:: State IO -> TState;
GameOver state=:(l,f,g,points,d,w,(files,highs),ls,seed) io
	| ItsAHighScore points highs = 	OpenModalDialog dialog state io`;
	= 	(state, Alert "Game Over, no high score." io`);
		where {
		io`		=: ChangeWindowCursor WindowID StandardCursor io;
		dialog	=: CommandDialog OverDlogID "Game Over"
				      [ItemSpace (MM 6.0) (MM 6.0)] OverOKID [st1,st2,et,ok];
		st1		=: StaticText 1 Left  "Game Over with a new high score!";
		st2		=: StaticText 2 Left  "Your name:";
		et		=: EditText   3 (RightTo 2) (MM 45.0) 1 "";
		ok		=: DialogButton OverOKID Center "OK" Able OverOK;
		};

OverOK	:: DialogInfo State IO -> TState;
OverOK dialog state=:(l,f,g,points,d,w,(files,highs),ls,seed) io
	| name == "" = 	(state, CloseActiveDialog io);
	= 	((l,f,g,points,d,w,(files,highs`),ls,seed), CloseActiveDialog io);
		where {
		highs`=: Take NrOfHiScores (AddScore (String13 name) points highs);
		name  =: GetEditText 3 dialog;
		};

ItsAHighScore	:: Points Highs -> Bool;
ItsAHighScore points scores
	| points == 0 = 	False;
	|  Length_new scores  < NrOfHiScores = 	True;
	= 	IsItReallyAHighScore points scores;

IsItReallyAHighScore	:: Points Highs -> Bool;
IsItReallyAHighScore points [] =  False;
IsItReallyAHighScore points [(name,score):rest]
	| points > score = 	True;
	= 	IsItReallyAHighScore points rest;

AddScore	:: String Points Highs -> Highs;
AddScore name points [] =  [(name,points)];
AddScore name points scores=:[score=:(n,pts):rest]
	| pts > points = 	[score : AddScore name points rest];
	= 	[(name,points) : scores];


/*	General Alert dialog.
*/

Alert	:: String IO -> IO;
Alert mes io =  io`;
		where {
		(id,io`)=: OpenNotice notice io;
		notice	=: Notice [mes] (NoticeButton 1 "OK") [];
		};


/*	Think of some new random food.
*/

NewFood	:: Worm Level RandomSeed -> (Food, RandomSeed);
NewFood worm level=:(f,s,l,obs1,obs2) seed
	| inworm || inobs1 || inobs2 = 	NewFood worm level seed3;
	= 	((pts, foodx, foody), seed3); 
		where {
		pts		=: IncMod (random1 >> 5) 9; foodx=: inc (IncMod (random2 >> 5) (SizeX - 2)); foody=: inc (IncMod (random3 >> 5) (SizeY - 2));
		inworm	=: Contains (foodx, foody) worm;
		inobs1	=: InRectangle foodx foody obs1;
		inobs2	=: InRectangle foodx foody obs2;
		(random1, seed1)
			= Random seed;
		(random2, seed2)
			= Random seed1;
		(random3, seed3)
			= Random seed2;
		};

InRectangle	:: Int Int Obstacle -> Bool;
InRectangle x y (lx,ty,rx,by)
	= 	not (   x < lx  ||  x > rx   ||  y < ty   ||  y > by );

// RWS, this is not the best random generator
Random :: RandomSeed -> (Int, RandomSeed);
Random seed
	=	(newSeed, newSeed)
	where {
		newSeed	= (seed * 75) mod 65537;
	}

IncMod	:: Int Int -> Int;
IncMod a b =  inc (a mod b);

/*	Make an new, initial worm.
*/

NewWorm	:: Level -> Worm;
NewWorm (f, s, level, o1, o2) =  MakeWorm (level mod NrOfLevels);

MakeWorm	:: Int -> [(Int,Int)];
MakeWorm 1 =  [(5,13),(4,13),(3,13),(2,13),(1,13)];
MakeWorm 2 =  [(5,5),(4,5),(3,5),(2,5),(1,5)];
MakeWorm 3 =  [(5,13),(4,13),(3,13),(2,13),(1,13)];
MakeWorm 4 =  [(5,13),(4,13),(3,13),(2,13),(1,13)];
MakeWorm 5 =  [(5,13),(4,13),(3,13),(2,13),(1,13)];
MakeWorm 6 =  [(5,1),(4,1),(3,1),(2,1),(1,1)];
MakeWorm 7 =  [(5,1),(4,1),(3,1),(2,1),(1,1)];
MakeWorm 0 =  [(5,14),(4,14),(3,14),(2,14),(1,14)];


/*	Construct the next level.
*/

InitLevel	:: Int Int -> Level;
InitLevel fix level =  (fix, fix, level, obs1, obs2);
	where {
	(obs1,obs2)=: SelectObstacles (level mod NrOfLevels);
	};

NewLevel	:: Int Int Int -> Level;
NewLevel fix speed level
	|  level mod NrOfLevels  == 0 = 	(fix, dec speed, level`, obs1, obs2);
	= 	(fix, speed, level`, obs1, obs2);
	where {
	level`=: inc level;
	(obs1,obs2)=: SelectObstacles (level` mod NrOfLevels);
	};

SelectObstacles	:: Int -> (Obstacle, Obstacle);
SelectObstacles 1 =  ((0,0,0,0),(0,0,0,0));
SelectObstacles 2 =  ((12,11,34,16),(0,0,0,0));
SelectObstacles 3 =  ((12,1,34,3),(12,24,34,26));
SelectObstacles 4 =  ((7,7,38,9),(7,17,38,19));
SelectObstacles 5 =  ((1,1,18,10),(28,17,45,26));
SelectObstacles 6 =  ((14,3,15,24),(30,3,31,24));
SelectObstacles 7 =  ((3,13,43,14),(22,3,24,24));
SelectObstacles 0 =  ((3,3,20,12),(26,15,43,24));


/*	Some miscellaneous functions.
*/

Length_new	:: [x]   -> Int;
Length_new []    =  0;
Length_new [x:y] =  inc (Length_new y);

Take	:: Int [x] -> [x];
Take 0 list  =  [];
Take n [x:y] =  [x : Take (dec n) y];
Take n list  =  list;

String13	:: String -> String;
String13 string |  size string  > 13 =  string % (0, 12);
					=  string;
