implementation module Mandelbrot;

import	StdClass;
import StdReal,StdInt,StdMisc,StdBool;
import FractalTypes, Complex;

    

:: FractalUpdArea :== ((Layer,GrainSize),Int,Int,Int);

    

// Drawing the image grained .
PaintSpot	:: Rectangle Int FunctionState Picture -> Picture;
PaintSpot rect color (area,palette,maxdepth,cf) p
	| color == maxdepth =  PaintRectangle rect (RGB 0.0 0.0 0.0) p;
	=  PaintRectangle rect (IndexToColor palette color) p;

PaintRectangle	:: Rectangle Colour Picture -> Picture;      
PaintRectangle rect=:((x,y),(x`,y`)) rgb_color p
	| (1 ==  x` - x ) && (1 ==  y` - y ) = 	LinePenTo (x,y) p`;
	= 	FillRectangle rect p`;
		where {
		p`=: MovePenTo (x,y) (SetPenColour rgb_color p);
		};

IndexToColor	:: Colours Int -> Colour;
IndexToColor (rs,ri,gs,gi,bs,bi) color =  RGB red green blue;
		where {
		red  =:  toReal (ri *  ReverseBits (1 << rs) color 0 )  / rfac;
		green=:  toReal (gi *  ReverseBits (1 << gs) color 0 )  / gfac;
		blue =:  toReal (bi *  ReverseBits (1 << bs) color 0 )  / bfac;
		rfac =: toReal (32704 >> rs);
		gfac =: toReal (32704 >> gs);
		bfac =: toReal (32704 >> bs);
		};

ReverseBits	:: Int Int Int -> Int;
ReverseBits mask number result
	| mask > 256 =  result;
	| (number bitand mask) == 0 =  ReverseBits mask` number result`;
	=  ReverseBits mask` number (inc result`);
	   where {
	   mask`  =: mask << 1;
	   result`=: result << 1;
	   };

// Draw one line of the image.
DrawFractalLine	:: FractalState IO -> (FractalState,IO);
DrawFractalLine (funcs,(layer,n,line),zoom) io 
	| line >= ScreenHeight =  ((funcs,(dec layer , n >> 1,0),zoom),io);
	=  ((funcs,(layer,n,line + n),zoom),DrawInWindow MyWindow df io);
		where {
		df=: [LazyDrawSpots (0,line) (ScreenWidth,ScreenHeight) n layer funcs];
		};

LazyDrawSpots	:: Point Point GrainSize Layer FunctionState Picture -> Picture;
LazyDrawSpots point=:(x,y) dim=:(h,v) n l funcs pic
	| x > h = 	pic;
	| DrawnSpot point l = 	LazyDrawSpots (xn,y) dim n l funcs pic;
	=  LazyDrawSpots (xn,y) dim n l funcs (PaintSpot ((x,y),(xn,yn)) value funcs pic);
		where {
		xn=: x + n;
		yn=: y + n;
		value=: Fractal_color point funcs ;
		};

// Update a specific area of the image.
UpdateFractalArea :: Rectangle FractalState -> (FractalState, DrawFunction);
UpdateFractalArea rect f=:(funcs,(layer,n,line),zoom)
   =  (f, LazyDrawArea upd firstline funcs);
   where {
   (upd,firstline)=: CalculateUpdate rect layer n;
   };
   
CalculateUpdate :: Rectangle Layer GrainSize -> (FractalUpdArea, Int);
CalculateUpdate ((x1,y1),(x2,y2)) layer n 
   =  (((layer,n),beginx,endx,endline),beginline);
      where {
      beginx=:     x1 / n  * n;
      endx=:      if ( x2 mod n  == 0) x2 (( x2 / n  + 1) * n);
      beginline=:  y1 / n  * n;
      endline=:   if ( y2 mod n  == 0) y2 (( y2 / n  + 1) * n);
      };

LazyDrawArea	:: FractalUpdArea Int FunctionState Picture -> Picture;
LazyDrawArea u=:((layer,n),beginx,endx,endline) line state pic
	| line >= endline = 	pic;
	= 	LazyDrawArea u (line + n) state (
			LazyDrawSpots` (beginx,line) endx n layer state pic);

LazyDrawSpots` :: Point Int GrainSize Layer FunctionState Picture -> Picture;
LazyDrawSpots` point=:(x,y) h n l funcs pic
   | x > h =  pic;
   =  LazyDrawSpots` (xn,y) h n l funcs (PaintSpot ((x,y),(xn,yn)) value funcs pic);
      where {
      xn=: x + n;
      yn=: y + n;
      value=: Fractal_color point funcs;
      };

// Check if a spot has already been done
DrawnSpot :: Point Int -> Bool;
DrawnSpot (x,y) l
   =   0 == (1 bitand  (x bitor y) >> l )  &&
          ( x <> 0  &&  y <> 0 );

Log2 :: Int -> Int;
Log2 1 =  1   ;
Log2 n =  inc (Log2 (n >> 1));

Log2_and_Power :: Int -> (!Int,!Int);
Log2_and_Power n 
   | halfpower == n =  (log2_1, halfpower); 
   =  (log2, power);
      where {
      log2  =: Log2 n; power=: 1 << log2;
      log2_1=: dec log2; halfpower=: 1 << log2_1;
      };

// The actual calculations
Fractal_color :: Point FunctionState -> Int;
Fractal_color (x,y) (area=: (((centerx,centery),width,height)),nrc,maxdepth,calc_func)=  depth mod NrOfColours;
      where {
      depth=: Calculate calc_func maxdepth rx ry;
      rx=:  centerx -  width / 2.0   +  ( toReal x  * width) /  toReal ScreenWidth  ;
      ry=:  centery -  height / 2.0   +  ( toReal y  * height) /  toReal ScreenHeight  ;
      };

Calculate	:: FractalFunction Int Real Real -> Int;
Calculate MSquare maxd rx ry =  MandelSquare maxd 0 0.0 0.0 rx ry;
Calculate MCube   maxd rx ry =  MandelCube   maxd 0 (0.0,0.0) (rx,ry);
Calculate MSin    maxd rx ry =  MandelSin    maxd 0 (0.0,0.0) (rx,ry);
Calculate MCos    maxd rx ry =  MandelCos    maxd 0 (0.0,0.0) (rx,ry);
Calculate MExp    maxd rx ry =  MandelExp    maxd 0 (0.0,0.0) (rx,ry);

MandelSquare :: Int Int Real Real Real Real -> Int;
MandelSquare maxdepth depth x y bx by
   | maxdepth == depth	=  maxdepth;
   |  sx + sy  > 2.8 	=  depth;
   =  MandelSquare maxdepth (inc depth) ( sx - sy  - bx) ( pxy + pxy  - by) bx by;
      where {
      sx=: x * x;
      sy=: y * y;
      pxy=: x * y;
      };

MandelCube :: Int Int ComplexNum ComplexNum -> Int;
MandelCube maxdepth depth zn c 
   | maxdepth == depth	=  maxdepth;
   |  FakeAbsC znp1  > 4.0		=  depth;
   =  MandelCube maxdepth (inc depth) znp1 c;
      where {
      znp1 =: AddC c (MulC zn (MulC zn zn));
      };

MandelSin :: Int Int ComplexNum ComplexNum -> Int;
MandelSin maxdepth depth zn c 
   | maxdepth == depth =  maxdepth;
   |  FakeAbsC znp1  > 4.0 =  depth;
   =  MandelSin maxdepth (inc depth) znp1 c;
      where {
      znp1 =: AddC c (SinC zn);
      };

MandelCos :: Int Int ComplexNum ComplexNum -> Int;
MandelCos maxdepth depth zn c 
   | maxdepth == depth =  maxdepth;
   |  FakeAbsC znp1  > 4.0 =  depth;
   =  MandelCos maxdepth (inc depth) znp1 c;
      where {
      znp1 =: AddC c (CosC zn);
      };

MandelExp :: Int Int ComplexNum ComplexNum -> Int;
MandelExp maxdepth depth zn c
   | maxdepth == depth =  maxdepth;
   |  FakeAbsC znp1  > 4.0 =  depth;
   =  MandelExp maxdepth (inc depth) znp1 c;
      where {
      znp1 =: AddC c (ExpC zn);
      };

// Timer device -> draw one line at a time 
DrawFractal :: TimerState FractalState IO -> (FractalState,IO);
DrawFractal x (funcs,(layer,0,line),zoom) io
   =  DoStopDrawing (funcs,(layer,1,line),zoom) io; 
DrawFractal x fstate io =  (fstate`, EnableTimer TimerID io`);
		where {
		(fstate`,io`)=: DrawFractalLine fstate (DisableTimer TimerID io);
		};

// Drawing has been stopped -> enable/disable menuitems/menus
DoStopDrawing :: FractalState IO -> (FractalState, IO);
DoStopDrawing state io 
   =  (state, io`);
      where {
      io`=: ChangeIOState [DisableTimer TimerID,
                          EnableMenus [OptionsID],
                          DisableMenuItems [StopDrawID, ContinueID],
                          EnableMenuItems [DrawID]] io;
      };	

// Set initial layer and grainsize.
InitDrawState	:: FractalState -> FractalState;
InitDrawState state
	= 	SetDrawState state (layer,size,0);
		where {
		(layer,size)=:Log2_and_Power (max ScreenHeight ScreenWidth);
		};
	              
