implementation module picture

//  Version 0.8.1

//
//  Drawing functions and other operations on Pictures. 
//

import StdEnv
import ostypes, ospicture
import font

		

::  UpdateArea      :== [Rectangle]
::  * Picture   :== UPicturePtr
::  DrawFunction :== Picture -> Picture

//  The predefined figures that can be drawn:

::  Point               :== (!Int, !Int)
::  Line                :== (!Point, !Point)
::  Curve               :== (!Oval, !Int, !Int)
::  Rectangle           :== (!Point, !Point)
::  Rect                :== (!Int,!Int,!Int,!Int)
::  RoundRectangle      :== (!Rectangle, !Int, !Int)
::  Oval                :== Rectangle
::  Circle              :== (!Point, !Int)
::  Wedge               :== (!Oval, !Int, !Int)
::  Polygon             :== (!Point, !PolygonShape)

::  PolygonShape        :== [Vector]
::  Vector              :== (!Int, !Int)

//  The pen attributes which influence the way figures are drawn:

::  PenSize     :== (!Int, !Int)
::  PenMode     = CopyMode     | OrMode        | XorMode       | ClearMode   | HiliteMode
	|   NotCopyMode | NotOrMode | NotXorMode    | NotClearMode
::  PenPattern  = BlackPattern
	|   DkGreyPattern
	|   GreyPattern
	|   LtGreyPattern
	|   WhitePattern

//  The colours:

::  Colour  = RGB Real Real Real
	|   BlackColour | RedColour
	|   WhiteColour | GreenColour
	|   BlueColour  | YellowColour
	|   CyanColour  | MagentaColour

		 
PI 		:== 3.1415926535898
MinRGB	:== 0.0
MaxRGB	:== 1.0

		

/*  Creating pictures, has to be done here to assure uniques of Picture
		  (instead of PicturePtr)
*/

NewPicture :: !WindowPtr -> PicturePtr
NewPicture w = OSNewPicture w
		
RemovePicture :: !PicturePtr -> Int
RemovePicture pic
		= OSRemovePicture pic


StartUpdate :: !WindowPtr !PicturePtr !(!Int,!Int) -> (!Picture, !UpdateArea)
StartUpdate w picptr (x_offset, y_offset)
		= (pic1, rects)
		  where 
		  (pic, n)      = OSStartUpdate w picptr x_offset y_offset
		  (pic1, rects) = getrects pic n
		  
		  getrects :: !Picture !Int -> (!Picture, UpdateArea)
		  getrects pic 0
		  		= (pic, [])
		  getrects pic m
		  		= (pic2, [rect : rects])
		  		  where
				  (pic1, x1, y1, x2, y2) = OSGetUpdateArea pic (n - m)
				  rect                   = ((x1, y1),(x2, y2))
				  (pic2, rects)          = getrects pic1 (m - 1)


EndUpdate :: ! Picture -> PicturePtr
EndUpdate pic
		= OSEndUpdate pic


StartDrawing :: ! WindowPtr !PicturePtr !(!Int,!Int) -> Picture
StartDrawing w pic (x_offset, y_offset)
		= OSStartDrawing w pic x_offset y_offset


EndDrawing :: ! Picture -> PicturePtr
EndDrawing pic
		= OSEndDrawing pic


NewButtonPicture :: !WindowPtr !Int !(!Int,!Int) !Bool -> Picture
NewButtonPicture w itemid (x_offset, y_offset) clearbackground
		= OSNewButtonPicture w itemid x_offset y_offset clearbackground

/*  Calculations with rects and regions.
maybe to be implemented ???

::  Rect_in_region !Rectangle !RgnHandle -> BOOL
		  Rect_in_region ((l,t),(r,b)) upd_rgn_handle
		->  DisposeAndCheckEmptyRgn sect_rgn rect_rgn !(NOT (OSGpiEmptyRgn sect_rgn 0)),
		  sect_rgn: OSGpiSectRgn rect_rgn upd_rgn_handle OSGpiNewRgn 0,
		  rect_rgn: OSGpiRectRgn OSGpiNewRgn (l,t,r,b) 0

::  DisposeAndCheckEmptyRgn !RgnHandle !RgnHandle !BOOL -> BOOL
		  DisposeAndCheckEmptyRgn rgn1 rgn2 b -> Before b (OSGpiDisposeRgn rgn1 (OSGpiDisposeRgn rgn2 0))
*/

/*  Rules setting the attributes of a Picture:
*/

SetPenSize  :: !PenSize !Picture -> Picture
SetPenSize (w,h) p = OSGpiPenSize ((w + h) / 2) p

SetPenMode  :: !PenMode !Picture -> Picture
SetPenMode CopyMode     p = OSGpiPenMode OSPatCopy    (OSGpiTextMode OSTextSrcOr p)
SetPenMode OrMode           p = OSGpiPenMode OSPatOr      (OSGpiTextMode OSTextSrcOr p)
SetPenMode XorMode      p = OSGpiPenMode OSPatXor     (OSGpiTextMode OSTextSrcXor p)
SetPenMode ClearMode    p = OSGpiPenMode OSPatBic     (OSGpiTextMode OSTextSrcBic p)
SetPenMode NotCopyMode  p = OSGpiPenMode OSNotPatCopy (OSGpiTextMode OSTextSrcOr p)
SetPenMode NotOrMode        p = OSGpiPenMode OSNotPatOr   (OSGpiTextMode OSTextSrcOr p)
SetPenMode NotXorMode   p = OSGpiPenMode OSNotPatXor  (OSGpiTextMode OSTextSrcOr p)
SetPenMode NotClearMode p = OSGpiPenMode OSNotPatBic  (OSGpiTextMode OSTextSrcOr p)
SetPenMode HiliteMode   p
//  -> OSGpiPenMode OSPatHilite (OSGpiTextMode OSPatHilite  p), IF HasColorOSGpiD
		= OSGpiPenMode OSPatXor    (OSGpiTextMode OSTextSrcOr p)

SetPenPattern   :: !PenPattern !Picture -> Picture
SetPenPattern BlackPattern  p = OSGpiPenPat OSPatBlack p
SetPenPattern DkGreyPattern p = OSGpiPenPat OSPatDkGray    p
SetPenPattern GreyPattern   p = OSGpiPenPat OSPatGray  p
SetPenPattern LtGreyPattern p = OSGpiPenPat OSPatLtGray    p
SetPenPattern WhitePattern  p = OSGpiPenPat OSPatWhite p

SetPenNormal    :: !Picture -> Picture
SetPenNormal p = OSGpiFGColour OSBlackColor
		  (OSGpiPenMode OSPatCopy (OSGpiPenPat OSPatBlack (OSGpiTextMode OSTextSrcOr p)))

/*  Using colours:
*/

SetPenColour    :: !Colour !Picture -> Picture
SetPenColour (RGB rd gr bl) p   = OSGpiRGBFGColour (REALToRGB rd) (REALToRGB gr) (REALToRGB bl) p
SetPenColour colour p           = OSGpiFGColour (ColourToNormalColor colour) p

// WHY IS THIS ?? (ERIC)
SetBackColour   :: !Colour !Picture -> Picture
SetBackColour p pic = pic
SetBackColour (RGB rd gr bl) p  = OSGpiRGBBGColour (REALToRGB rd) (REALToRGB gr) (REALToRGB bl) p
SetBackColour colour p              = OSGpiBGColour (ColourToNormalColor colour) p

REALToRGB   :: !Real -> Int
REALToRGB real | real >= MaxRGB = 255
	| real <= MinRGB = 0
		= toInt (real * 255.0)

ColourToNormalColor :: !Colour -> Int
ColourToNormalColor BlackColour                     = OSBlackColor
ColourToNormalColor RedColour                       = OSRedColor
ColourToNormalColor WhiteColour                     = OSWhiteColor
ColourToNormalColor GreenColour                     = OSGreenColor
ColourToNormalColor BlueColour                      = OSBlueColor
ColourToNormalColor YellowColour                    = OSYellowColor
ColourToNormalColor CyanColour                      = OSCyanColor
ColourToNormalColor MagentaColour                   = OSMagentaColor
ColourToNormalColor (RGB MaxRGB MaxRGB MaxRGB)      = OSWhiteColor
ColourToNormalColor rgb                             = OSBlackColor

/*  Using fonts:
*/

SetFont :: !Font !Picture -> Picture
SetFont font p  = p`
		  where 
		  (ok, p`)=: OSGpiSetFont name styleid size p
		  styleid=: StylesToStyleID style
		  (name,style,size)=: FontAtts font
												   

SetFontName :: !FontName !Picture -> Picture
SetFontName name p = OSGpiSetFontName name p

SetFontStyle    :: ![FontStyle] !Picture -> Picture
SetFontStyle styles p = OSGpiSetFontStyle (StylesToStyleID styles) p
		
SetFontSize :: !FontSize !Picture -> Picture
SetFontSize size p = OSGpiSetFontSize (max MinFontSize (min size MaxFontSize)) p

PictureCharWidth    :: !Char !Picture -> (!Int, !Picture)
PictureCharWidth char p = OSGpiCharWidth char p

PictureStringWidth  :: !String !Picture -> (!Int, !Picture)
PictureStringWidth string p = OSGpiStringWidth string p

PictureFontMetrics  :: !Picture -> (!FontInfo, !Picture)
PictureFontMetrics p = ((i1,i2,i3,i4),p2)
		  where 
		  (i1,i2,i3,i4,p2)=: OSGpiGetFontMetrics p
																   

/*  Rules changing the position of the pen:
*/

/*  Absolute and relative pen move operations (without drawing).
*/

MovePenTo   :: !Point !Picture -> Picture
MovePenTo (x, y) p = OSGpiMoveTo x y p

MovePen :: !Vector !Picture -> Picture
MovePen (vx, vy) p = OSGpiMove vx vy p

/*  Absolute and relative pen move operations (with drawing).
*/

LinePenTo   :: !Point !Picture -> Picture
LinePenTo (x, y) p = OSGpiLineTo x y p
		
LinePen :: !Vector !Picture -> Picture
LinePen (vx, vy) p = OSGpiLine vx vy p

/*  Drawing text:
*/

DrawChar    :: !Char !Picture -> Picture
DrawChar c p = OSGpiDrawChar c p
		
DrawString  :: !String !Picture -> Picture
DrawString s p = OSGpiDrawString s p

/*  Rules not changing the position of the pen after drawing:
*/

/*  Non plane figures:
*/

DrawPoint   :: !Point !Picture -> Picture
DrawPoint point p   = DrawPoint` point (x,y) p` 
		  where 
		  (x,y,p`)=: OSGpiGetPen p
														   

DrawLine    :: !Line !Picture -> Picture
DrawLine line p = DrawLine` line (x,y) p`
		  where 
		  (x,y,p`)=: OSGpiGetPen p
														   

DrawCurve   :: !Curve !Picture -> Picture
DrawCurve (r, s, t) p = OSGpiCurve (RectangleToRect r) s t p

DrawCPoint  :: !Point !Colour !Picture -> Picture
DrawCPoint point colour p = DrawPoint point (SetPenColour colour p)

DrawCLine   :: !Line !Colour !Picture -> Picture
DrawCLine line colour p = DrawLine line (SetPenColour colour p)

DrawCCurve  :: !Curve !Colour !Picture -> Picture
DrawCCurve curve colour p = DrawCurve curve (SetPenColour colour p)

DrawPoint`  :: !Point !Point !Picture -> Picture
DrawPoint` (x, y) (cx, cy) p = OSGpiMoveTo cx cy (OSGpiLine 0 0 (OSGpiMoveTo x y p))

DrawLine`   :: !Line !Point !Picture -> Picture
DrawLine` ((px,py),(qx,qy)) (cx, cy) p = OSGpiMoveTo cx cy (OSGpiLineTo qx qy (OSGpiMoveTo px py p))


/*  Rectangles:
*/

DrawRectangle   :: !Rectangle !Picture -> Picture
DrawRectangle r p = OSGpiRect OSPaintOutline (RectangleToRect r) p

FillRectangle   :: !Rectangle !Picture -> Picture
FillRectangle r p = OSGpiRect OSPaintFill (RectangleToRect r) p
		
EraseRectangle  :: !Rectangle !Picture -> Picture
EraseRectangle r p = OSGpiRect OSPaintErase (RectangleToRect r) p
		
InvertRectangle :: !Rectangle !Picture -> Picture
InvertRectangle r p = OSGpiRect OSPaintInvert (RectangleToRect r) p

MoveRectangleTo :: !Rectangle !Point !Picture -> Picture
MoveRectangleTo r (x, y) pict
		= OSGpiMoveRectangle pict x0 y0 x2 y2 x y True
		  where 
		  (x0,y0,x2,y2)=: RectangleToRect r
				   

MoveRectangle   :: !Rectangle !Vector !Picture -> Picture
MoveRectangle r (dx, dy) pict
		= OSGpiMoveRectangle pict x y x2 y2 (x + dx) (y + dy) True
		  where 
		  (x,y,x2,y2)=: RectangleToRect r
				   

CopyRectangleTo :: !Rectangle !Point !Picture -> Picture
CopyRectangleTo r (x, y) pict
		= OSGpiMoveRectangle pict x0 y0 x2 y2 x y False
		  where 
		  (x0,y0,x2,y2)=: RectangleToRect r
				   

CopyRectangle   :: !Rectangle !Vector !Picture -> Picture
CopyRectangle r (dx, dy) pict
		= OSGpiMoveRectangle pict x y x2 y2 (x + dx) (y + dy) False
		  where 
		  (x,y,x2,y2)=: RectangleToRect r
				   

/*  Rounded corner rectangles:
*/

DrawRoundRectangle  :: !RoundRectangle !Picture -> Picture
DrawRoundRectangle (r, w, h) p = OSGpiRoundRect OSPaintOutline (RectangleToRect r) w h p
		
FillRoundRectangle  :: !RoundRectangle !Picture -> Picture
FillRoundRectangle (r, w, h) p = OSGpiRoundRect OSPaintFill (RectangleToRect r) w h p
		
EraseRoundRectangle :: !RoundRectangle !Picture -> Picture
EraseRoundRectangle (r, w, h) p = OSGpiRoundRect OSPaintErase (RectangleToRect r) w h p
		
InvertRoundRectangle    :: !RoundRectangle !Picture -> Picture
InvertRoundRectangle (r, w, h) p = OSGpiRoundRect OSPaintInvert (RectangleToRect r) w h p

RectangleToRect :: !Rectangle -> Rect
RectangleToRect ((x,y), (x`,y`)) 
	| x_less_x` && y_less_y`= (x,y,x`,y`)
	| x_less_x`= (x,y`,x`,y)
	| y_less_y`= (x`,y,x,y`)
		= (x`,y`,x,y)
		  where 
		  x_less_x`=: x <= x`
		  y_less_y`=: y <= y`
				 

/*  Ovals:
*/

DrawOval    :: !Oval !Picture -> Picture
DrawOval r p = OSGpiOval OSPaintOutline (RectangleToRect r) p

FillOval    :: !Oval !Picture -> Picture
FillOval r p = OSGpiOval OSPaintFill (RectangleToRect r) p

EraseOval   :: !Oval !Picture -> Picture
EraseOval r p = OSGpiOval OSPaintErase (RectangleToRect r) p

InvertOval  :: !Oval !Picture -> Picture
InvertOval r p = OSGpiOval OSPaintInvert (RectangleToRect r) p

/*  Circles:
*/

DrawCircle  :: !Circle !Picture -> Picture
DrawCircle ((cx, cy), r) p
		= OSGpiOval OSPaintOutline (RectangleToRect ((cx - r, cy - r), (cx + r, cy + r))) p

FillCircle  :: !Circle !Picture -> Picture
FillCircle ((cx, cy), r) p
		= OSGpiOval OSPaintFill (RectangleToRect ((cx - r, cy - r), (cx + r, cy + r))) p

EraseCircle :: !Circle !Picture -> Picture
EraseCircle ((cx, cy), r) p
		= OSGpiOval OSPaintErase (RectangleToRect ((cx - r, cy - r), (cx + r, cy + r))) p

InvertCircle    :: !Circle !Picture -> Picture
InvertCircle ((cx, cy), r) p
		= OSGpiOval OSPaintInvert (RectangleToRect ((cx - r, cy - r), (cx + r, cy + r))) p

/*  Wedges:
*/

DrawWedge   :: !Wedge !Picture -> Picture
DrawWedge (r, s, t) p = OSGpiWedge OSPaintOutline (RectangleToRect r) s t p

FillWedge   :: !Wedge !Picture -> Picture
FillWedge (r, s, t) p = OSGpiWedge OSPaintFill (RectangleToRect r) s t p

EraseWedge  :: !Wedge !Picture -> Picture
EraseWedge (r, s, t) p = OSGpiWedge OSPaintErase (RectangleToRect r) s t p

InvertWedge :: !Wedge !Picture -> Picture
InvertWedge (r, s, t) p = OSGpiWedge OSPaintInvert (RectangleToRect r) s t p
		

/*  Polygons:
*/
NrPoints :: !PolygonShape !Int -> Int
NrPoints []    n    = n
NrPoints [p:r] n    = NrPoints r (inc n)

BuildPoly :: ![Vector] !Point !Int !Picture -> Picture
BuildPoly [(dx,dy):r] (x,y) i p = BuildPoly r (x + dx, y + dy) (inc i) (OSGpiAddToPoly x y i p)
BuildPoly []          (x,y) i p = OSGpiAddToPoly x y i p
   
NewPoly :: !Int !Polygon !Picture -> Picture
NewPoly paintkind ((x,y), shape) p
		= OSGpiStopPoly paintkind nr_points poly_pic
		  where 
		  poly_pic=:  BuildPoly shape (x,y) 0 start_pic
		  start_pic=: OSGpiStartPoly paintkind nr_points (OSGpiMoveTo x y p)
		  nr_points=: NrPoints shape 1
				   

DrawPolygon :: !Polygon !Picture    -> Picture
DrawPolygon poly pic = NewPoly OSPaintOutline poly pic
		
FillPolygon :: !Polygon !Picture    -> Picture
FillPolygon poly pic = NewPoly OSPaintFill poly pic
		
ErasePolygon    :: !Polygon !Picture    -> Picture
ErasePolygon poly pic = NewPoly OSPaintErase poly pic
		
InvertPolygon   :: !Polygon !Picture    -> Picture
InvertPolygon poly pic = NewPoly OSPaintInvert poly pic
		

ScalePolygon    :: !Int !Polygon -> Polygon
ScalePolygon k (position, shape) = (position, ScaleShape shape k)
				
ScaleShape  :: !PolygonShape !Int -> PolygonShape
ScaleShape [v : vs] k
		= let! 
		  	strict1
		  	strict2
		  in
				[strict1 : strict2]
		  where 
		  strict1=ScaleVector k v
		  strict2=ScaleShape vs k
				
		
ScaleShape [] k = []
		
MovePolygonTo   :: !Point !Polygon -> Polygon
MovePolygonTo p` (p, shape) = (p`, shape)
		
MovePolygon :: !Vector !Polygon -> Polygon
MovePolygon v (position, shape) = (TranslatePoint position v, shape)

TranslatePoint  :: !Point !Vector -> Point
TranslatePoint (x,y) (v,w)
		= let! 
			strict1
			strict2
		  in
		  (strict1, strict2)
		  where 
		  strict1=x + v
		  strict2=y + w
				
		

ScaleVector :: !Int !Vector -> Vector
ScaleVector k (v,w)
		= let! 
			  strict1
			  strict2
		  in
		  (strict1, strict2)
		  where 
		  strict1=k * v
		  strict2=k * w
				
		

/* BitMap Images (experimental)
		  Peter Jansen, november/december '94
*/

ShowBitMap    :: !String !Rectangle !Picture -> Picture
ShowBitMap  s r p = OSGpiShowBitMap s (RectangleToRect r) p

GetBitMapSize :: !String -> (Int,Int)
GetBitMapSize s= OSGpiGetBitMapSizes s

