implementation module Best.Shares

import Best.Types
import Data.Either
import Data.Func
import Data.Map => qualified get, updateAt
import Data.Map.GenJSON
import Data.Tuple
import Text
import iTasks
import iTasks.Extensions.DateTime

from Data.List import instance Foldable [], instance Traversable []
import Data.Traversable => qualified sequence

import mTask.Interpret.Device.TCP

import iTasks.Extensions.SQLDatabase

db :== SQLiteDatabase "best.db"

generic gToSqlRow a :: a [SQLValue] -> [SQLValue]
toSqlRow :: a -> [SQLValue] | gToSqlRow{|*|} a
toSqlRow a = gToSqlRow{|*|} a []

gToSqlRow{|Int|} i c = [SQLVInteger i:c]
gToSqlRow{|{#Char}|} s c = [SQLVText s:c]
gToSqlRow{|Bool|} b c = [SQLVInteger (if b 1 0):c]
gToSqlRow{|Real|} r c = [SQLVReal r:c]
gToSqlRow{|RECORD|} f (RECORD a) c = f a c
gToSqlRow{|PAIR|} fl fr (PAIR l r) c = fl l $ fr r c
gToSqlRow{|FIELD|} f (FIELD a) c = f a c
gToSqlRow{|OBJECT|} f (OBJECT a) c = f a c
gToSqlRow{|CONS|} f (CONS a) c = f a c
gToSqlRow{|DateTime|} dt c = [SQLVText (toString dt):c]
derive gToSqlRow BestDeviceReadingData, TCPSettings, (,)

generic gFromSqlRow a :: [SQLValue] -> MaybeErrorString (a, [SQLValue])
fromSqlRow :: [SQLValue] -> MaybeErrorString a | gFromSqlRow{|*|} a
fromSqlRow a = case gFromSqlRow{|*|} a of
	Error e = abort (e +++ "\n")
	Ok (a, []) = Ok a
	Ok _ = Error "fromSqlRow: Not all columns were parsed"
gFromSqlRow{|String|} [SQLVText i:s] = Ok (i, s)
gFromSqlRow{|String|} _ = Error "Require an SQLVText"
gFromSqlRow{|Int|} [SQLVInteger i:s] = Ok (i, s)
gFromSqlRow{|Int|} r = Error ("Require an SQLVInteger but got:" +++ toSingleLineText r)
gFromSqlRow{|Bool|} s = appFst ((==)1) <$> gFromSqlRow{|*|} s
gFromSqlRow{|Real|} [SQLVReal i:s] = Ok (i, s)
gFromSqlRow{|Real|} [SQLVFloat i:s] = Ok (i, s)
gFromSqlRow{|Real|} r = Error ("Require an SQLVReal but got:" +++ toSingleLineText r)
gFromSqlRow{|RECORD|} f s = appFst (\x->RECORD x) <$> f s
gFromSqlRow{|OBJECT|} f s = appFst (\x->OBJECT x) <$> f s
gFromSqlRow{|FIELD|} f s = appFst (\x->FIELD x) <$> f s
gFromSqlRow{|CONS|} f s = appFst (\x->CONS x) <$> f s
gFromSqlRow{|PAIR|} fl fr s = case fl s of
	Error e = Error e
	Ok (l, s) = case fr s of
		Error e = Error e
		Ok (r, s) = Ok (PAIR l r, s)
gFromSqlRow{|DateTime|} [SQLVText t:s] = case parseDateTime t of
	Error e = Error e
	Ok t = Ok (t, s)
gFromSqlRow{|DateTime|} _ = Error "Require an SQLVText"
derive gFromSqlRow BestDeviceReadingData, TCPSettings, (,)

initializeDatabase :: Task ()
initializeDatabase
	=   sqlExecute db [] (execute` readingtable [])
	>-| sqlExecute db [] (execute` devicetable [])
where
	readingtable = concat
		[ "CREATE TABLE IF NOT EXISTS readings"
		, "( device INTEGER "
		, ", stamp DATETIME DEFAULT (STRFTIME('%Y-%m-%d %H:%M:%S', 'NOW','localtime'))"
		, ", temperature REAL"
		, ", humidity REAL"
		, ", noise INTEGER"
		, ", light REAL"
		, ", motion INTEGER"
		, ", co2 INTEGER"
		, ", FOREIGN KEY(device) REFERENCES devices(rowid)"
		, ")"
		]
	devicetable = concat
		[ "CREATE TABLE IF NOT EXISTS devices"
		, "( name TEXT UNIQUE PRIMARY KEY"
		, ", host TEXT"
		, ", port INTEGER"
		, ")"
		]

execute` query args = appFst (maybe (Ok ()) (Error o toString)) o execute query args

initialDevices :: [(String, TCPSettings)]
initialDevices = 
	[ (n, {TCPSettings|host="localhost",port=p})
	\\p<-[8123..8127]
	& n<-["living", "bath", "office", "bed", "meeting"]
	]

devicesStore :: SDSLens () [String] ()
devicesStore
	= sdsFocus (db, ())
	$ sqlShare "BestDevices"
		(\_ cur->case execSelect "SELECT name FROM devices" [] cur of
			(Error e, cur) = (Error e, cur)
			(Ok x, cur) = (sequenceA (map fromSqlRow x), cur))
		(\_ _ c->(Ok (), c))

deviceStore :: SDSParallel String BestDevice BestDevice
deviceStore
	= sdsParallel
		"deviceStore"
		(\p->(p, ()))
		fst
		(SDSWriteConst \p w->Ok (Just w))
		(SDSWriteConst \_ _->Ok (Just ()))
		(sdsTranslate "tran" (tuple db) $ sqlShare "sql" readFun writeFun)
		devicesStore
where
	readFun p cur
		= case execSelect "SELECT host, port FROM devices WHERE name=? LIMIT 1" [SQLVText p] cur of
			(Ok [r], cur) = (fromSqlRow r, cur)
			(Ok _, cur) = (Error "No such device found", cur)
			(Error e, cur) = (Error ("read devicestore: " +++ toString e), cur)
	writeFun p w = execute` "INSERT OR REPLACE INTO devices VALUES (?, ?, ?)" (toSqlRow (p, w))

deviceReadingStore :: SDSLens String [BestDeviceReading] BestDeviceReadingData
deviceReadingStore
	= sdsTranslate "tran" (tuple db) $ sqlShare "BestReading" readFun writeFun
where
	readFun p cur
		# (rows, cur) = execSelect "SELECT stamp, temperature, humidity, noise, light, motion, co2 FROM readings, devices WHERE devices.rowid=readings.device AND devices.name=? ORDER BY readings.rowid DESC LIMIT 1" [SQLVText p] cur
		# (_, rows) = trace_stdout ("read", rows)
		= (case rows of
			Ok r = sequenceA (map fromSqlRow r)
			Error e = Error (toString e)
		, cur)

	writeFun p w cur
		# (rows, cur) = execSelect "SELECT rowid FROM devices WHERE name=? LIMIT 1" (toSqlRow p) cur
		= case rows of
			Ok [[id=:SQLVInteger _]] = execute` "INSERT INTO readings (device,temperature,humidity,noise,light,motion,co2) VALUES (?, ?, ?, ?, ?, ?, ?)" [id:toSqlRow w] cur
			Ok [_] = (Error "Malformed device found", cur)
			Ok _ = (Error "No device found", cur)
			Error e = (Error (toString e), cur)

import StdDebug, StdMisc, Debug.Trace
