implementation module Data.GenC 

import StdEnv, StdGeneric

import Control.Applicative
import Data.Array
import Data.Func
import Data.Functor
import Data.Generics
import Data.List
import Data.Maybe
import Data.Either
import Text

import mTask.Interpret.UInt

derive bimap Box

generateC :: (Box String a) String -> ([String], [String]) | gGenC a
generateC b=:(Box basename) prefix
	# (ctype, ctypedef) = (\(a, b)->(concat a, b)) $ toCType prefix $ toCTypeB b
	# imps = intersperse "\n"
		$ includes
		[ "#include \"" +++ basename +++ ".h\""
		, ""
		, parsefun ctype prefix, "{", "\t" +++ ctype +++ " r;"
		:toCParser prefix (toCParserB b)
		["\treturn r;","}"
		, ""
		, printfun ctype prefix, "{"
		:toCPrinter prefix (toCPrinterB b) ["}"]]
		]
	# defs = intersperse "\n"
		[ "#ifndef " +++ sanitize basename
		, "#define " +++ sanitize basename
		: includes
			[ concat ctypedef
			, ""
			, parsefun ctype prefix +++ ";"
			, printfun ctype prefix +++ ";"
			, "#endif"
			]
		]
	= (defs, imps)
where
	includes :: [String] -> [String]
	includes c = ["#include <stdint.h>", "#include <stdbool.h>", "#include <stdarg.h>", "":c]

	toCTypeB :: (v a) -> CTypeMaker a | gToCType{|*|} a
	toCTypeB _ = gToCType{|*|}

	toCParserB :: (v a) -> CParser a | gToCParser{|*|} a
	toCParserB _ = gToCParser{|*|}

	toCPrinterB :: (v a) -> CPrinter a | gToCPrinter{|*|} a
	toCPrinterB _ = gToCPrinter{|*|}

parsefun :: String String -> String
parsefun ctype prefix = concat
	[ctype, " parse", sanitize ctype, " (\n"
	, "\tuint8_t (*get)(void),\n"
	, "\tvoid (*err)(const char *errmsg, ...))"
	]
printfun :: String String -> String
printfun ctype prefix = concat
	["void print", sanitize ctype, " (\n"
	, "\t", ctype, " r, void (*putc)(uint8_t))"
	]

sanitize s = {if (isAlphanum c) c '_'\\c<-:s}

class prefix a :: a -> String
instance prefix String where prefix s = s
instance prefix SMInput where prefix s = s.SMInput.prefix
instance prefix CPState where prefix s = s.CPState.prefix

(<.>) infixl 9
(<.>) l r = l +++ "." +++ r

toEnumType st gtd = "enum " +++ toConsName st gtd
toEnumItem st gcd = prefix st +++ gcd.gcd_type_def.gtd_name +++ "_" +++ gcd.gcd_name

toStructType st gtd = "struct " +++ toName st gtd

toTypedefType st type name = "typedef " +++ type +++ " " +++ toName st name

toConsName st gtd = toName (prefix st +++ "cons_") gtd
toName st gtd = sanitize (prefix st +++ genericDescriptorName gtd)

toEnumDef :: String [String] -> [String]
toEnumDef _ [x] = []
toEnumDef name [e:es] = [name, "\n":enumJoin ("\t{ " +++ e) es]
where
	enumJoin acc [] = [acc, "\n\t}"]
	enumJoin acc [e:es]
		| foldrArr ((+) o width) 0 acc + size ", " + size e >= 80
			= [acc:enumJoin ("\n\t, " +++ e) es]
		= enumJoin (concat [acc, ", ",e]) es
	width '\t' = 8
	width c = 1

formatStructType :: ([String] -> String)
formatStructType = join "\n" o indent 0 o split "\n" o concat
where
	indent i [] = []
	indent i [s:ss]
		| endsWith "{" s   = [createArray i '\t' +++ s:indent (inc i) ss]
		| startsWith "}" s = [createArray (dec i) '\t' +++ s:indent (dec i) ss]
		| otherwise        = [createArray i '\t' +++ s:indent i ss]

toCType :: String (CTypeMaker a) -> ([String], [String])
toCType prefix f
	# (ctype, {imps,defs}) = (\(a, b)->(a, unBox b)) $ f {prefix=prefix,dict=[],fresh=0,inRecord=False} []
	= (ctype, intersperse "\n" ([i+++";"\\i<-map snd defs] ++ reverse imps))

:: SMInput  = {prefix :: String, fresh :: Int, inRecord :: Bool, dict :: [(String, String)]}
:: SMOutput = {defs :: [(String, String)], imps :: [String]}

unionAL = unionBy (\(x, _) (y, _)->x == y)

instance + SMOutput where (+) a b = {defs=unionAL a.defs b.defs, imps=a.imps++b.imps}
instance zero SMOutput where zero = {defs=[], imps=[]}
instance zero (Box a b) | zero a where zero = Box zero

gToCType{|Char|} _ c = (["char":c], zero)
gToCType{|UInt8|} _ c = (["uint8_t":c], zero)
gToCType{|UInt16|} _ c = (["uint16_t":c], zero)
gToCType{|Int|} _ c = (["int64_t":c], zero)
gToCType{|Real|} _ c = (["double":c], zero)
gToCType{|Bool|} _ c = (["bool":c], zero)
gToCType{|UNIT|} _ c = (c, zero)
gToCType{|EITHER|} fl fr st c
	# (c, Box oa) = fl st c
	# (c, Box ob) = fr {st & dict=unionAL st.dict oa.defs} c
	= (c, Box (oa + ob))
gToCType{|PAIR|} fl fr st c
	# (c, Box oa) = fr st c
	# st & dict = unionAL st.dict oa.defs
	# (c, Box ob) = if st.inRecord
		(fl st c)
		(fl {st & fresh=st.fresh+1} [" f", toString st.fresh, ";\n":c])
	= (c, Box (oa + ob))
gToCType{|OBJECT of gtd|} f st c
	= case lookup gtd.gtd_name st.dict of
		Just n = ([n:c], zero)
		Nothing
			//Newtype
			| gtd.gtd_num_conses == 0
				# (c`, Box o) = f st []
				# defs = [(gtd.gtd_name, toTypedefType st (concat c`) gtd):o.defs]
				= ([toName st gtd:c], Box {defs=defs, imps=o.imps})
			//Single constructor, no consenum needed
			| gtd.gtd_num_conses == 1
				# (c`, Box o) = f st ["};"]
				# def = [(gtd.gtd_name, toStructType st gtd):o.defs]
				# obj = formatStructType [toStructType st gtd, " {\n":c`]
				= ([toStructType st gtd:c], Box {defs=def, imps=[obj:o.imps]})
			# enum = \name->concat $ toEnumDef name $ map (toEnumItem st) gtd.gtd_conses
			//If it is just an enumeration, Just the enum
			| and [gcd.gcd_arity == 0\\gcd<-gtd.gtd_conses]
				# def = [(gtd.gtd_name, enum $ toEnumType st gtd):st.dict]
				= ([toEnumType st gtd:c], Box {defs=def, imps=[]})
			//Constructors with data fields
			# (c`, Box o) = f st ["} data;\n};"]
			# def = [(gtd.gtd_name, toStructType st gtd):o.defs]
			# obj = formatStructType [toStructType st gtd, " {\n", "enum ", toConsName st gtd, " cons;\nunion {\n":c`]
			= ([toStructType st gtd:c], Box {defs=def, imps=[obj,enum ("enum " +++ toConsName st gtd) +++ ";":o.imps]})
gToCType{|CONS of gcd|} f st c
	//No data field
	| gcd.gcd_arity == 0 = (c, zero)
	//Only one data field
	| gcd.gcd_arity == 1 = (\(a, b)->(a, reBox b)) $ f st [" ", toName st gcd, ";\n":c]
	//Multiple data fields
	# (c, Box o) = f st [" f", toString (gcd.gcd_arity - 1), ";\n} ", toName st gcd, ";\n":c]
	= (["struct " +++ toName st gcd +++ " {\n":c], Box o)
gToCType{|RECORD of grd|} f st c
	# (c`, Box o) = f {st & inRecord=True} ["};"]
	# defs = [(grd.grd_name, toStructType st grd +++ ";"):o.defs]
	# obj = formatStructType [toStructType st grd, " {\n":c`]
	= ([toStructType st grd:c], Box {defs=defs, imps=[obj:o.imps]})
gToCType{|FIELD of gfd|} f st c
	= (\(a, b)->(a, reBox b)) $ f st [" ", gfd.gfd_name,";\n":c]

toCParser :: String (CParser a) [String] -> [String]
toCParser prefix f c = unBox $ f {field=Nothing,indent=1,prefix=prefix,current="r"} c

:: CParser a :== CPState [String] -> Box [String] a
:: CPState = {indent :: Int, prefix :: String, current :: String, field :: Maybe Int}
assign = modify "="
modify m s v = indent s $ join " " [s.current +++ fieldmod s, m, v +++ ";"]
fieldmod {field=Nothing} = ""
fieldmod {field=Just s} = ".f" +++ toString s
indent s str = createArray s.indent '\t' +++ str

gToCParser{|Char|} s c = Box [assign s "get()":c]
gToCParser{|UInt8|} s c = Box [assign s "get()":c]
gToCParser{|UInt16|} s c = Box [assign s "get()<< 8", modify "+=" s "get()":c]
gToCParser{|Int|} s c = Box
	[ assign s "get()<< 56",      modify "+=" s "get()<< 48"
	, modify "+=" s "get()<< 40", modify "+=" s "get()<< 32"
	, modify "+=" s "get()<< 24", modify "+=" s "get()<< 16"
	, modify "+=" s "get()<< 8",  modify "+=" s "get()":c]
gToCParser{|Bool|} s c = Box [assign s "get() == 1":c]
gToCParser{|UNIT|} s c = Box c
gToCParser{|EITHER|} fl fr s c = reBox $ fl s $ unBox $ fr s c
gToCParser{|PAIR|} fl fr s c = reBox $ fl s $ unBox $ fr {s & field=inc <$> s.field} c
gToCParser{|FIELD of gfd|} f s c = reBox $ f {s & current=s.current <.> gfd.gfd_name} c
gToCParser{|RECORD|} f s c = reBox $ f s c
gToCParser{|OBJECT of gtd|} f s c
	//Newtype
	| gtd.gtd_num_conses == 0
		= reBox $ f s c
	//Single constructor, no consenum needed
	| gtd.gtd_num_conses == 1
		= reBox $ f s c
	//Enumeration
	| and [gcd.gcd_arity == 0\\gcd<-gtd.gtd_conses]
		= box $ [assign s "get()":c]
	//Constructors with data fields
	# s & current = s.current +++ fieldmod s
	= box $
		[indent s $ s.current <.> "cons = get();"
		,indent s $ "switch (" +++ s.current <.> "cons) {"
		:unBox $ f {s & indent=inc s.indent,field=Nothing}
		[indent s "default:"
		,indent s $ "\terr(\"Unknown constructor: \", " +++ s.current <.> "cons);"
		,indent s "}":c]]
gToCParser{|CONS of gcd|} f s c
	# s & field = if (gcd.gcd_arity > 1) (Just 0) Nothing
	//One of single constructor
	| gcd.gcd_type_def.gtd_num_conses == 1
		= reBox $ f {s & current=s.current <.> toName s gcd} c
	//One of multiple constructors
	| gcd.gcd_type_def.gtd_num_conses > 1
		= box $
			[indent {s & indent=dec s.indent} $ "case " +++ toEnumItem s gcd +++ ":"
			:unBox $ f {s & current=s.current <.> "data" <.> toName s gcd}
			[indent s "break;":c]]

toCPrinter :: String (CPrinter a) [String] -> [String]
toCPrinter prefix f c = unBox $ f {field=Nothing,indent=1,prefix=prefix,current="r"} c

:: CPrinter a :== CPState [String] -> Box [String] a
generic gToCPrinter a :: CPState [String] -> Box [String] a
put s before after = indent s $ concat ["putc(", before, s.current, fieldmod s, after, ");"]
printInt 1 s c = [put s "" " & 0xff":c]
printInt b s c =
	[put s "(" (" >> " +++ toString ((b-1)*8) +++ ") & 0xff")
	:printInt (dec b) s c]
gToCPrinter{|UInt8|} s c = Box (printInt 1 s c)
gToCPrinter{|UInt16|} s c = Box (printInt 2 s c)
gToCPrinter{|Int|} s c = Box (printInt 8 s c)
gToCPrinter{|Bool|} s c = Box [put s "" "":c]
gToCPrinter{|Char|} s c = Box [put s "" "":c]
gToCPrinter{|UNIT|} s c = Box c
gToCPrinter{|EITHER|} fl fr s c = reBox $ fl s $ unBox $ fr s c
gToCPrinter{|PAIR|} fl fr s c = reBox $ fl s $ unBox $ fr {s & field=inc <$> s.field} c
gToCPrinter{|CONS of gcd|} f s c
	# s & field = if (gcd.gcd_arity > 1) (Just 0) Nothing
	//One of single constructor
	| gcd.gcd_type_def.gtd_num_conses == 1
		= reBox $ f {s & current=s.current <.> toName s gcd} c
	//One of multiple constructors
	| gcd.gcd_type_def.gtd_num_conses > 1
		= box $
			[indent {s & indent=dec s.indent} $ "case " +++ toEnumItem s gcd +++ ":"
			:unBox $ f {s & current=s.current <.> "data" <.> toName s gcd}
			[indent s "break;":c]]
gToCPrinter{|FIELD of gfd|} f s c = reBox $ f {s & current=s.current <.> gfd.gfd_name} c
gToCPrinter{|RECORD|} f s c = reBox $ f s c
gToCPrinter{|OBJECT of gtd|} f s c
	//Newtype
	| gtd.gtd_num_conses == 0
		= reBox $ f s c
	//Single constructor, no consenum needed
	| gtd.gtd_num_conses == 1
		= reBox $ f s c
	//Enumeration
	| and [gcd.gcd_arity == 0\\gcd<-gtd.gtd_conses]
		= Box [put s "" "":c]
	//Constructors with data fields
	# s & current = s.current +++ fieldmod s
	= box $
		[indent s $ "putc(" +++ s.current <.> "cons);"
		,indent s $ "switch (" +++ s.current <.> "cons) {"
		:unBox $ f {s & indent=inc s.indent,field=Nothing}
		[indent s "}":c]]

toCValue :: a [Char] -> [Char] | gToCValue{|*|} a
toCValue a c = gToCValue{|*|} a c

gToCValue{|Char|} x c = [x:c]
gToCValue{|UInt8|} (UInt8 x) c = [toChar x:c]
gToCValue{|UInt16|} (UInt16 x) c = [toChar (x >> 8), toChar x:c]
gToCValue{|Int|} x c =
	[ toChar (x >> 56)
	, toChar (x >> 48)
	, toChar (x >> 40)
	, toChar (x >> 32)
	, toChar (x >> 24)
	, toChar (x >> 16)
	, toChar (x >> 8)
	, toChar x:c]
gToCValue{|Bool|} x c = [toChar (if x 1 0):c]
gToCValue{|UNIT|} x c = c
gToCValue{|EITHER|} l _ (LEFT x) c = l x c
gToCValue{|EITHER|} _ r (RIGHT x) c = r x c
gToCValue{|PAIR|} l r (PAIR x y) c = l x $ r y c
gToCValue{|CONS of {gcd_index}|} f (CONS x) c = [toChar gcd_index:f x c]
gToCValue{|FIELD|} f (FIELD x) c = f x c
gToCValue{|RECORD|} f (RECORD x) c = f x c
gToCValue{|OBJECT|} f (OBJECT x) c = f x c

fromCValue :: [Char] -> Either FromCValueError (a, [Char]) | gFromCValue{|*|} a
fromCValue i = gFromCValue{|*|} i

:: Parser a :== [Char] -> Either FromCValueError (a, [Char])
top :: Parser Char
top = satisfy (\_->True) CVEInputExhausted

satisfy :: (Char -> Bool) FromCValueError -> Parser Char
satisfy f e = \c->case c of
	[c:cs]
		| f c = Right (c, cs)
		      = Left e
	[] = Left CVEInputExhausted

yield :: a -> Parser a
yield a = \c->Right (a, c)

list :: [Parser a] -> Parser [a]
list [] = yield []
list [x:xs] = cons <<$>> x <<*>> list xs

cons x xs = [x:xs]

(<<$>>) infixl 4 :: (a -> b) (Parser a) -> Parser b
(<<$>>) f a = fmap (\(a, b)->(f a, b)) <$> a

(<<*>>) infixl 4 :: (Parser (a -> b)) (Parser a) -> Parser b
(<<*>>) f a = either Left (\(fa, c)->(fa <<$>> a) c) o f

(<<|>>) infixr 4 :: (Parser a) (Parser a) -> Parser a
(<<|>>) l r = \c->either (\_->r c) Right $ l c

int b = sum <<$>> list [(\x->toInt x << (i*8)) <<$>> top \\i<-[b-1,b-2..0]]
gFromCValue{|Char|} = top
gFromCValue{|UInt8|} = fromInt <<$>> int 1
gFromCValue{|UInt16|} = fromInt <<$>> int 2
gFromCValue{|Int|} = fromInt <<$>> int 8
gFromCValue{|Bool|} = ((==) '\1') <<$>> top
gFromCValue{|UNIT|} = yield UNIT
gFromCValue{|EITHER|} l r = (LEFT <<$>> l) <<|>> (RIGHT <<$>> r)
gFromCValue{|PAIR|} l r = PAIR <<$>> l <<*>> r
gFromCValue{|CONS of {gcd_index}|} f
	= (\x->CONS) <<$>> satisfy ((==)(toChar gcd_index)) CVEUnknownConstructor <<*>> f
gFromCValue{|FIELD|} f = (\x->FIELD x) <<$>> f
gFromCValue{|RECORD|} f = RECORD <<$>> f
gFromCValue{|OBJECT|} f = (\x->OBJECT x) <<$>> f
