implementation module System._Process

import StdEnv

import Data.Func
import Data.Maybe
from System.File import fileExists
import System.FilePath
import System.OSError
import System.Process
import System._Pointer
from System._WinBase import
	:: DWORD, :: SIZE_T, :: LPVOID, :: LPDWORD, :: LPCTSTR, :: HANDLE,
	:: PHANDLE, :: LPSECURITY_ATTRIBUTES, :: LPPROCESS_INFORMATION,
	:: LPSTARTUPINFO, :: LPPROC_THREAD_ATTRIBUTE_LIST, :: LPOVERLAPPED,
	:: SECURITY_ATTRIBUTES, :: LPTHREAD_START_ROUTINE,
	SECURITY_ATTRIBUTES_SIZE_INT, SECURITY_ATTRIBUTES_SIZE_BYTES,
	SECURITY_ATTRIBUTES_nLength_INT_OFFSET,
	SECURITY_ATTRIBUTES_bInheritHandle_INT_OFFSET,
	PROCESS_INFORMATION_size_bytes, PROCESS_INFORMATION_hProcess_bytes_offset,
	PROCESS_INFORMATION_hThread_bytes_offset,
	STARTUPINFOEX_size_int, STARTUPINFOEX_size_bytes,
	STARTUPINFO_cb_int_offset, STARTUPINFO_hStdInput_int_offset,
	STARTUPINFO_hStdOutput_int_offset, STARTUPINFO_hStdError_int_offset,
	STARTUPINFOEX_lpAttributeList_int_offset,
	STARTUPINFO_set_dwFlags,
	PROC_THREAD_ATTRIBUTE_HANDLE_LIST,
	STARTF_USESTDHANDLES, EXTENDED_STARTUPINFO_PRESENT,
	HANDLE_FLAG_INHERIT, NULL, TRUE, INFINITE, STILL_ACTIVE,
	getProcessHeap, heapAlloc, heapFree,
	getLastError,
	createPipe, setHandleInformation, closeHandle,
	peekNamedPipe, readFile, writeFile,
	CreateThread, cancelSynchronousIo, getExitCodeThread, sleep,
	waitForSingleObject, waitForMultipleObjects,
	initializeProcThreadAttributeList, updateProcThreadAttribute,
	deleteProcThreadAttributeList,
	createProcessA, createProcessA_dir, createProcessA_noApplication,
	createProcessA_noApplication_dir, getExitCodeProcess
import qualified System._WinBase
import Text
import Text.GenJSON

import code from "systemprocess.o"

:: WritePipe =: WritePipe Int
:: ReadPipe =: ReadPipe Int

derive JSONEncode WritePipe, ReadPipe
derive JSONDecode WritePipe, ReadPipe

_openPipePair :: !Bool !*World -> (!MaybeOSError (Int, Int), !*World)
_openPipePair close_right_not_left w
	# (heap, w) = getProcessHeap w
	# (ptr, w) = heapAlloc heap 0 (IF_INT_64_OR_32 16 8) w
	| ptr == 0 = abort "heapAlloc failed"
	# (ok, w) = createPipe ptr (ptr + IF_INT_64_OR_32 8 4) securityAttributes 8192 w
	| not ok
		# (_, w) = heapFree heap 0 ptr w
		= getLastOSError w
	# (rEnd, ptr)  = readIntP ptr 0
	# (wEnd, ptr)  = readIntP ptr (IF_INT_64_OR_32 8 4)
	# (_, w) = heapFree heap 0 ptr w
    # (ok, w) = setHandleInformation (if close_right_not_left wEnd rEnd) HANDLE_FLAG_INHERIT 0 w
    | not ok
		= getLastOSError w
		= (Ok (rEnd, wEnd), w)
where
	securityAttributes =
		{ createArray SECURITY_ATTRIBUTES_SIZE_INT 0
		& [SECURITY_ATTRIBUTES_nLength_INT_OFFSET]        = SECURITY_ATTRIBUTES_SIZE_BYTES
		, [SECURITY_ATTRIBUTES_bInheritHandle_INT_OFFSET] = TRUE
		}

instance closePipe WritePipe
where
	closePipe :: !WritePipe !*World -> (!MaybeOSError (), !*World)
	closePipe (WritePipe pipe) w = closePipe` pipe w

instance closePipe ReadPipe
where
	closePipe :: !ReadPipe !*World -> (!MaybeOSError (), !*World)
	closePipe (ReadPipe pipe) w = closePipe` pipe w

closePipe` :: !Int !*World -> (!MaybeOSError (), !*World)
closePipe` pipe w
	# (res, w) = closeHandle pipe w
	| not res = getLastOSError w
	| otherwise = (Ok (), w)

_blockPipe :: !ReadPipe !*World -> (!MaybeOSError (), !*World)
_blockPipe (ReadPipe pipe) w
	# (ok, w) = readFile pipe NULL 0 NULL NULL w
	| not ok
		# (err, w) = getLastError w
		| err == 109 // broken pipe: see comments on _startProcess why we ignore this
			= (Ok (), w)
			= getLastOSError w
	| otherwise
		= (Ok (), w)

/* NB: Windows' WaitForMultipleObjects does not work on pipes. For this reason
 * we create threads for each ReadPipe. Each thread receives a pipe, on which
 * it does a ReadFile with an empty buffer (see systemprocess.c) to block on
 * the pipe. We wait on these threads. When the wait is done, we terminate all
 * threads.
 * The threads to read the pipes must be implemented in C to prevent them from
 * corrupting the Clean heap.
 */
_blockAnyPipe :: ![ReadPipe] !*World -> (!MaybeOSError (), !*World)
_blockAnyPipe pipes w
	# npipes = length pipes
	# (threads, w) = mapSt
		(\(ReadPipe p) w
			#! (handle,id,w) = CreateThread 0 4096 readFileInSeparateThreadAddress p 0 w
			-> (handle, w))
		pipes
		w
	# threads_arr = {t \\ t <- threads}
	#! (i, w) = waitForMultipleObjects npipes threads_arr False 0xffffffff w
	#! (errs,w) = mapSt stopThread threads w // Cancel all ReadPipe calls so that the threads exit
	| any isError errs
		= (hd (filter isError errs), w)
	| 0 <= i && i < npipes
		= (Ok (), w)
	| 0x80 <= i && i < 0x80+npipes
		= abort "_blockAnyPipe: waitForMultipleObjects returned WAIT_ABANDONED"
	| i == 0x102 // WAIT_TIMEOUT; should not occur with 0xffffffff as timeout
		= abort "_blockAnyPipe: waitForMultipleObjects returned WAIT_TIMEOUT"
	| i == 0xffffffff // WAIT_FAILED
		= getLastOSError w
	| otherwise
		= abort ("_blockAnyPipe: waitForMultipleObjects returned unknown response value "+++toString i)
where
	readFileInSeparateThreadAddress = IF_INT_64_OR_32 addr64 addr32
	where
		addr64 :: Int
		addr64 = code {
			pushLc readFileInSeparateThread
		}
		addr32 :: Int
		addr32 = code {
			pushLc readFileInSeparateThread@4
		}

	/**
	 * After one of the threads has returned, we need to stop the other threads
	 * gracefully. We do this the following way:
	 * 1. If the thread has an exit code, we're done.
	 * 2. Cancel any synchronous IO.
	 * 3. If there was no IO to cancel, the thread probably didn't read the
	 *    `ReadFile` yet (or it passed the `ReadFile`, but did not return yet).
	 *    We repeat from step 1.
	 * 3. If there was IO to cancel, the thread will return soon. Repeat from
	 *    step 1.
	 */
	stopThread h w
		# (ok,r,w) = getExitCodeThread h w
		| not ok = getLastOSError w
		| r <> 259 // not STILL_ACTIVE
			# (ok,w) = closeHandle h w
			| not ok
				= getLastOSError w
				= (Ok (), w)
		# (ok,w) = cancelSynchronousIo h w
		| not ok
			# (e,w) = getLastError w
			| e == 1168 // ERROR_NOT_FOUND: there was no IO request to cancel
				= stopThread h (sleep 10 w)
				= getLastOSError w
			= stopThread h (sleep 10 w)

_peekPipe :: !ReadPipe !*World -> (!MaybeOSError Int, !*World)
_peekPipe (ReadPipe pipe) w
	# (heap, w) = getProcessHeap w
	# (nBytesPtr, w) = heapAlloc heap 0 4 w
	| nBytesPtr == 0 = abort "heapAlloc failed"
	# (ok, w) = peekNamedPipe pipe NULL 0 NULL nBytesPtr NULL w
	# (nBytes, nBytesPtr) = readIntP nBytesPtr 0
	# nBytes = nBytes bitand 0xffffffff
	# (_, w) = heapFree heap 0 nBytesPtr w
	| not ok
		# (err, w) = getLastError w
		| err == 109 // broken pipe: see comments on _startProcess why we ignore this
			= (Ok 0, w)
			= getLastOSError w
	| otherwise
		= (Ok nBytes, w)

_readPipeNonBlocking :: !ReadPipe !Int !*World -> (!MaybeOSError String, !*World)
_readPipeNonBlocking (ReadPipe pipe) nBytes w
	# (heap, w) = getProcessHeap w
	# (buf, w) = heapAlloc heap 0 nBytes w
	| buf == 0 = abort "heapAlloc failed"
	# (ok, w) = readFile pipe buf nBytes NULL NULL w
	| not ok
		# (_, w) = heapFree heap 0 buf w
		# (err, w) = getLastError w
		| err == 109 // broken pipe: see comments on _startProcess why we ignore this
			= (Ok "", w)
			= getLastOSError w
	# (str, buf) = readP (\ptr -> derefCharArray ptr nBytes) buf
	# (_, w) = heapFree heap 0 buf w
	= (Ok str, w)

_writePipe :: !String !WritePipe !*World -> (!MaybeOSError (), !*World)
_writePipe data (WritePipe pipe) w
	# (ok, w) = writeFile pipe data (size data) NULL NULL w
	| ok
		= (Ok (), w)
		= getLastOSError w

_equalPipe :: !WritePipe !ReadPipe -> Bool
_equalPipe (WritePipe x) (ReadPipe y) = x == y

_startProcess ::
	!FilePath ![String] !(?String)
	!(?((Int,Int), (Int,Int), (Int,Int)))
	!*World -> (!MaybeOSError (ProcessHandle, ?ProcessIO), !*World)
_startProcess exe args dir mbPipes w
	# (heap,w) = getProcessHeap w
	# (startupInfo,freeablePointers,mbAttributeList,w) = makeStartupInfo heap w
	/* NB: Windows will not search for `exe` in `PATH`. To get the desired
	 * behaviour, we check if `exe` exists as a regular file. If it does, we
	 * call `CreateProcessA` with that file as the application name. If it does
	 * not, we specify `NULL` as the application name. The behaviour of
	 * `CreateProcessA` is that it will search for the first argument of the
	 * command line in `PATH` (with/without `.exe`). However, we have to quote
	 * this argument; otherwise, e.g. `my dir\program` will match `my.exe`. See
	 * the `CreateProcessA` documentation for details. */
	# (exeExists, w) = fileExists exe w
	# commandLine = packString (foldr (\a b -> concat3 a " " b) "" (map escape [if exeExists exe (concat3 "\"" exe "\""):args]))
	# (processInformation,w) = heapAlloc heap 0 PROCESS_INFORMATION_size_bytes w
	| processInformation == 0 = abort "heapAlloc failed"
	# (ok, w) = case dir of
		?Just dir
			| exeExists -> createProcessA_dir (packString exe) commandLine 0 0 True EXTENDED_STARTUPINFO_PRESENT 0 (packString dir) startupInfo processInformation w
			| otherwise -> createProcessA_noApplication_dir NULL commandLine 0 0 True EXTENDED_STARTUPINFO_PRESENT 0 (packString dir) startupInfo processInformation w
		?None
			| exeExists -> createProcessA (packString exe) commandLine 0 0 True EXTENDED_STARTUPINFO_PRESENT 0 NULL startupInfo processInformation w
			| otherwise -> createProcessA_noApplication NULL commandLine 0 0 True EXTENDED_STARTUPINFO_PRESENT 0 NULL startupInfo processInformation w
	# w = seqSt (\p w -> snd (heapFree heap 0 p w)) freeablePointers w
	| not ok
		# (_,w) = heapFree heap 0 processInformation w
		= getLastOSError w
	# (hProcess,processInformation) = readIntP processInformation PROCESS_INFORMATION_hProcess_bytes_offset
	  (hThread,processInformation) = readIntP processInformation PROCESS_INFORMATION_hThread_bytes_offset
	  processHandle = {processHandle=hProcess, threadHandle=hThread}
	  (_,w) = heapFree heap 0 processInformation w
	| isNone mbPipes = (Ok (processHandle, ?None), w)
	/* NB: the pipeStdInOut, pipeStdOutIn, and pipeStdErrIn handles are
	 * inherited by the child process. We must close the handles that the
	 * parent process still has, to avoid ReadFile to block when the child
	 * process has terminated. When the child terminates and the pipe is empty,
	 * ReadFile will return error 109 (broken pipe), which is how we know that
	 * the reading is done. For details, see:
	 * https://docs.microsoft.com/en-us/windows/win32/ipc/pipe-handle-inheritance
	 */
	# (_,w) = closeHandle pipeStdInOut w
	# (_,w) = closeHandle pipeStdOutIn w
	# (_,w) = closeHandle pipeStdErrIn w
	# w = deleteProcThreadAttributeList (fromJust mbAttributeList) w
	=
		( Ok
			( processHandle
			, ?Just
				{ stdIn  = WritePipe pipeStdInIn
				, stdOut = ReadPipe  pipeStdOutOut
				, stdErr = ReadPipe  pipeStdErrOut
				}
			)
		, w
		)
where
	// We only evaluate this when mbPipes is ?Just:
	((pipeStdInOut, pipeStdInIn), (pipeStdOutOut, pipeStdOutIn), (pipeStdErrOut, pipeStdErrIn)) = fromJust mbPipes

	makeStartupInfo heap w
		| isNone mbPipes =
			( STARTUPINFO_set_dwFlags STARTF_USESTDHANDLES
				{ createArray STARTUPINFOEX_size_int 0
				& [STARTUPINFO_cb_int_offset] = STARTUPINFOEX_size_bytes
				}
			, [], ?None, w
			)
		# (attributeListSize_p, w) = heapAlloc heap 0 (IF_INT_64_OR_32 8 4) w
		| attributeListSize_p == 0 = abort "heapAlloc failed"
		# (_,w) = initializeProcThreadAttributeList NULL 1 0 attributeListSize_p w // returns an error by design
		# (attributeList, w) = heapAlloc heap 0 (readInt attributeListSize_p 0) w
		| attributeList == 0 = abort "heapAlloc failed"
		# (rgHandlesToInherit, w) = heapAlloc heap 0 (IF_INT_64_OR_32 24 12) w
		| rgHandlesToInherit == 0 = abort "heapAlloc failed"
		# rgHandlesToInherit = writeInt rgHandlesToInherit 0 pipeStdInOut
		  rgHandlesToInherit = writeInt rgHandlesToInherit (IF_INT_64_OR_32 8 4) pipeStdOutIn
		  rgHandlesToInherit = writeInt rgHandlesToInherit (IF_INT_64_OR_32 16 8) pipeStdErrIn
		# (ok,w) = initializeProcThreadAttributeList attributeList 1 0 attributeListSize_p w
		| not ok = abort "initializeProcThreadAttributeList failed"
		# (ok,w) = updateProcThreadAttribute attributeList 0 PROC_THREAD_ATTRIBUTE_HANDLE_LIST rgHandlesToInherit (IF_INT_64_OR_32 24 12) NULL NULL w
		| not ok = abort "updateProcThreadAttribute failed"
		# startupInfo =
			STARTUPINFO_set_dwFlags STARTF_USESTDHANDLES
			{ createArray STARTUPINFOEX_size_int 0
			& [STARTUPINFO_cb_int_offset] = STARTUPINFOEX_size_bytes
			, [STARTUPINFO_hStdInput_int_offset] = pipeStdInOut
			, [STARTUPINFO_hStdOutput_int_offset] = pipeStdOutIn
			, [STARTUPINFO_hStdError_int_offset] = pipeStdErrIn
			, [STARTUPINFOEX_lpAttributeList_int_offset] = attributeList
			}
		= (startupInfo, [attributeListSize_p, attributeList, rgHandlesToInherit], ?Just attributeList, w)

	escape :: !String -> String
	escape s
		| indexOf " " s == -1
			= s
		| size s >= 2 && s.[0] == '"' && (s.[size s - 1] == '"')
			= s
			= concat3 "\"" s "\""

_startProcessPty ::
	!FilePath ![String] !(?String) !ProcessPtyOptions
	!*World -> (!MaybeOSError (ProcessHandle, ProcessIO), !*World)
_startProcessPty _ _ _ _ _ = abort "_startProcessPty"

_checkProcess :: !ProcessHandle !*World -> (!MaybeOSError (?Int), !*World)
_checkProcess handle=:{processHandle} w
	# (ok,exitCode,w) = getExitCodeProcess processHandle w
	| not ok = getLastOSError w
	| exitCode == STILL_ACTIVE = (Ok ?None, w)
	# (mbError,w) = closeProcessHandle handle w
	= (Ok (?Just exitCode), w)

_waitForProcess :: !ProcessHandle !*World -> (!MaybeOSError Int, !*World)
_waitForProcess handle=:{processHandle} w
	# (res,w) = waitForSingleObject processHandle INFINITE w
	# (ok,exitCode,w) = getExitCodeProcess processHandle w
	| not ok = getLastOSError w
	# (mbError,w) = closeProcessHandle handle w
	= (Ok exitCode, w)

_terminateProcess :: !ProcessHandle !Int !*World -> (!MaybeOSError (), !*World)
_terminateProcess hProc=:{processHandle} exitCode w
	# (ok, w) = 'System._WinBase'.terminateProcess processHandle exitCode w
	= closeProcessHandle hProc w

closeProcessHandle :: !ProcessHandle !*World -> (MaybeOSError (), *World)
closeProcessHandle handle world
	# (ok,world) = closeHandle handle.processHandle world
	| not ok = getLastOSError world
	# (ok, world) = closeHandle handle.threadHandle world
	| not ok = getLastOSError world
	= (Ok (), world)
