implementation module iTasks.Extensions.GIS.Leaflet

import StdEnv
import iTasks
import iTasks.UI.Definition, iTasks.UI.Editor, iTasks.UI.Editor.Modifiers
import StdMisc, Data.Tuple, Data.Error, Data.Func, Text, Data.Functor
from Data.List import concatMap, partition
import qualified Data.Map as DM
import Data.Maybe
import qualified Data.Set as Set
import Data.Set.GenJSON
import Text.HTML
from Text.Encodings.Base64 import base64Encode
from iTasks.UI.Editor.Common import diffChildren, :: ChildUpdate (..)
import ABC.Interpreter.JavaScript
import StdArray

LEAFLET_JS              :== "leaflet/leaflet.js"
LEAFLET_CSS             :== "leaflet/leaflet.css"
LEAFLET_JS_FIX          :== "leaflet-fix.js"
LEAFLET_JS_WINDOW       :== "leaflet-window.js"
LEAFLET_JS_MORPHDOM     :== "leaflet/morphdom-umd.min.js" // required for "leaflet-window.js"
LEAFLET_JS_MARKERS      :== "leaflet-markers.js"
LEAFLET_JS_EDITABLE     :== "leaflet/Leaflet.Editable.js"
LEAFLET_CSS_WINDOW      :== "leaflet-window.css"
PIXI_OVERLAY_JS         :== "leaflet/L.PixiOverlay.min.js"
PIXI_JS                 :== "leaflet/pixi.min.js"
PIXI_POPUP_CSS          :== "pixi-popup.css"
LEAFLET_MOUSEPOS_JS     :== "leaflet/L.Control.MousePosition.js"
LEAFLET_MOUSEPOS_CSS    :== "leaflet/L.Control.MousePosition.css"
LEAFLET_DISTMEASURE_JS  :== "leaflet/Leaflet.PolylineMeasure.js"
LEAFLET_DISTMEASURE_CSS :== "leaflet/Leaflet.PolylineMeasure.css"
LEAFLET_DMS_FORM        :== "leaflet-dms.js"
LEAFLET_LEGEND_JS       :== "leaflet-legend.js"

DEFAULT_WINDOW_POS   :== (50, 50)
MAX_WINDOW_OFFSET    :== 100

:: IconOptions =
	{ iconUrl   :: !String
	, iconSize  :: ![Int]
	}

derive JSONEncode IconOptions

derive gToJS MapOptions, LeafletLatLng

leafletObjectIdOf :: !LeafletObject -> LeafletObjectID
leafletObjectIdOf (Marker m)    = m.markerId
leafletObjectIdOf (Polyline p)  = p.polylineId
leafletObjectIdOf (Polygon p)   = p.polygonId
leafletObjectIdOf (Circle c)    = c.circleId
leafletObjectIdOf (Rectangle r) = r.rectangleId
leafletObjectIdOf (Window w)    = w.windowId

leafletPointsOf :: !LeafletObject -> [LeafletLatLng]
leafletPointsOf (Marker m) = [m.position]
leafletPointsOf (Polyline l) = l.LeafletPolyline.points
leafletPointsOf (Polygon p) = p.LeafletPolygon.points
leafletPointsOf (Circle c) = [c.LeafletCircle.center]
leafletPointsOf (Rectangle {LeafletRectangle | bounds=b}) = [b.southWest, b.northEast]
leafletPointsOf (Window w) = []

leafletBoundingRectangleOf :: ![LeafletObject] -> LeafletBounds
leafletBoundingRectangleOf objects
	| isEmpty points = defaultValue
	| otherwise =
		{ southWest = {lat=minList lats, lng=minList lngs}
		, northEast = {lat=maxList lats, lng=maxList lngs}
		}
where
	points = concatMap leafletPointsOf objects
	lats = [p.lat \\ p <- points]
	lngs = [p.lng \\ p <- points]

:: LeafletEdit
	= LDSetManualPerspective
	//Current state
	| LDSetZoom         !Int
	| LDSetCenter       !LeafletLatLng
	| LDSetBounds       !LeafletBounds
	//Updating windows
	| LDRemoveWindow    !LeafletObjectID
	| LDClosePopup      !LeafletObjectID
	| LDUpdateObject    !LeafletObjectID !LeafletObjectUpdate
	//Events
	| LDMapClick        !LeafletLatLng
	| LDMapDblClick     !LeafletLatLng
	| LDMarkerClick     !LeafletObjectID
	| LDHtmlEvent       !String

:: LeafletObjectUpdate
	= UpdatePolyline  ![LeafletLatLng]
	| UpdatePolygon   ![LeafletLatLng]
	| UpdateCircle    !LeafletLatLng !Real
	| UpdateRectangle !LeafletBounds

svgIconURL :: !SVGElt !(!Int,!Int) -> String
svgIconURL svgelt (width,height) = "data:image/svg+xml;base64," +++ base64Encode svg
where
    svg = concat ["<svg xmlns=\"http://www.w3.org/2000/svg\" width=\""
		, toString width, "\" height=\"", toString height, "\">", toString svgelt, "</svg>"]

openStreetMapTiles :: TileLayer
openStreetMapTiles =
	{ url         = "http://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png"
	, attribution = ?Just $ Html "&copy; <a href=\"https://www.openstreetmap.org/copyright\">OpenStreetMap</a>"
	}

leafletEditor :: !Bool -> Editor LeafletMap LeafletMap
leafletEditor viewOnly =
	leafEditorToEditor $
		leafletEditor`
			{ viewOnly = viewOnly
			, viewMousePosition = False
			, viewMeasureDistance = False
			, viewLegend = ?None
			, attributionControl = True
			, zoomControl = True
			}
			(const id)

leafletEditor` :: !MapOptions !(JSVal *JSWorld -> *JSWorld) -> LeafEditor [LeafletEdit] LeafletMap LeafletMap LeafletMap
leafletEditor` mapOptions=:{viewOnly, viewMousePosition, viewMeasureDistance, viewLegend} postInitUI
	= withClientSideInitOnLeafEditor initUI
	{ LeafEditor
	| onReset        = onReset
	, onEdit         = onEdit
	, onRefresh      = onRefresh
	, writeValue     = writeValue
	}
where
	onReset attr mbval vst
		# val=:{LeafletMap|perspective,tilesUrls,objects,icons} = fromMaybe gDefault{|*|} mbval
		# mapAttr = 'DM'.fromList
			[("perspective", encodePerspective perspective)
			,("tilesUrls"
			 , JSONArray $
				(\tile ->
					JSONObject
						[ ("url", toJSON tile.url)
						: maybeToList $ (\attr -> ("attribution", toJSON $ toString attr)) <$> tile.attribution
						]
				) <$>
					tilesUrls
			 )
			,("icons", JSONArray [toJSON (iconId,{IconOptions|iconUrl=iconUrl,iconSize=[w,h]}) \\ {iconId,iconUrl,iconSize=(w,h)} <- icons])
			,("viewMousePosition", toJSON viewMousePosition)
			,("viewMeasureDistance", toJSON viewMeasureDistance)
			,("viewLegend", toJSON (isJust viewLegend))
			:[("fitbounds", attr) \\ attr <- fitBoundsAttribute val]
			]
		# (markers, others) = partition (\o -> o=:(Marker _)) objects
		// We have to convert the title to a string here, because the toString function will no longer be available once
		// we call the leafletCreateMarkers JS function
		# markers = map
			(\(Marker m) -> {LeafletMarker | m & title = fmap (Html o toString) m.LeafletMarker.title})
			markers
		# others = map encodeUI others
		# attr = 'DM'.unions
			[ mapAttr
			, sizeAttr (ExactSize 500) (ExactSize 150)
			, 'DM'.singleton "markers" $ toJSON markers
			, attr
			]
		= (Ok (uiac UIHtmlView attr others, val, ?None), vst)

	encodePerspective :: !LeafletPerspective -> JSONNode
	encodePerspective (CenterAndZoom center zoom) = JSONArray
		[ JSONString "CenterAndZoom"
		, JSONArray [JSONReal center.lat, JSONReal center.lng]
		, JSONInt zoom
		]
	encodePerspective (FitToBounds options _) = JSONArray
		[ JSONString "FitToBounds"
		, toJSON options
		]

	encodeUI (Polyline o)
		# (JSONObject attr) = toJSON o
		= uia UIData ('DM'.fromList [("type",JSONString "polyline"):attr])
	encodeUI (Polygon o)
		# (JSONObject attr) = toJSON o
		= uia UIData ('DM'.fromList [("type",JSONString "polygon") : attr])
	encodeUI (Circle o)
		# (JSONObject attr) = toJSON o
		= uia UIData ('DM'.fromList [("type",JSONString "circle"): attr])
	encodeUI (Rectangle o)
		# (JSONObject attr) = toJSON o
		= uia UIData ('DM'.fromList [("type",JSONString "rectangle") : attr])
	encodeUI (Window o)
		# (JSONObject attr) = toJSON o
		# dataMap = 'DM'.fromList [("type",JSONString "window"): attr]
		// translate HtmlTag to HTML code
		# dataMap` = 'DM'.put "content" (JSONString (toString o.content)) dataMap
		= uia UIData dataMap`

	initUI {FrontendEngineOptions|serverDirectory} me world
		# (jsInitDOM,world) = jsWrapFun (initDOM me) me world
		# (viewMousePos, world) = me .# "attributes.viewMousePosition" .? world
		# viewMousePos = jsValToBool viewMousePos == ?Just True
		# (viewMeasureDistance, world) = me .# "attributes.viewMeasureDistance" .? world
		# viewMeasureDistance = jsValToBool viewMeasureDistance == ?Just True
		# (viewLegend, world) = me .# "attributes.viewLegend" .? world
		# viewLegend = jsValToBool viewLegend == ?Just True
		# world = addCSSFromUrl (serverDirectory+++PIXI_POPUP_CSS) ?None world
		# world = addCSSFromUrl (serverDirectory+++LEAFLET_CSS_WINDOW) ?None world
		# world = if viewMousePos (addCSSFromUrl (serverDirectory+++LEAFLET_MOUSEPOS_CSS) ?None world) world
		# world = if viewMeasureDistance (addCSSFromUrl (serverDirectory+++LEAFLET_DISTMEASURE_CSS) ?None world) world
		// We can only load the leaflet javascript if the leaflet css has already been loaded
		// hence we perform a callback on the add
		# (cb,world) =
			jsWrapFun (loadJS serverDirectory jsInitDOM viewMousePos viewMeasureDistance viewLegend me) me world
		# world = addCSSFromUrl (serverDirectory+++LEAFLET_CSS) (?Just cb) world
		= world
	loadJS serverDirectory jsInitDOM viewMousePos viewMeasureDistance viewLegend me _ world
		# world = addJSFromUrls
			([ serverDirectory+++LEAFLET_JS
			 , serverDirectory+++LEAFLET_JS_EDITABLE
			 , serverDirectory+++PIXI_JS
			 , serverDirectory+++PIXI_OVERLAY_JS
			 , serverDirectory+++LEAFLET_JS_FIX
			 , serverDirectory+++LEAFLET_JS_MARKERS
			 , serverDirectory+++LEAFLET_JS_WINDOW
			 , serverDirectory+++LEAFLET_JS_MORPHDOM
			 ] ++ optionalUrls
			) (?Just jsInitDOM) me world
		= world
	where
		optionalUrls :: [String]
		optionalUrls
			=  if viewMousePos [serverDirectory +++ LEAFLET_DMS_FORM, serverDirectory +++ LEAFLET_MOUSEPOS_JS] [] ++
			   if viewMeasureDistance [serverDirectory +++ LEAFLET_DISTMEASURE_JS] [] ++
			   if viewLegend [serverDirectory +++ LEAFLET_LEGEND_JS] []

	initDOM me args world
		# (l,world)         = jsGlobal "L" .? world
		# (domEl,world)     = me .# "domEl" .? world
		//Create the map; set `editable` option to `true` as shapes are possibly editable (causing exceptions otherwise)
		# (mapOpts, world)  = jsNew (jsGlobal "Object") (toJS mapOptions) world
		# world             = mapOpts .# "editable" .= True $ world
		# (mapObj,world)    = (l .# "map" .$ (domEl, mapOpts)) world
		# world             = (me .# "map" .= mapObj) world
		//Set perspective
		# world             = setMapPerspective me True (me .# "attributes.perspective") world
		//Add icons
		# world             = setMapIcons me (me .# "attributes.icons") world
		//Create tile layer
		# (tilesUrls,world) = me .# "attributes.tilesUrls" .? world
		# world             = forall (addMapTilesLayer me mapObj) tilesUrls world
		//Synchronize lat/lng bounds to server (they depend on the size of the map in the browser)
		# world             = syncCurrentState me False world
		// Create global PIXI Container
		# (container, world) = jsNew "PIXI.Container" () world
		# world              = (me .# "container" .= container) world
		//Add initial objects
		# (overlay, world) = createPixiOverlay me mapObj world
		# world            = (me .# "overlay" .= overlay) world
		# (objects, world) = me .# "children" .? world
		# (windows, world) = jsNew "Map" () world
		# world            = (me .# "windows" .= windows) world
		# world            = (me .# "windowOffset" .= 0) world
		# world            = createMapObjects me mapObj objects world
		//Add mouse position control
		# (viewMousePos, world) = me .# "attributes.viewMousePosition" .? world
		# viewMousePos          = jsValToBool viewMousePos == ?Just True
		# world =
			if viewMousePos
				(snd $ l .# "control.mousePosition({formatter: L.dmsFormatter}).addTo" .$ mapObj $ world)
				world
		//Add distance measurement control
		# (viewMeasureDistance, world)  = me .# "attributes.viewMeasureDistance" .? world
		# viewMeasureDistance = jsValToBool viewMeasureDistance == ?Just True
		# unitControlTitleRecord = jsRecord
			[ "text" :> "Change units"
			, "metres" :> "metres" // No capitalization on purpose.
			, "landmiles" :> "land miles"
			, "nauticalmiles" :> "nautical miles"
			]
		# optionsRecord = jsRecord
			[ "measureControlTitleOn"   :> "Start measuring distances"
			, "measureControlTitleOff"  :> "Stop measuring distances"
			, "clearControlTitle"       :> "Clear measurements"
			, "unitControlTitle"        :> unitControlTitleRecord
			, "showClearControl"        :> True
			, "showUnitControl"         :> True
			, "clearMeasurementsOnStop" :> False
			, "measureControlLabel"     :> "&#8596;" // Right arrow icon for turning on/off measurement.
			, "unit"                    :> "nauticalmiles"
			, "backgroundColor"         :> "#00cc00"
			, "measureControlClasses"   :> ["leaflet-interactive"] // Makes cursor have same style as other buttons.
			, "clearControlClasses"     :> ["leaflet-interactive"]
			, "unitControlClasses"      :> ["leaflet-interactive"]
			]
		# world =
			if viewMeasureDistance
				(snd $ jsCall (l .# "control" .# "polylineMeasure") optionsRecord .# "addTo" .$ mapObj $ world)
				world
		//Add legend control
		# (viewLegend, world)  = me .# "attributes.viewLegend" .? world
		# viewLegend = jsValToBool viewLegend == ?Just True
		# optionsRecord = (jsRecord ["url" :> fromJust mapOptions.viewLegend, "label" :> "?"])
		# world = if viewLegend (snd $ jsCall (l .# "control" .# "legend") optionsRecord .# "addTo" .$ mapObj $ world) world
		//Add event handlers
		# (cb,world)       = jsWrapFun (\a w -> onResize me w) me world
		# world            = (me .# "onResize" .= cb) world
		# (cb,world)       = jsWrapFun (\a w -> onShow me w) me world
		# world            = (me .# "onShow" .= cb) world
		# (cb,world)       = jsWrapFun (\a w -> onAttributeChange me a w) me world
		# world            = (me .# "onAttributeChange" .= cb) world
		# (cb,world)       = jsWrapFun (\a w -> onAfterChildInsert me a w) me world
		# world            = (me .# "afterChildInsert" .= cb) world
		# (cb,world)       = jsWrapFun (\a w -> onBeforeChildRemove me a w) me world
		# world            = (me .# "beforeChildRemove" .= cb) world
		# (cb,world)       = jsWrapFun (\a w -> onViewportChange me w) me world
		# world            = (me .# "onViewportChange" .= cb) world
		# (vp,world)       = (me .# "getViewport" .$ ()) world
		# world            = (vp .# "addChangeListener" .$! me) world
		# (cb,world)       = jsWrapFun (\a w -> beforeRemove me w) me world
		# world            = (me .# "beforeRemove" .= cb) world
		# (cb,world)       = jsWrapFun (\a w -> onHtmlEvent me a w) me world
		# world            = (me .# "onHtmlEvent" .= cb) world
		# world = case viewOnly of
            True
				= world
			False
				# (cb,world)       = jsWrapFun (onMapMoveEnd me) me world
				# world            = (mapObj .# "addEventListener" .$! ("moveend",cb)) world
				# (cb,world)       = jsWrapFun (onMapMoveStart me) me world
				# world            = (mapObj .# "addEventListener" .$! ("movestart",cb)) world
				# (cb,world)       = jsWrapFun (onMapClick False me) me world
				# world            = (mapObj .# "addEventListener" .$! ("click",cb)) world
				# (cb,world)       = jsWrapFun (onMapClick True me) me world
				# world            = (mapObj .# "addEventListener" .$! ("dblclick",cb)) world
				= world
		# world = postInitUI mapObj world
		= world

	syncCurrentState me manualP world
		| viewOnly = world
		# (taskId,world)    = me .# "attributes.taskId" .? world
		# (editorId,world)  = me .# "attributes.editorId" .? world
		# mapObj            = me .# "map"
		# (bounds,world)    = getMapBounds mapObj world
		# (center,world)    = getMapCenter mapObj world
		# (zoom,world)      = getMapZoom mapObj world
		# edit = toJSON [LDSetBounds bounds,LDSetCenter center,LDSetZoom zoom: if manualP [LDSetManualPerspective] []]
		# world             = (me .# "doEditEvent" .$! (taskId,editorId,edit)) world
		= world

	onResize me world
		# world             = (me .# "map.invalidateSize" .$! ()) world
		= world

	onShow me world
		# world             = (me .# "map.invalidateSize" .$! ()) world
		# world             = setMapPerspective me True (me .# "attributes.perspective") world
		= world

	onMapMoveEnd me args world
		# (ignore,world) = me .# "attributes.ignoreNextOnmoveend" .?? (False, world)
		# world          = me .# "attributes.ignoreNextOnmoveend" .= False $ world
		| ignore = world
		= syncCurrentState me True world

	onMapMoveStart me args world
		# (tooltip, world) = (jsDocument .# "getElementById" .$ "tooltip") world
		| jsIsNull tooltip = world
		# world = (jsDocument .# "body.removeChild" .$! tooltip) world
		= world

	onMapClick double me args world
		# (taskId,world)    = me .# "attributes.taskId" .? world
		# (editorId,world)  = me .# "attributes.editorId" .? world
		# (clickPos,world)  = args.[0] .# "latlng" .? world
		# (position,world)  = toLatLng clickPos world
		# edit              = toJSON [if double LDMapDblClick LDMapClick position]
		# world             = (me .# "doEditEvent" .$! (taskId,editorId,edit)) world
		= world

	onMarkerClick me markerId args world
		# (taskId,world)    = me .# "attributes.taskId" .? world
		# (editorId,world)  = me .# "attributes.editorId" .? world
		# edit              = toJSON [LDMarkerClick markerId]
		# world             = (me .# "doEditEvent" .$! (taskId,editorId,edit)) world
		= world

	onAttributeChange me args world = case fromJS "" args.[0] of
		"perspective" -> setMapPerspective me False args.[1] world
		"icons"       -> setMapIcons me args.[1] world
		"fitbounds"   -> fitBounds me False args.[1] (me .# "attributes.perspective" .# 1) world
		"markers"
			# (taskId,world)    = me .# "attributes.taskId" .? world
			# (editorId,world)  = me .# "attributes.editorId" .? world
			-> (jsGlobal "leafletCreateMarkers" .$! (me, taskId, editorId, args.[1])) world
		_             -> jsTrace "unknown attribute change" world

	onHtmlEvent me args world
		# (taskId,world)    = me .# "attributes.taskId" .? world
		# (editorId,world)  = me .# "attributes.editorId" .? world
		= case (jsValToString args.[0]) of
			?Just event
				# edit = toJSON [LDHtmlEvent event]
				# world = (me .# "doEditEvent" .$! (taskId,editorId,edit)) world
				= world
			_	= world


	onAfterChildInsert me args world
		# (l, world)        = jsGlobal "L" .? world
		# (mapObj,world)    = me .# "map" .? world
		= createMapObject me mapObj args.[1] l world

	// Called by itasks-core.js when a child of the leaflet editor is removed
	onBeforeChildRemove me args world
		# (type,world) = args.[1] .# "attributes.type" .? world
		= case jsValToString type of
			?Just "window" = removeWindow me args world
			?Just "marker" = jsTrace "This should not be reached, markers are no longer children" world
			?Just _ = removeOther  me args world
			_ = world
	where
		removeWindow me args world
			# (layer, world) = args.[1] .# "layer" .? world
			# windows = me .# "windows"
			// We must remove the window from the map before we continue, this makes sure we do not try
			// to access the window when the `markers` attribute is updated.
			# world = (layer .# "removeFromMap" .$! windows) world
			= removeOther me args world
		removeOther me args world
			# (layer,world)     = args.[1] .# "layer" .? world
			# world             = (me .# "map.removeLayer" .$! layer) world
			// for windows, based on control class
			# (removeMethod, world) = layer .# "remove" .? world
			| not (jsIsUndefined removeMethod) = (layer .# "remove" .$! ()) world
			// for all other objects
			= world

	onViewportChange me world
		# world             = (me .# "map.invalidateSize" .$! ()) world
		= world

	beforeRemove me world
		# (vp,world) = (me .# "getViewport" .$ ()) world
		# world      = (vp .# "removeChangeListener" .$! me) world
		# world      = (me .# "container" .# "destroy" .$! True) world
		# world      = (me .# "map" .# "remove" .$! ()) world
		= world

	// Called when the x button on a window is pressed
	onWindowRemove me windowId _ world
		// remove children from iTasks component
		# (children,world)  = me .# "children" .? world
		# world             = forall (removeWindow me) children world
		// send edit event to server
		# (taskId,world)    = me .# "attributes.taskId" .? world
		# (editorId,world)  = me .# "attributes.editorId" .? world
		# edit              = toJSON [LDRemoveWindow windowId]
		# world             = (me .# "doEditEvent" .$! (taskId,editorId,edit)) world
		= world
	where
		removeWindow me idx layer world
			# (layerWindowId, world)  = layer .# "attributes.windowId" .?? ("", world)
			| LeafletObjectID layerWindowId == windowId
				# world = (me .# "removeChild" .$! idx) world
				# windows = me .# "windows"
				# world = (layer .# "layer.removeFromMap" .$! windows) world
				= world
			= world

	//Map object access
	toLatLng obj world
		# (lat,world)     = obj .# "lat" .?? (0.0, world)
		# (lng,world)     = obj .# "lng" .?? (0.0, world)
		= ({LeafletLatLng|lat=lat,lng=lng}, world)

	toBounds bounds env
		# (sw,env)          = (bounds .# "getSouthWest" .$ ()) env
		# (ne,env)          = (bounds .# "getNorthEast" .$ ()) env
		# (swpos,env)       = toLatLng sw env
		# (nepos,env)       = toLatLng ne env
		= ({southWest=swpos,northEast=nepos},env)

	getMapBounds mapObj env
		# (bounds,env) = (mapObj .# "getBounds" .$ ()) env
		= toBounds bounds env

	getMapZoom mapObj world
		= (mapObj .# "getZoom" .$? ()) (1, world)

	getMapCenter mapObj world
		# (center,world) = (mapObj .# "getCenter" .$ ()) world
		= toLatLng center world

	setMapPerspective me initialization attr world
		# (type,world) = attr .# 0 .?? ("", world)
		= case type of
			"CenterAndZoom"
				# world = (me .# "attributes.ignoreNextOnmoveend" .= True) world
				# world = (me .# "map.setView" .$! (attr .# 1, attr .# 2)) world
				-> syncCurrentState me False world
			"FitToBounds"
				# world = fitBounds me initialization (me .# "attributes.fitbounds") (attr .# 1) world
				-> syncCurrentState me False world
			_
				-> jsTraceVal attr (jsTrace "failed to set perspective" world)

	fitBounds me initialization bounds options world
		// If we fit the bounds this will cause a `onmoveend` event, which has to be ignored.
		// Only when the map is initialized the event will not be fired.
		# world = if initialization world (me .# "attributes.ignoreNextOnmoveend" .= True $ world)
		# mapFitBoundsFuncName = if initialization "map.fitBounds" "map.flyToBounds"
		= (me .# mapFitBoundsFuncName .$! (bounds, options)) world

	addMapTilesLayer me mapObj _ tiles world
		# (tilesUrl, world)    = tiles .# "url" .? world
		| jsIsNull tilesUrl    = world
		# (attribution, world) = tiles .# "attribution" .? world
		# (options, world)     = jsEmptyObject world
		# world                = (options .# "attribution" .= attribution) world
		# (l, world)           = jsGlobal "L" .? world
		# (layer,world)        = l .# "tileLayer" .$ (tilesUrl, options) $ world
		# world                = (layer .# "addTo" .$! mapObj) world
		= world

	setMapIcons me icons world
		# (l, world)           = jsGlobal "L" .? world
		# (iconIndex,world)    = jsEmptyObject world
		# world                = (me .# "icons" .= iconIndex) world
		# (textureIndex,world) = jsEmptyObject world
		# world                = (me .# "textures" .= textureIndex) world
		// use a loader to make sure initial markers are only drawn after all textures are loaded
		# (loader, world)      = jsNew "PIXI.Loader" () world
		# world                = forall (createMapIcon l loader iconIndex textureIndex) icons world
		# (taskId,world)       = me .# "attributes.taskId" .? world
		# (editorId,world)     = me .# "attributes.editorId" .? world
		# (markers,world)      = me .# "attributes.markers" .? world
		// load all textures and drawn initial markers when finished
		# (cb, world) = jsWrapFun (\_ -> jsGlobal "leafletCreateMarkers" .$! (me, taskId, editorId, markers)) me world
		# world                = loader.# "load" .$! (cb) $ world
		= world
	where
		createMapIcon :: !JSVal !JSVal !JSVal !JSVal x !JSVal !*JSWorld -> *JSWorld
		createMapIcon l loader iconIndex textureIndex _ def world
			# (iconId,world)   = def .# 0 .?? ("", world)
			# (iconSpec,world) = def .# 1 .? world
			# (icon,world)     = (l .# "icon" .$ iconSpec) world
			# world            = (iconIndex .# iconId .= icon) world
			# (url, world)     = icon .# "options.iconUrl" .? world
			# (cb, world)      = jsWrapFun (onTextureLoaded iconId) me world
			# world = loader .# "add" .$! (iconId, url, cb) $ world
			= world
		where
			onTextureLoaded :: !String !{!JSVal} !*JSWorld -> *JSWorld
			onTextureLoaded iconId args world
				# (texture, world) = args.[0] .# "texture" .? world
				# world            = textureIndex .# iconId .= texture  $ world
				= world

	createPixiOverlay me mapObj world
		# (container, world)   = me .# "container" .? world
		# world                = (me .# "scale" .= 0.0) world
		# (cb, world)          = jsWrapFun (\args w -> (jsGlobal "leafletPixiDraw" .$! (me, args.[0])) w) me world
		# (options, world)     = jsEmptyObject world
		# world                = (options .# "pane" .= "markerPane") world
		# (pixiOverlay, world) = (jsGlobal "L" .# "pixiOverlay" .$ (cb, container, options)) world
		# world                = (pixiOverlay .# "addTo" .$! mapObj) world
		# (cb, world)          = jsWrapFun (onPixiMapClick me pixiOverlay container) me world
		# world                = (mapObj .# "on" .$! ("click", cb)) world
		= (pixiOverlay, world)
	where
		onPixiMapClick me overlay container args world
			# (e, world) = args.[0] .? world
			# (interaction, world) = overlay .# "_renderer.plugins.interaction" .? world
			# (pointerEvent, world) = e .# "originalEvent" .? world
			# (pixiPoint, world) = jsNew "PIXI.Point" () world
			# (pex, world) = pointerEvent .# "clientX" .? world
			# (pey, world) = pointerEvent .# "clientY" .? world
			# world = (interaction .# "mapPositionToPoint" .$! (pixiPoint, pex, pey)) world
			# (target, world) = (interaction .# "hitTest" .$ (pixiPoint, container)) world
			| jsIsNull target = world
			| otherwise
				# (markerId, world) = target .# "markerId" .?? ("", world)
				= onMarkerClick me (LeafletObjectID markerId) args world

	createMapObjects me mapObj objects world
		# (l, world) = jsGlobal "L" .? world
		# world = forall (\_ object world -> createMapObject me mapObj object l world) objects world
		= world

	createMapObject me mapObj object l world
		# (type,world) = object .# "attributes.type" .? world
		= case jsValToString type of
			?Just "polyline"  = createPolyline  me mapObj l object world
			?Just "polygon"   = createPolygon   me mapObj l object world
			?Just "circle"    = createCircle    me mapObj l object world
			?Just "rectangle" = createRectangle me mapObj l object world
			?Just "window"    = createWindow    me mapObj l object world
			?Just "marker"    = jsTrace "This code should not be reached, markers should not longer children" world
			_                 = world

	createPolyline me mapObj l object world
		//Set options
		# (options,world)     = jsEmptyObject world
		# (style,world)       = object .# "attributes.style" .? world
		# world               = forall (applyLineStyle options) style world
		# (points,world)      = object .# "attributes.points" .? world
		# (layer,world)       = (l .# "polyline" .$ (points ,options)) world
		# world               = (layer .# "addTo" .$! mapObj) world
		# world               = enableEdit "polylineId" me layer object getUpdate world
		# world               = (object .# "layer" .= layer) world
		= world
	where
		getUpdate layer world
			# (points, world) = (layer .# "getLatLngs" .$ ()) world
			# (points, world) = jsValToList` points id world
			# (points, world) = foldl (\(res, world) point = appFst (\latLng -> [latLng: res]) $ toLatLng point world)
			                          ([], world)
			                          points
			= (UpdatePolyline $ reverse points, world)

	createPolygon me mapObj l object world
		//Set options
		# (options,world)     = jsEmptyObject world
		# (style,world)       = object .# "attributes.style" .? world
		# world               = forall (applyAreaStyle options) style world
		# (points,world)      = object .# "attributes.points" .? world
		# (layer,world)       = (l .# "polygon" .$ (points ,options)) world
		# world               = (layer .# "addTo" .$! mapObj) world
		# world               = enableEdit "polygonId" me layer object getUpdate world
		# world               = (object .# "layer" .= layer) world
		= world
	where
		getUpdate layer world
			# (points, world) = (layer .# "getLatLngs" .$ ()) world
			# (points, world) = points .# 0 .? world
			# (points, world) = jsValToList` points id world
			# (points, world) = foldl (\(res, world) point = appFst (\latLng -> [latLng: res]) $ toLatLng point world)
			                          ([], world)
			                          points
			= (UpdatePolygon $ reverse points, world)

	createCircle me mapObj l object world
		//Set options
		# (options,world)     = jsEmptyObject world
		# (style,world)       = object .# "attributes.style" .? world
		# world               = forall (applyAreaStyle options) style world
		# (center,world)      = object .# "attributes.center" .? world
		# (radius,world)      = object .# "attributes.radius" .? world
		# world               = (options .# "radius" .= radius) world
		# (layer,world)       = (l .# "circle" .$ (center, options)) world
		# world               = (layer .# "addTo" .$! mapObj) world
		# world               = enableEdit "circleId" me layer object getUpdate world
		# world               = (object .# "layer" .= layer) world
		= world
	where
		getUpdate layer world
			# (radius,   world) = (layer .# "getRadius" .$? ()) (0.0, world)
			# (center,   world) = (layer .# "getLatLng" .$ ()) world
			# (center,   world) = toLatLng center world
			= (UpdateCircle center radius, world)

	createRectangle me mapObj l object world
		//Set options
		# (options,world)     = jsEmptyObject world
		# (style,world)       = object .# "attributes.style" .? world
		# world               = forall (applyAreaStyle options) style world
		# (sw,world)          = object .# "attributes.bounds.southWest" .? world
		# (ne,world)          = object .# "attributes.bounds.northEast" .? world
		# (layer,world)       = (l .# "rectangle" .$ ([sw, ne], options)) world
		# world               = (layer .# "addTo" .$! mapObj) world
		# world               = enableEdit "rectangleId" me layer object getUpdate world
		# world               = (object .# "layer" .= layer) world
		= world
	where
		getUpdate layer world
			# (bounds, world) = (layer .# "getBounds" .$ ()) world
			# (bounds, world) = toBounds bounds world
			= (UpdateRectangle bounds, world)

	enableEdit idFieldName me layer object getUpdate world
		# (isEditable,world)  = object .# "attributes.editable" .?? (False, world)
		| not isEditable = world
		# (_, world)  = (layer .# "enableEdit" .$ ()) world
		# (cb, world) = jsWrapFun (onEditing layer) me world
		# (_, world)  = (layer .# "addEventListener" .$ ("editable:vertex:dragend", cb)) world
		# (_, world)  = (layer .# "addEventListener" .$ ("editable:vertex:new",     cb)) world
		# (_, world)  = (layer .# "addEventListener" .$ ("editable:vertex:deleted", cb)) world
		= world
	where
		onEditing layer _ world
			# (update,   world) = getUpdate layer world
			# (objectId, world) = object .# "attributes." +++ idFieldName .?? ("", world)
			# edit              = toJSON [LDUpdateObject (LeafletObjectID objectId) update]
			# (taskId,   world) = me .# "attributes.taskId" .? world
			# (editorId, world) = me .# "attributes.editorId" .? world
			# (_,        world) = (me .# "doEditEvent" .$ (taskId, editorId, edit)) world
			= world

	applyAreaStyle options _ style world
		# (styleType, world) = style .# 0 .? world
		# styleType = jsValToString styleType
		| styleType == ?Just "Style"
			# (directStyle, world) = style .# 1 .? world
			# (directStyleType, world) = directStyle .# 0 .? world
			# (directStyleVal, world)  = directStyle .# 1 .? world
			# directStyleType = jsValToString directStyleType
			= case directStyleType of
				?Just "AreaLineStrokeColor" = (options .# "color"       .= directStyleVal) world
				?Just "AreaLineStrokeWidth" = (options .# "weight"      .= directStyleVal) world
				?Just "AreaLineOpacity"     = (options .# "opacity"     .= directStyleVal) world
				?Just "AreaLineDashArray"   = (options .# "dashArray"   .= directStyleVal) world
				?Just "AreaNoFill"          = (options .# "fill"        .= False)          world
				?Just "AreaFillColor"       = (options .# "fillColor"   .= directStyleVal) world
				?Just "AreaFillOpacity"     = (options .# "fillOpacity" .= directStyleVal) world
				_                           = abort "unknown style"
		| styleType == ?Just "Class"
			# (cls, world) = style .# 1 .? world
			= (options .# "className" .= cls) world
		= abort "unknown style"

	calcInitPos initPos me world
		| jsIsUndefined initPos
			# (offset, world)    = me .# "windowOffset" .?? (0, world)
			# (initPos`, world)  = jsEmptyObject world
			# world              = (initPos` .# "x" .= fst DEFAULT_WINDOW_POS + offset) world
			# world              = (initPos` .# "y" .= snd DEFAULT_WINDOW_POS + offset) world
			# offset             = offset + 10
			# world              =
				(me .# "windowOffset" .= (if (offset > MAX_WINDOW_OFFSET) (offset - MAX_WINDOW_OFFSET) offset)) world
			= (initPos`, world)
		| otherwise = (initPos, world)

	createWindow me mapObj l object world
		# (layer,world)      = l .# "window" .$ () $ world
		# world              = (object .# "layer" .= layer) world
		# (initPos,world)    = object .# "attributes.initPosition" .? world
		# (initPos,world)    = calcInitPos initPos me world
		# (_, world)         = (layer .# "setInitPos" .$ initPos) world
		# (title,world)      = object .# "attributes.title" .? world
		# (_, world)         = (layer .# "setTitle" .$ title) world
		# (content,world)    = object .# "attributes.content" .? world
		# (_, world)         = (layer .# "setContent" .$ content) world
		# (relMarkers,world) = object .# "attributes.relatedMarkers" .? world
		# world              = forall (\_ relMarker world
			# world = (layer .# "addRelatedMarker" .$! relMarker) world
			# (markerId, world) = relMarker .# 0 .? world
			# (windows, world) = (me .# "windows.get" .$ markerId) world
			| jsIsUndefined windows = createNewSet me markerId layer world
			| otherwise = (windows .# "add" .$! layer) world
			)
		                       relMarkers
		                       world
		// inject function to send event on window remove
		# world = case viewOnly of
			True
				= world
			False
				# (windowId,world)   = object .# "attributes.windowId" .?? ("", world)
				# (onWRemove, world) = jsWrapFun (onWindowRemove me (LeafletObjectID windowId)) me world
				= (layer .# "_onWindowClose" .= onWRemove) world
		// inject function to handle window update
		# (cb,world)         = jsWrapFun (onUIChange layer) me world
		# world              = ((object .# "onUIChange") .= cb) world
		// add to map
		# world              = (layer .# "addTo" .$! mapObj) world
		# (children, world)  = me .# "container.children" .? world
		# world              = (layer .# "onCreate" .$! children) world
		= world
	where
		onUIChange layer changes world
			# world = foldl doChange world [c \\ c <-: changes]
			= world
		where
			doChange world change
				# (attrUpdates, world) = change .# "attributes" .? world
				# world = forall updateAttr attrUpdates world
				= world

			updateAttr _ attrChange world
				# (name,  world) = attrChange .# "name" .?? ("", world)
				# (value, world) = attrChange .# "value" .? world
				= case name of
					"content"        = layer .# "setContent"        .$! value $ world
					"title"          = layer .# "setTitle"          .$! value $ world
					"relatedMarkers" = updateRelatedMarkers me value world
					_                = abort $ concat ["unknown attribute of leaflet window: \"", name, "\"\n"]
			where
				updateRelatedMarkers me value world
					# world = layer .# "setRelatedMarkers" .$! (me, value) $ world
					# world = forall (\_ relMarker world
						# world = (layer .# "addRelatedMarker" .$! relMarker) world
						# (markerId, world) = relMarker .# 0 .? world
						# (windows, world) = (me .# "windows.get" .$ markerId) world
						| jsIsUndefined windows = createNewSet me markerId layer world
						| otherwise = (windows .# "add" .$! layer) world
						) value world
					= world

	createNewSet me markerId layer world
		# (set, world) = jsNew "Set" () world
		# world = (set .# "add" .$! layer) world
		# world = (me .# "windows.set" .$! (markerId, set)) world
		= world

	applyLineStyle options _ style world
	# (styleType, world) = style .# 0 .? world
	# styleType = jsValToString styleType
	| styleType == ?Just "Style"
		# (directStyle, world) = style .# 1 .? world
		# (directStyleType, world) = directStyle .# 0 .? world
		# (directStyleVal, world)  = directStyle .# 1 .? world
		# directStyleType = jsValToString directStyleType
		= case directStyleType of
			?Just "LineStrokeColor" = (options .# "color"     .= directStyleVal) world
			?Just "LineStrokeWidth" = (options .# "weight"    .= directStyleVal) world
			?Just "LineOpacity"     = (options .# "opacity"   .= directStyleVal) world
			?Just "LineDashArray"   = (options .# "dashArray" .= directStyleVal) world
			_                       = abort "unknown style"
	| styleType == ?Just "Class"
		# (cls, world) = style .# 1 .? world
		= (options .# "className" .= cls) world
	= abort "unknown style"

	//Loop through a javascript array
    forall :: !(Int JSVal *JSWorld -> *JSWorld) !JSVal !*JSWorld -> *JSWorld
	forall f array world
		# (len,world) = array .# "length" .?? (0, world)
		= forall` 0 len world
	where
        forall` :: !Int !Int !*JSWorld -> *JSWorld
		forall` i len world
			| i >= len  = world
			| otherwise = forall` (i + 1) len (f i (array .# i) world)

	//Process the edits received from the client
	onEdit diffs m vst
		# m = foldl app m diffs
		= (Ok (NoChange, m, ?Just m), vst)
	where
		app m LDSetManualPerspective      = case (m.LeafletMap.center, m.zoom) of
			(?Just center, ?Just zoom) = {m & perspective=CenterAndZoom center zoom}
			_                          = m // should not happen
		app m (LDSetZoom zoom)            = {LeafletMap|m & zoom = ?Just zoom}
		app m (LDSetCenter center)        = {LeafletMap|m & center = ?Just center}
		app m (LDSetBounds bounds)        = {LeafletMap|m & bounds = ?Just bounds}
		app m (LDRemoveWindow idToRemove) = {LeafletMap|m & objects = filter notToRemove m.LeafletMap.objects}
		where
			notToRemove (Window {windowId}) = windowId =!= idToRemove
			notToRemove _                   = True
		app m (LDClosePopup markerId`) = {LeafletMap|m & objects = withClosedPopup <$> m.LeafletMap.objects}
		where
			withClosedPopup :: !LeafletObject -> LeafletObject
			withClosedPopup (Marker m=:{markerId}) | markerId == markerId` = Marker {m & popup = ?None}
			withClosedPopup obj                                            = obj
		app m (LDUpdateObject objectId upd) = {LeafletMap|m & objects = withUpdatedObject <$> m.LeafletMap.objects}
		where
			withUpdatedObject :: !LeafletObject -> LeafletObject
			withUpdatedObject obj | leafletObjectIdOf obj === objectId = case (obj, upd) of
				(Polyline polyline, UpdatePolyline points)
					= Polyline {LeafletPolyline| polyline & points = points}
				(Polygon polygon, UpdatePolygon points)
					= Polygon {LeafletPolygon| polygon & points = points}
				(Circle circle, UpdateCircle center radius)
					= Circle {LeafletCircle| circle & center = center, radius = radius}
				(Rectangle rect, UpdateRectangle bounds)
					= Rectangle {LeafletRectangle| rect & bounds = bounds}
			withUpdatedObject obj = obj
		app m _ = m

	//Check for changed objects and update the client
	onRefresh mbNewMap oldMap vst
		# newMap = fromMaybe gDefault{|*|} mbNewMap
		//Determine attribute changes
		# attrChanges = diffAttributes oldMap newMap
		//Separate markers from other objects
		# (oldMarkers, oldOthers) = partition (\o -> o=:(Marker _)) oldMap.LeafletMap.objects
		# (newMarkers, newOthers) = partition (\o -> o=:(Marker _)) newMap.LeafletMap.objects
		//Determine object changes
		# childChanges = diffChildren oldOthers newOthers updateFromOldToNew encodeUI
		# attrChanges = if (isEmpty childChanges)
			attrChanges
			([SetAttribute "fitbounds" attr \\ attr <- fitBoundsAttribute newMap] ++ attrChanges)
		// We have to convert the title and popup to a string here,
		// because the toString function will no longer be available once we call the leafletCreateMarkers JS function
		# newMarkersWithHtmlStrings = map
			( \(Marker m=:{title, popup}) ->
				{LeafletMarker| m & title = (Html o toString) <$> title, popup = (Html o toString) <$> popup}
			)
			newMarkers
		# attrChanges =
			if (oldMarkers === newMarkers)
				attrChanges
				[SetAttribute "markers" $ toJSON newMarkersWithHtmlStrings : attrChanges]
		= (Ok (ChangeUI attrChanges childChanges, newMap, ?None),vst)
	where
		//Only center and zoom are synced to the client, bounds are only synced from client to server
		diffAttributes {LeafletMap|perspective=p1,icons=i1} n=:{LeafletMap|perspective=p2,icons=i2}
			//Perspective
			# perspective = if (p1 === p2) []
				[SetAttribute "perspective" (encodePerspective p2)
				: [SetAttribute "fitbounds" attr \\ attr <- fitBoundsAttribute n]]
			//Icons
			# icons = if (i2 === i1) [] [SetAttribute "icons" (JSONArray [toJSON (iconId,{IconOptions|iconUrl=iconUrl,iconSize=[w,h]}) \\ {iconId,iconUrl,iconSize=(w,h)} <- i2])]
			= perspective ++ icons

		updateFromOldToNew :: !LeafletObject !LeafletObject -> ChildUpdate
		updateFromOldToNew (Window old) (Window new) | old.windowId === new.windowId && not (isEmpty changes) =
			ChildUpdate $ ChangeUI changes []
		where
			changes = catMaybes
				[ if (old.LeafletWindow.title == new.LeafletWindow.title)
				     ?None
				     (?Just $ SetAttribute "title" $ toJSON $ new.LeafletWindow.title)
				, if (old.content === new.content)
				     ?None
				     (?Just $ SetAttribute "content" $ toJSON $ toString new.content)
				, if (old.relatedMarkers === new.relatedMarkers)
				     ?None
				     (?Just $ SetAttribute "relatedMarkers" $ toJSON new.relatedMarkers)
				]
		updateFromOldToNew old new | old === new = NoChildUpdateRequired
		                           | otherwise   = ChildUpdateImpossible

	writeValue m = Ok m

	fitBoundsAttribute :: !LeafletMap -> [JSONNode]
	fitBoundsAttribute {perspective=p=:FitToBounds _ region,objects} = [encodeBounds (bounds region)]
	where
		bounds (SpecificRegion bounds) = bounds
		bounds (SelectedObjects ids) = leafletBoundingRectangleOf [o \\ o <- objects | 'Set'.member (leafletObjectIdOf o) ids]
		bounds AllObjects = leafletBoundingRectangleOf objects

		encodeBounds :: !LeafletBounds -> JSONNode
		encodeBounds {southWest=sw,northEast=ne} = JSONArray
			[ JSONArray [JSONReal sw.lat, JSONReal sw.lng]
			, JSONArray [JSONReal ne.lat, JSONReal ne.lng]
			]
	fitBoundsAttribute _ = []

gEditor{|LeafletMap|} purpose = mapEditorWrite ValidEditor $ leafletEditor (purpose =: ViewValue)

gDefault{|LeafletMap|} =
	{ LeafletMap
	| perspective = defaultValue
	, bounds      = ?None
	, center      = ?None
	, zoom        = ?None
	, tilesUrls   = [openStreetMapTiles]
	, objects     = [Marker homeMarker]
	, icons       = [blackSquareIcon]
	}
where
	homeMarker =
		{ markerId = LeafletObjectID "home"
		, tint = ?None
		, rotation = ?None
		, position = {LeafletLatLng|lat = 51.82, lng = 5.86}
		, title = ?Just $ Text "HOME"
		, icon = ?Just blackSquareID
		, popup = ?None
		}

	blackSquareID :: LeafletIconID
	blackSquareID = LeafletIconID "black_square"

	blackSquareIcon :: LeafletIcon
	blackSquareIcon =
		{ LeafletIcon
		| iconId = blackSquareID
		, iconUrl = "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAoAAAAKEAYAAADdohP+AAABcGlDQ1BpY2MAACiRdZE7SwNBFIW/RCU+IiKKiFikiGJhQBREO41FmiAhKhi1STYvIZssuwkSbAUbC8FCtPFV+A+0FWwVBEERRCz8Bb4aCesdE0iQZJbZ+3FmzmXmDDiDGU23msdAz+bNcMDvWY6seFxvtNFHO91MRzXLmA2FgjQc3w84VL33qV6N99UdHfGEpYGjVXhSM8y88IxwcCNvKN4R7tXS0bjwsfCoKQcUvlF6rMyvilNl/lRsLobnwKl6elI1HKthLW3qwiPCXj1T0CrnUTdxJ7JLC1IHZA5iESaAHw8xCqyTIY9PalYyq+8b+/PNkxOPJn+DIqY4UqTFOypqQbompCZFT8iXoahy/5+nlZwYL3d3+6HlxbY/hsC1B6Vd2/45se3SKTQ9w1W26s9JTlNfou9WNe8RdG3BxXVVi+3D5Tb0PxlRM/onNcl0JpPwfg6dEei5g/bVclaVdc4eYXFTnugWDg5hWPZ3rf0CAfloC2Ri6QAAAAAJcEhZcwAACxIAAAsSAdLdfvwAAAA1SURBVCgVY2QAg///ITTlJAvMiP9gAOORTjOCAQMDE+la8esYNRB/+BAjO/jDkBHiDeolbADKMAoRMpO23QAAAABJRU5ErkJggg=="
		, iconSize = (24, 24)
		}

gDefault{|LeafletPerspective|} = CenterAndZoom {LeafletLatLng|lat = 51.82, lng = 5.86} 7

gDefault{|LeafletBounds|} =
	{ southWest = {lat=50.82, lng=4.86}
	, northEast = {lat=52.82, lng=6.86}
	}

//Comparing reals may have unexpected results, especially when comparing constants to previously stored ones
gEq{|LeafletLatLng|} x y = (toString x.lat == toString y.lat) && (toString x.lng == toString y.lng)

simpleStateEventHandlers :: LeafletEventHandlers LeafletSimpleState
simpleStateEventHandlers =
	[ OnMapClick \position (l,s) -> (addCursorMarker position l,{LeafletSimpleState|s & cursor = ?Just position})
	, OnMarkerClick \markerId (l,s) -> (l,{LeafletSimpleState|s & selection = toggle markerId s.LeafletSimpleState.selection})
	]
where
	addCursorMarker position l=:{LeafletMap|objects,icons} = {l & objects = addCursorObject objects, icons=addCursorIcon icons}
	where
		addCursorObject [] = [cursor position]
		addCursorObject [o=:(Marker {LeafletMarker|markerId}):os]
			| markerId =: (LeafletObjectID "cursor") = [cursor position:os]
			| otherwise = [o:addCursorObject os]
		addCursorIcon [] = [icon]
		addCursorIcon [i=:{iconId}:is]
			| iconId =: (LeafletIconID "cursor") = [i:is]
			| otherwise = [i:addCursorIcon is]

	cursor position = Marker
		{LeafletMarker|markerId=LeafletObjectID "cursor", position= position
		, rotation = ?None,tint = ?None,icon = ?Just (LeafletIconID "cursor"),title = ?None,popup = ?None}
	icon = {LeafletIcon|iconId=LeafletIconID "cursor", iconUrl= svgIconURL (CircleElt hattrs sattrs) (10,10), iconSize = (10,10)}
	where
        sattrs = [CxAttr (SVGLength "5" PX),CyAttr (SVGLength "5" PX),RAttr (SVGLength "3" PX)]
		hattrs = [StyleAttr "fill:none;stroke:#00f;stroke-width:2"]

	toggle (LeafletObjectID "cursor") xs = xs //The cursor can't be selected
	toggle x xs = if (isMember x xs) (removeMember x xs) ([x:xs])

customLeafletEditor :: !MapOptions !(LeafletEventHandlers s) s -> Editor (LeafletMap, s) (LeafletMap, s) | iTask s
customLeafletEditor mapOptions handlers initial = leafEditorToEditor (customLeafletEditor` mapOptions handlers initial)

customLeafletEditor` ::
	!MapOptions !(LeafletEventHandlers s) s -> LeafEditor [LeafletEdit] (LeafletMap,s) (LeafletMap,s) (LeafletMap,s)
	| iTask s
customLeafletEditor` mapOptions handlers initial =
	{ LeafEditor
	| onReset        = onReset
	, onEdit         = onEdit
	, onRefresh      = onRefresh
	, writeValue     = writeValue
	}
where
	baseEditor = leafletEditor` mapOptions $ case [h \\ OnMapDblClick h <- handlers] of
		[_:_] -> \me -> me .# "doubleClickZoom" .# "disable" .$! ()
		[]    -> const id

	onReset attributes mbval vst = case baseEditor.LeafEditor.onReset attributes (fst <$> mbval) vst of
			(Error e, vst) = (Error e, vst)
			(Ok (ui,mapState,mbw),vst) = (Ok (ui,(mapState, initial), mapMaybe (\s -> (s,initial)) mbw),vst)

	onEdit edit (mapState,customState) vst = case baseEditor.LeafEditor.onEdit edit mapState vst of
		(Error e, vst) = (Error e, vst)
		(Ok (mapChange,mapState,mbw1), vst)
			//Apply event handlers
			# (newMapState,customState) = updateCustomState handlers edit (mapState,customState)
			//Determine the change to the map
			= case baseEditor.LeafEditor.onRefresh (?Just newMapState) mapState vst of
				(Error e, vst) = (Error e, vst)
				(Ok (mapRefreshChange,mapState,mbw2),vst)
					# w = if (mbw1 =: ?None && mbw2 =: ?None) ?None (?Just (mapState,customState))
					= (Ok (mergeUIChanges mapChange mapRefreshChange, (mapState,customState), w),vst)

	onRefresh (?Just (newMapState,newCustomState)) (curMapState,curCustomState) vst
		 = case baseEditor.LeafEditor.onRefresh (?Just newMapState) curMapState vst of
			(Error e, vst) = (Error e, vst)
			(Ok (mapChange,mapState,mbw),vst) = (Ok (mapChange,(mapState,newCustomState),mapMaybe (\s -> (s,newCustomState)) mbw),vst)
	onRefresh ?None (curMapState,curCustomState) vst
		 = case baseEditor.LeafEditor.onRefresh ?None curMapState vst of
			(Error e, vst) = (Error e, vst)
			(Ok (mapChange,mapState,mbw),vst) = (Ok (mapChange,(mapState,initial),mapMaybe (\s -> (s,initial)) mbw),vst)

	writeValue m_and_s = Ok m_and_s

	updateCustomState handlers edits state = foldl (\s e -> foldl (update e) s handlers) state edits
	where
		update (LDMapClick position)    state (OnMapClick f)    = f position state
		update (LDMapDblClick position) state (OnMapDblClick f) = f position state
		update (LDMarkerClick markerId) state (OnMarkerClick f) = f markerId state
		update (LDHtmlEvent event)      state (OnHtmlEvent f)   = f event state
		update _                        state _                 = state

instance == LeafletObjectID where (==) (LeafletObjectID x) (LeafletObjectID y) = x == y
instance == LeafletIconID where (==) (LeafletIconID x) (LeafletIconID y) = x == y

instance < LeafletObjectID where (<) (LeafletObjectID x) (LeafletObjectID y) = x < y

gDefault{|FitToBoundsOptions|} =
	{ padding = (100, 100)
	, maxZoom = 10
	}

derive JSONEncode LeafletMap, LeafletLatLng, TileLayer
derive JSONDecode LeafletMap, LeafletLatLng, TileLayer
derive gDefault   LeafletLatLng
derive gEq        LeafletMap, TileLayer
derive gText      LeafletMap, LeafletLatLng, TileLayer
derive gEditor    LeafletLatLng
derive class iTask
	LeafletIcon, LeafletBounds, LeafletObject, LeafletMarker, LeafletPolyline,
	LeafletPolygon, LeafletEdit, LeafletWindow, LeafletWindowPos,
	LeafletLineStyle, LeafletStyleDef, LeafletAreaStyle, LeafletObjectID,
	CSSClass, LeafletIconID, LeafletCircle, LeafletObjectUpdate,
	LeafletRectangle, LeafletSimpleState, LeafletPerspective,
	FitToBoundsOptions, FitToBoundsRegion
