implementation module StdFile

// ****************************************************************************************
//	Concurrent Clean Standard Library Module Version 1.3
//	Copyright 1994 University of Nijmegen
// ****************************************************************************************

//	File modes synonyms

import	StdClass, StdMisc, StdString, StdInt

//	File modes

FReadText	:== 0	//	read from a text file
FWriteText	:== 1	//	write to a text file
FAppendText :== 2	//	append to an existing text file
FReadData	:== 3	//	read from a data file
FWriteData	:== 4	//	write to a data file
FAppendData :== 5	//	append to an existing data file

//	Seek modes

FSeekSet :== 0		//	new position is the seek offset
FSeekCur :== 1		//	new position is the current position plus the seek offset
FSeekEnd :== 2		//	new position is the size of the file plus the seek offset

::	* Files = Files;

class FileSystem f where
	fopen		:: !{#Char} !Int !*f -> (!Bool,!*File,!*f)
	/*	Opens a file for the first time in a certain mode (read, write or append, text or data).
		The boolean output parameter reports success or failure. */
	fclose		:: !*File !*f -> (!Bool,!*f)
	stdio		:: !*f -> (!*File,!*f)
	/*	Open the 'Console' for reading and writing. */
	sfopen		:: !{#Char} !Int !*f -> (!Bool,!File,!*f)
	/*	With sfopen a file can be opened for reading more than once.
		On a file opened by sfopen only the operations beginning with sf can be used.
		The sf... operations work just like the corresponding f... operations.
		They can't be used for files opened with fopen or freopen. */

instance FileSystem Files
where
	fopen :: !{#Char} !Int !*Files -> (!Bool,!*File,!*Files)
	fopen s i w
		# (b,f) = fopen_ s i;
		= (b,f,w);

	fclose :: !*File !*Files -> (!Bool,!*Files)
	fclose f w
		# b = fclose_ f
		= (b,w);

	stdio :: !*Files -> (!*File,!*Files)
	stdio w
		# f = stdio_
		= (f,w);

	sfopen :: !{#Char} !Int !*Files -> (!Bool,!File,!*Files)
	sfopen s i w
		# (b,f) = sfopen_ s i
		= (b,f,w)

instance FileSystem World
where
	fopen::!{#Char} !Int !*World -> (!Bool,!*File,!*World)
	fopen s i w
		# (b,f) = fopen_ s i;
		= (b,f,w);

	fclose :: !*File !*World -> (!Bool,!*World)
	fclose f w
		# b = fclose_ f
		= (b,w);

	stdio::!*World -> (!*File,!*World)
	stdio w
		# f = stdio_
		= (f,w);
	sfopen::!{#Char} !Int !*World -> (!Bool,!File,!*World)
	sfopen s i w
		# (b,f) = sfopen_ s i
		= (b,f,w)

fopen_ ::!{#Char} !Int -> (!Bool,!*File)
fopen_ s i = code {
	.inline fopen_
	.d 1 1 i
		jsr	openF
	.o 0 3 b f
	.end
	}

fclose_ :: !*File -> Bool
fclose_ f = code {
	.inline fclose_
	.d 0 2 f
		jsr	closeF
	.o 0 1 b
	.end
	}

/*	Open the 'Console' for reading and writing. */
stdio_ :: *File
stdio_ = code {
	.inline stdio_
	.d 0 0
		jsr	stdioF
	.o 0 2 f
	.end
	}

sfopen_ ::!{#Char} !Int -> (!Bool,!File)
sfopen_ s i
= code {
	.inline sfopen_
	.d 1 1 i
		jsr	openSF
	.o 0 3 b f
	.end
	}

/*
	openfiles::!*World -> (!*Files,!*World)
	openfiles world
	| (1 bitand w) == 0
		=	OpenFiles2 (StoreWorld (w bitor 1) world)
		=	abort "openfiles: This world doesn't contain files"
			where  w = LoadWorld world

	OpenFiles2::!*World -> (!*Files,!*World)
	OpenFiles2 w
	= code {
		pushI 0
	}

	LoadWorld :: !World -> Int;
	LoadWorld w = code {
		pushI_a 0
		pop_a 1
	};

	StoreWorld :: !Int !World -> *World;
	StoreWorld i w = code {
		fillI_b 0 1
		pop_b 1
		pop_a 1
	};

	closefiles::!*Files !*World -> *World
	closefiles f world
	=	CloseFiles2 f (StoreWorld ((LoadWorld world) bitand (-2)) world)

	CloseFiles2::!*Files !*World -> *World
	CloseFiles2 f w
	= code {
		pop_b 1
		fill_a 0 1
		pop_a 1
	}
*/

freopen::!*File !Int -> (!Bool,!*File)
/*	Re-opens an open file in a possibly different mode.
	The boolean indicates whether the file was successfully closed before reopening. */
freopen f m 
	= code {
		.inline freopen
		.d 0 3 f i
			jsr reopenF
		.o 0 3 b f
		.end
	}

//	Input. The boolean output parameter reports success or failure of the operations.

freadc::!*File -> (!Bool,!Char,!*File)
/*	Reads a character from a text file or a byte from a datafile. */
freadc f
	= code {
		.inline freadc
		.d 0 2 f
			jsr	readFC
		.o 0 4 b c f
		.end
	}

freadi::!*File -> (!Bool,!Int,!*File)
/*	Reads an integer from a textfile by skipping spaces, tabs and newlines and
	then reading digits, which may be preceeded by a plus or minus sign.
	From a datafile freadi will just read four bytes (a Clean Int). */
freadi f
	= code {
		.inline freadi
		.d 0 2 f
			jsr	readFI
		.o 0 4 b i f
		.end
	}

freadr::!*File -> (!Bool,!Real,!*File)
/*	Reads a real from a textfile by skipping spaces, tabs and newlines and then
	reading a character representation of a real number.
	From a datafile freadr will just read eight bytes (a Clean Real). */
freadr f
	= code {
		.inline freadr
		.d 0 2 f
			jsr	readFR
		.o 0 5 b r f
		.end
	}

freads::!*File !Int -> (!*{#Char},!*File)
/*	Reads n characters from a text or data file, which are returned as a {#Char}.
	If the file doesn't contain n characters the file will be read to the end
	of the file. An empty {#Char} is returned if no characters can be read. */
freads f l
	= code {
		.inline freads
		.d 0 3 f i
			jsr readFS
		.o 1 2 f
		.end
	}

freadline::!*File -> (!*{#Char},!*File)
/*	Reads a line from a textfile. (including a newline character, except for the last
	line) freadline cannot be used on data files. */
freadline f
	= code  {
		.inline freadline
		.d 0 2 f
			jsr readLineF
		.o 1 2 f
		.end
	}

//	Output. Use FError to check for write errors.

fwritec::!Char !*File -> *File
/*	Writes a character to a textfile.
	To a datafile fwritec writes one byte (a Clean Char). */
fwritec c f
	= code {
		.inline fwritec
		.d 0 3 c f
			jsr writeFC
		.o 0 2 f
		.end
	}

fwritei::!Int !*File -> *File
/*	Writes an integer (its textual representation) to a text file.
	To a datafile fwritec writes four bytes (a Clean Int). */
fwritei i f 
	= code {
		.inline fwritei
		.d 0 3 i f
			jsr writeFI
		.o 0 2 f
		.end
	}

fwriter::!Real !*File -> *File
/*	Writes a real (its textual representation) to a text file.
	To a datafile fwriter writes eight bytes (a Clean Real). */
fwriter r f
	= code {
		.inline fwriter
		.d 0 4 r f
			jsr writeFR
		.o 0 2 f
		.end
	}

fwrites::!{#Char} !*File -> *File
/*	Writes a {#Char} to a text or data file. */
fwrites s f 
	= code {
		.inline fwrites
		.d 1 2 f
			jsr writeFS
		.o 0 2 f
		.end
	}


//	Tests

fend::!*File -> (!Bool,!*File)
/*	Tests for end-of-file. */
fend f 
	= code {
		.inline fend
		.d 0 2 f
			jsr endF
		.o 0 3 b f
		.end
	}

ferror::!*File -> (!Bool,!*File)
/*	Has an error occurred during previous file I/O operations? */
ferror f 
	= code {
		.inline ferror
		.d 0 2 f
			jsr errorF
		.o 0 3 b f
		.end
	}

fposition::!*File -> (!Int,!*File)
/*	returns the current position of the file pointer as an integer.
	This position can be used later on for the fseek function. */
fposition f 
	= code {
		.inline fposition
		.d 0 2 f
			jsr positionF
		.o 0 3 i f
		.end
	}

fseek::!*File !Int !Int -> (!Bool,!*File)
/*	Move to a different position in the file, the first integer argument is the offset,
	the second argument is a seek mode. (see above). True is returned if successful. */
fseek f p m 
	= code {
		.inline fseek
		.d 0 4 f i i
			jsr seekF
		.o 0 3 b f
		.end
	}


//	Predefined files.

stderr::   *File
/*	Open the 'Errors' file for writing only. May be opened more than once. */
stderr 
	= code {
		.inline stderr
		.d 0 0
			jsr	stderrF
		.o 0 2 f
		.end
	}

sfreadc::!File -> (!Bool,!Char,!File)
sfreadc f 
	= code {
		.inline sfreadc
		.d 0 2 f
			jsr	readSFC
		.o 0 4 b c f
		.end
	}

sfreadi::!File -> (!Bool,!Int,!File)
sfreadi f
	= code {
		.inline sfreadi
		.d 0 2 f
			jsr	readSFI
		.o 0 4 b i f
		.end
	}

sfreadr::!File -> (!Bool,!Real,!File)
sfreadr f 
	= code {
		.inline sfreadr
		.d 0 2 f
			jsr	readSFR
		.o 0 5 b r f
		.end
	}


sfreads::!File !Int -> (!*{#Char},!File)
sfreads f i 
	= code {
		.inline sfreads
		.d 0 3 f i
			jsr readSFS
		.o 1 2 f
		.end
	}

sfreadline::!File -> (!*{#Char},!File)
sfreadline f 
	= code {
		.inline sfreadline
		.d 0 2 f
			jsr readLineSF
		.o 1 2 f
		.end
	}

sfseek::!File !Int !Int -> (!Bool,!File)
sfseek f i1 i2 
	= code {
		.inline sfseek
		.d 0 4 f i i
			jsr seekSF
		.o 0 3 b f
		.end
	}

/*	Change a file so that from now it can only be used with sfF... operations. */
fshare::!*File -> File
fshare f 
	= code {
		.inline fshare
		.d 0 2 f
			jsr shareF
		.o 0 2 f
		.end
	}

/*	The functions sfend and sfposition work like fend and fposition, but don't return a
	new file on which other operations can continue. They can be used for files opened
	with sfopen or after fshare, and in guards for files opened with fopen or freopen. */
sfend::!File -> Bool
sfend f
	= code {
		.inline sfend
		.d 0 2 f
			jsr endSF
		.o 0 1 b
		.end
	}

sfposition::!File -> Int
sfposition f
	= code {
		.inline sfposition
		.d 0 2 f
			jsr positionSF
		.o 0 1 i
		.end
		.inline fopen;w
		.end
		.inline fclose;w
		.end
		.inline stdio;w
		.end
		.inline sfopen;w
		.end
		.inline fopen;Files
		.end
		.inline fclose;Files
		.end
		.inline stdio;Files
		.end
		.inline sfopen;Files
		.end
		.inline accFiles;w
		.end
		.inline appFiles;w
		.end
	}

/*
		.inline openfiles
		.end
		.inline closefiles
		.end
*/

class (<<<) infixl a :: !*File !a -> *File

instance <<< Int where
//  (<<<) file i = fwritei i file
 (<<<) file i = code {
	.inline <<<;i
		push_b 2
		update_b 2 3
		update_b 1 2
		update_b 0 1
		pop_b 1
	.d 0 3 i f
		jsr writeFI
	.o 0 2 f
	.end
  }

instance <<< Char where
//  (<<<) file c = fwritec c file
 (<<<) file c = code {
	.inline <<<;c
		push_b 2
		update_b 2 3
		update_b 1 2
		update_b 0 1
		pop_b 1
	.d 0 3 c f
		jsr writeFC
	.o 0 2 f
	.end
  }

instance <<< {#Char} where
//  (<<<) file s = fwrites s file
 (<<<) file s = code {
	.inline <<<;#
	.d 1 2 f
		jsr writeFS
	.o 0 2 f
	.end
  }

instance <<< Real where
//  (<<<) file r = fwriter r file
 (<<<) file r = code {
	.inline <<<;r
		push_b 3
		push_b 3
		update_b 3 5
		update_b 2 4
		update_b 1 3
		update_b 0 2
		pop_b 2
	.d 0 4 r f
		jsr writeFR
	.o 0 2 f
	.end
  }

//	Access to the file system:

class FileEnv env where
	accFiles :: !.(*Files -> (.x,*Files)) !*env -> (!.x,!*env)
	appFiles :: !.(*Files -> *Files)      !*env -> *env

instance FileEnv World where
	accFiles :: !.(*Files -> (.x,*Files)) !*World -> (!.x,!*World)
	accFiles accfun world
		#! files=create_files
		#! (r,files) = accfun files
		= do_files2 files r world
		where
			do_files2 :: !*Files !.x !*World -> (!.x,!*World)
			do_files2 files r world
			 = (r,world)

	appFiles :: !.(*Files -> *Files) !*World -> *World
	appFiles appfun world
		#! files=create_files
		#! files=appfun files
		=  do_files files world
		where
			do_files :: !*Files !*World -> *World
			do_files files world
			 = world
/*			 = code {
			 	.inline do_files
				fill_a 1 2
				pop_a 2
				.end
			 }
*/
create_files :== Files;
