module FractalDemo;

/*
	A interactive fractal drawing program.
	
	This program will not run on a Macintosh Plus. It requires a
	Macintosh with at least 16 colors. To really create beautiful
	fractals in a reasonable time you need a Macintosh with 256
	colors and a mathematical co-processor.

	Run the program using the "No Console" option (Application options).

	To generate an application for this program the memory of the Clean
	0.8 application should be set to at least 1800K.
	The linker needs an additional 500K of free memory inside or outside
	the Clean 0.8 application.
	To launch the generated application 700K of free memory is needed
	outside the Clean 0.8 application.
*/

import StdInt, StdMisc, StdString, StdBool, StdReal, StdArray;
import Mandelbrot, deltaDialog, deltaControls;

from clCrossCall import Iprint; 
    
:: ZoomFunction :== ZoomState ->  Area -> Area ;

     
  Colour1 :== (0,99,4,99,2,60);
  Colour2 :== (4,99,0,99,2,75);
  Colour3 :== (4,99,2,70,0,99);
  Colour4 :== (0,80,1,80,3,60);
  Colour5 :== (2,99,0,99,1,99);

    
Start :: * World -> * World;
Start world =  CloseEvents events` world`;
             where {
             (state,events`)=: StartIO [window, menu, timer] InitState [] events;
             (events,world`)=: OpenEvents world;
             window =: WindowSystem [FixedWindow MyWindow MyPos "Fractal Demo" 
                                    ((0,0),(ScreenWidth, ScreenHeight))
                                    FractalUpdate [GoAway Quit,
                                    Mouse Unable (Track ZoomInArea)]];
             menu   =: MenuSystem [file, options, commands];
             file   =: PullDownMenu FileID "File" Able [
                       MenuItem QuitID "Quit" (Key 'Q') Able Quit];
             options=: PullDownMenu OptionsID "Options" Able [
                       fixed_depths,
                       MenuSeparator,
                       areas,
                       MenuSeparator,
                       functions,
                       MenuSeparator,
                       colours,
                       MenuItem 1000 "Set Palette..." (Key 'P') Able SetPalette];
             fixed_depths=: SubMenuItem FixedDepthsID "Fixed depths" Able [
                              MenuRadioItems Depth128ID [
                                 MenuRadioItem Depth32ID "32" NoKey Able
                                               (DefDepth 32),
                                 MenuRadioItem Depth64ID "64" NoKey Able
                                               (DefDepth 64),
                                 MenuRadioItem Depth128ID "128" NoKey Able
                                               (DefDepth 128),
                                 MenuRadioItem Depth256ID "256" NoKey Able
                                               (DefDepth 256),
                                 MenuRadioItem Depth512ID "512" NoKey Able
                                               (DefDepth 512),
                                 MenuRadioItem Depth1024ID "1024" NoKey Able
                                               (DefDepth 1024)]];
             areas=: SubMenuItem AreasID "Predefined Areas" Able [
                       CheckMenuItem Area1ID "normal mandelbrot" NoKey Able Mark
                                (DefArea Area1ID ((0.75,0.0),3.0,2.0)),
                       CheckMenuItem Area2ID "head" NoKey Able NoMark 
                                (DefArea Area2ID ((1.26,0.0),1.0,0.7)),
                       CheckMenuItem Area3ID "spike" NoKey Able NoMark 
                                (DefArea Area3ID ((1.54,0.0),0.20,0.14)),
                       CheckMenuItem Area4ID "spike detail" NoKey Able NoMark 
                                (DefArea Area4ID ((1.4814,-0.0013),0.0366,0.0278)),
                       CheckMenuItem Area5ID "back valley" NoKey Able NoMark 
                                (DefArea Area5ID ((-0.2963,-0.0152),0.1057,0.0926)),
                       CheckMenuItem Area6ID "head valley" NoKey Able NoMark 
                                (DefArea Area6ID ((0.8,-0.2),0.37,0.29)),
                       CheckMenuItem Area7ID "antenna" NoKey Able NoMark	
                                (DefArea Area7ID ((0.9203,-0.2889),0.0597,0.0606))];
             functions=: SubMenuItem FunctionsID "Mandelbrot Functions" Able [
                           MenuRadioItems Function1ID [
                              MenuRadioItem Function1ID "z = z*z + c" NoKey Able
                                            (DefaultFunction MSquare),
                              MenuRadioItem Function2ID "z = z*z*z + c" NoKey Able
                                            (DefFunction MCube),
                              MenuRadioItem Function3ID "z = sin z + c" NoKey Able
                                            (DefFunction MSin),
                              MenuRadioItem Function4ID "z = cos z + c" NoKey Able
                                            (DefFunction MCos),
                              MenuRadioItem Function5ID "z = exp z + c" NoKey Able
                                            (DefFunction MExp)]];
             colours  =: SubMenuItem ColoursID "Predefined Palettes" Able [
                           MenuRadioItems Colour1ID [
                              MenuRadioItem Colour1ID "Red" NoKey Able
                                            (SetColour Colour1),
                              MenuRadioItem Colour2ID "Green" NoKey Able
                                            (SetColour Colour2),
                              MenuRadioItem Colour3ID "Blue" NoKey Able
                                            (SetColour Colour3),
                              MenuRadioItem Colour4ID "Camouflage" NoKey Able
                                            (SetColour Colour4),
                              MenuRadioItem Colour5ID "Pastel" NoKey Able
                                            (SetColour Colour5)]];
             commands=: PullDownMenu CommandsID "Commands" Able [
                        MenuItem DrawID "Draw Mandelbrot" (Key 'M')
                                 Able DoMandelDraw,
                        MenuSeparator,
                        MenuItem ZoomInID "Zoom in" (Key 'Z') Unable
                                 (DoZoomFractal ZoomInArea),
                        MenuItem ZoomOutID "Zoom out" (Key 'O') Unable
                                 (DoZoomFractal ZoomOutArea),
                        MenuSeparator,
                        MenuItem StopDrawID "Halt Drawing" (Key 'S') Unable 
                                 DoHaltDrawing,
                        MenuItem ContinueID "Continue Drawing" NoKey Unable
                                 DoContinueDrawing];
             timer=: TimerSystem [Timer TimerID Unable 0 DrawFractal];
             };

InitState	::    FractalState;
InitState = ((((0.75,0.0),3.0,2.0),Colour1,128,MSquare), 
	              (0,0,0),
	              ((0,0),(0,0))
	             );

/*	Real update:
*/

FractalUpdate	:: UpdateArea FractalState -> (FractalState, [DrawFunction]);
FractalUpdate [] state =  (state, []);
FractalUpdate upd_area state=:(funcs, (0,0,0), zoom) =  (state, []);
FractalUpdate upd_area state = 	FractalUpdate` upd_area state;

FractalUpdate`	:: UpdateArea FractalState -> (FractalState, [DrawFunction]);
FractalUpdate` [first : rest] state =  (state``, [update_area : update_rest]);
		where {
		(state``,update_rest)=: FractalUpdate` rest state`;
		(state` ,update_area)=: UpdateFractalArea first state;
		};
FractalUpdate` [] state =  (state, []);


/*	File menu function:
*/

Quit	:: FractalState IO -> (FractalState, IO);
Quit f io =  (f, QuitIO io);


/*	Options menu functions:
*/

DefDepth	:: CalcDepth FractalState IO -> (FractalState, IO);
DefDepth depth state io =  (SetCalcDepth state depth, io);

DefArea	:: MenuItemId Area FractalState IO -> (FractalState, IO);
DefArea id area state io 
	= 	(SetArea state area, MarkMenuItems [id] (UnmarkAreas io));

DefFunction	:: FractalFunction FractalState IO -> (FractalState, IO);
DefFunction func state io =  (SetFFunction state func, io`);
	where {
	io`=: DisableMenuItems [Area1ID,Area2ID,Area3ID,Area4ID,Area5ID,Area6ID,Area7ID] io;
	};

DefaultFunction	:: FractalFunction FractalState IO -> (FractalState, IO);
DefaultFunction func state io =  (SetFFunction state func, io`);
	where {
	io`=: EnableMenuItems [Area1ID,Area2ID,Area3ID,Area4ID,Area5ID,Area6ID,Area7ID] io;
	};

SetColour	:: Colours FractalState IO -> (FractalState, IO);
SetColour colour=:(rd,ri,gd,gi,bd,bi) state io
	=  (SetNrOfColours state colour, ChangeDialog 1 [
			ChangeSliderBar 12 ( rd * 10  + 5), ChangeDynamicText 13 (toString rd),
			ChangeSliderBar 22 ri             , ChangeDynamicText 23 (toString ri),
			ChangeSliderBar 32 ( gd * 10  + 5), ChangeDynamicText 33 (toString gd),
			ChangeSliderBar 42 gi             , ChangeDynamicText 43 (toString gi),
			ChangeSliderBar 52 ( bd * 10  + 5), ChangeDynamicText 53 (toString bd),
			ChangeSliderBar 62 bi             , ChangeDynamicText 63 (toString bi)] io);

SetPalette	:: FractalState IO -> (FractalState, IO);
SetPalette state=:((a,(rd,ri,gd,gi,bd,bi),d,f),p,z) io
	=  (state, OpenDialog dialog io);
		where {
		dialog=: CommandDialog 1 "Palette" [ItemSpace (Pixel 6) (Pixel 12)] 1 [
		           ColourText 11 Left RedColour "Depth:",
		           PaletteSlider 12 (RightTo 11) ( rd * 10  + 5) 10,
		           DynamicText 13 (RightTo 12) (Pixel 30) (toString rd),
		           ColourText 21 (YOffset 11 (Pixel 6)) RedColour "Brightness:",
		           PaletteSlider 22 (RightTo 21) ri 1,
		           DynamicText 23 (RightTo 22) (Pixel 30) (toString ri),
		           ColourText 31 Left GreenColour "Depth:",
		           PaletteSlider 32 (RightTo 31) ( gd * 10  + 5) 10,
		           DynamicText 33 (RightTo 32) (Pixel 30) (toString gd),
		           ColourText 41 (YOffset 31 (Pixel 6)) GreenColour "Brightness:",
		           PaletteSlider 42 (RightTo 41) gi 1,
		           DynamicText 43 (RightTo 42) (Pixel 30) (toString gi),
		           ColourText 51 Left BlueColour "Depth:",
		           PaletteSlider 52 (RightTo 51) ( bd * 10  + 5) 10,
		           DynamicText 53 (RightTo 52) (Pixel 30) (toString bd),
		           ColourText 61 (YOffset 51 (Pixel 6)) BlueColour "Brightness:",
		           PaletteSlider 62 (RightTo 61) bi 1,
		           DynamicText 63 (RightTo 62) (Pixel 30) (toString bi),
		           DialogButton 1 Center "OK" Able PaletteOK];
		};

ColourText	:: DialogItemId ItemPos Colour String -> DialogItem FractalState IO;
ColourText id pos col text
	=  DialogIconButton id pos domain (DrawText at_new col text) Unable ButIdle;
		where {
		domain				=: ((0,0),(wid,at_new + (dt + ld)));
		wid					=: FontStringWidth "Brightness:" dfont;
		(at_new,dt,mw,ld)		=: FontMetrics dfont;
		(b,dfont)			=: SelectFont font style size;
		(font,style,size)	=: DefaultFont;
		};

DrawText	:: Int Colour String SelectState -> [DrawFunction];
DrawText y col text a =  [SetPenColour col, MovePenTo (0,y), DrawString text];

PaletteSlider	:: DialogItemId ItemPos SliderPos Int -> DialogItem FractalState IO;
PaletteSlider id pos slider val
	=  SliderBar id pos Able Horizontal slider 99 (ChangeValue id val);

ChangeValue	:: DialogItemId Int DialogInfo
                           (DialogState FractalState IO) -> DialogState FractalState IO;
ChangeValue id val dinfo dstate =  ChangeDynamicText (id+1) (toString pos) dstate;
		where {
		pos=:  GetSliderPosition id dinfo  / val;
		};

ButIdle	:: DialogInfo FractalState IO -> (FractalState, IO);
ButIdle dialog state io =  (state,io);

PaletteOK	:: DialogInfo FractalState IO -> (FractalState, IO);
PaletteOK dialog state io
	=  (SetNrOfColours state (rd,ri,gd,gi,bd,bi), ActivateWindow MyWindow io);
		where {
		rd=:  GetSliderPosition 12 dialog  / 10;
		ri=: GetSliderPosition 22 dialog;
		gd=:  GetSliderPosition 32 dialog  / 10;
		gi=: GetSliderPosition 42 dialog;
		bd=:  GetSliderPosition 52 dialog  / 10;
		bi=: GetSliderPosition 62 dialog;
		};

StrToInt	:: String -> Int;
StrToInt string =  TextToNumber string 0;

TextToNumber	:: String Int -> Int;
TextToNumber "" n =  n;
TextToNumber s  n | isdigit =  number;
	                  =  0;
	where {
	(isdigit,d)=: Digit (s.[0]);
	number	  =: TextToNumber (s % (1, size s - 1)) ( 10 * n  + d);
		};

Digit	:: Char -> (Bool, Int);
Digit '0'  =  (True, 0); 	  Digit '1' =  (True, 1);
Digit '2'  =  (True, 2); 	  Digit '3' =  (True, 3);
Digit '4'  =  (True, 4); 	  Digit '5' =  (True, 5);
Digit '6'  =  (True, 6); 	  Digit '7' =  (True, 7);
Digit '8'  =  (True, 8);	  Digit '9' =  (True, 9);
Digit c	  =  (False,0);

/*	Commands Menu functions:
*/

DoZoomFractal	:: ZoomFunction FractalState IO -> (FractalState,IO);
DoZoomFractal zoomfunc state io 
	= 	(state, io`);
		where {
		io`=: ChangeIOState [EnableMouse MyWindow,
							DisableTimer TimerID,
							DisableMenus [OptionsID, CommandsID],
							ChangeMouseFunction MyWindow (Track zoomfunc)] io;
		};

DoMandelDraw	:: FractalState IO -> (FractalState,IO);
DoMandelDraw state io 
	= 	(InitDrawState state, io`);
		where {
		io`=: ChangeIOState [EnableTimer      TimerID,
							DisableMenus	 [OptionsID],
							EnableMenuItems  [StopDrawID, ZoomInID, ZoomOutID],
							DisableMenuItems [DrawID, ContinueID]] io;
		};

DoHaltDrawing	:: FractalState IO -> (FractalState,IO);
DoHaltDrawing state io
	= 	(state`, EnableMenuItems [ContinueID] io`);
		where {
		(state`, io`)=: DoStopDrawing state io;
		};
					
DoContinueDrawing :: FractalState IO -> (FractalState,IO);
DoContinueDrawing state io
   =  (state, io`);
      where {
      io`=: ChangeIOState [EnableTimer TimerID, DisableMenus [OptionsID],
                          EnableMenuItems  [StopDrawID, ZoomInID, ZoomOutID],
                          DisableMenuItems [DrawID, ContinueID]] io;
      };

// Zooming
Track :: ZoomFunction MouseState FractalState IO -> (FractalState, IO);
Track zoomfun (point, ButtonUp, mod_new) state=:(fun=: ((area,c,d,f)),draw,zoom) io| TooSmall zoom` =  (state, DrawInWindow MyWindow [ReadyZoom zoom] iono);
   =  (InitDrawState state`,DrawInWindow MyWindow [ReadyZoom zoom] io`);
       where {
       state`=: SetArea state (zoomfun zoom` area);
       zoom`=: CorrectRect zoom;
       io`=: ChangeIOState [EnableMenus [	CommandsID],
                           DisableMenuItems [DrawID, ContinueID],
                           EnableMenuItems [ZoomInID, ZoomOutID,StopDrawID],
                           DisableMouse MyWindow,
                           EnableTimer TimerID] (UnmarkAreas io);
       iono=: ChangeIOState [EnableMenus [OptionsID, CommandsID],
                            EnableMenuItems [DrawID],
                            DisableMenuItems [StopDrawID]] io;
       };
Track zoomfun (point, ButtonStillDown, mod_new) (fun,draw,zoom=:(base,last)) io
   | EqualPoint last point =  ((fun,draw,zoom), io);
   =  ((fun,draw,rect`), DrawInWindow MyWindow [DrawFrame zoom rect`] io);
      where {
      rect`=: (base, point);
      };
Track zoomfun (point, ButtonDown, mod_new) (fun,draw,zoom) io
   =  ((fun,draw,rect`), DrawInWindow MyWindow [ZoomFrame rect`] io);
      where {
      rect`=: (point,point);
      };

ReadyZoom :: Rectangle Picture -> Picture;
ReadyZoom rect p
   =  SetPenNormal (DrawRectangle rect p);

DrawFrame :: Rectangle Rectangle Picture -> Picture;
DrawFrame oldrect rect p 
   =  DrawRectangle rect (DrawRectangle oldrect p);

ZoomFrame :: Rectangle Picture -> Picture;
ZoomFrame rect p 
   =  DrawRectangle rect (SetPenMode XorMode (SetPenColour BlackColour p));

TooSmall :: Rectangle -> Bool;
TooSmall ((x1,y1),(x2,y2))
   =    x2 - x1  < 8  ||   y2 - y1  < 8 ;

CorrectRect :: Rectangle -> Rectangle;
CorrectRect ((x1,y1),(x2,y2)) 
   =  ((xcor1,ycor1),(xcor2,ycor2));
      where {
      xcor1=:if (x1 < x2) x1 x2;
      ycor1=:if (y1 < y2) y1 y2;
      xcor2=:if (x2 < x1) x1 x2;
      ycor2=:if (y2 < y1) y1 y2;
      };
 
ZoomInArea :: ZoomState Area -> Area;
ZoomInArea ((x1,y1),(x2,y2)) ((xc,yc),areawidth,areaheight)
   =  ((centerx,centery),width,height);
      where {
      centerx=: xc +   areawidth /  toReal ScreenWidth   *
                         ( toReal ( x1 + x2  - ScreenWidth)  / 2.0) ;  
      centery=: yc +   areaheight /  toReal ScreenHeight   *
                         ( toReal ( y1 + y2  - ScreenHeight)  / 2.0) ;
      width  =: areawidth * ( toReal (x2 - x1)  /  toReal ScreenWidth );
      height =: areaheight * ( toReal (y2 - y1)  /  toReal ScreenHeight );
      };

ZoomOutArea :: ZoomState Area -> Area;
ZoomOutArea ((x1,y1),(x2,y2)) ((xc,yc),areawidth,areaheight)
   =  ((centerx,centery),width,height);
      where {
      centerx=: xc -   width /  toReal ScreenWidth   *
                         ( toReal ( x1 + x2  - ScreenWidth)  / 2.0) ;  
      centery=: yc -   height /  toReal ScreenHeight   *
                         ( toReal ( y1 + y2  - ScreenHeight)  / 2.0) ;
      width=:   toReal ScreenWidth  /  toReal (x2 - x1)   * areawidth;
      height=:   toReal ScreenHeight  /  toReal (y2 - y1)   * areaheight;
      };

UnmarkAreas :: (IOState s) -> IOState s;
UnmarkAreas io =  UnmarkMenuItems AreaIds io;

EqualPoint :: Point Point -> Bool;
EqualPoint (x,y) (h,v) =   x == h  &&  y == v ;
