#include <Clean.h>
#include <stdlib.h>
#include <string.h>
#include <mysql.h>
#include <stdio.h>
#include "_MySQL.h"

#undef DEBUG
//#define DEBUG

struct st_cursor {
	//Connection
    MYSQL* connection;
	
	//Result
    MYSQL_RES* result;
	unsigned int result_num_fields; 
	MYSQL_FIELD * result_fields;

	//Row
	MYSQL_ROW row;
	unsigned long* row_data_lengths;

}; //CURSOR

//Fetching error information from a connection 
void sql_connectionErrorC(MYSQL* mysql, int* errno_out, int* errsize_out, MYSQL** mysql_out) {

	*errno_out = mysql_errno(mysql);
	if(*errno_out != 0) {
		*errsize_out = strlen(mysql_error(mysql));
	} else {
		*errsize_out = 0;
	}
	*mysql_out = mysql;
}
MYSQL* sql_connectionErrorMessageC(CleanString message, MYSQL* mysql) {
	
	strncpy(CleanStringCharacters(message), mysql_error(mysql), CleanStringLength(message));
	return mysql;
}
//Fetching error information from a connection
void sql_cursorErrorC(CURSOR* cursor, int* errno_out, int* errsize_out, CURSOR** cursor_out) {

	MYSQL* mysql = cursor->connection;

	sql_connectionErrorC(mysql, errno_out, errsize_out, &mysql);

	*cursor_out = cursor;
}
CURSOR* sql_cursorErrorMessageC(CleanString message, CURSOR* cursor) {
	sql_connectionErrorMessageC(message, cursor->connection);
	return cursor;
}

int sql_initC() {
	#ifdef DEBUG
	printf("Initializing MySQL Library.\n");
	#endif

	return mysql_library_init(-1, NULL,NULL);
}

void sql_endC(int context) {
	mysql_library_end();

	#ifdef DEBUG
	printf("Finalized MySQL Library.\n");
	#endif
}

MYSQL* sql_openConnectionC(CleanString host, CleanString username, CleanString password, CleanString database) {

	MYSQL* mysql = NULL;	

	//Allocate memory for the connection parameters
	char* buff_host = calloc(sizeof(char),CleanStringLength(host) + 1);
	char* buff_username = calloc(sizeof(char),CleanStringLength(username) + 1);
	char* buff_password = calloc(sizeof(char),CleanStringLength(password) + 1);
	char* buff_database = calloc(sizeof(char),CleanStringLength(database) + 1);

	//Copy the connection parameters from Clean
	strncpy(buff_host,CleanStringCharacters(host),CleanStringLength(host));
	strncpy(buff_username,CleanStringCharacters(username),CleanStringLength(username));
	strncpy(buff_password,CleanStringCharacters(password),CleanStringLength(password));
	strncpy(buff_database,CleanStringCharacters(database),CleanStringLength(database));
	
	#ifdef DEBUG
	printf("Connecting to database (%s,%s,%s,%s)...\n",buff_host,buff_username,buff_password,buff_database);
	#endif

	//Initialize the connection handle
	mysql = mysql_init(NULL);

	//Connect
	int ok = !mysql_real_connect(mysql,buff_host,buff_username,buff_password,buff_database,0,NULL,CLIENT_FOUND_ROWS); 


	#ifdef DEBUG
	if(ok) {
		fprintf(stderr, "Could not connect to database.\n");
        fprintf(stderr, "%s\n",mysql_error(mysql));
	} else {
		printf("Connected.\n");
	}
	#endif
	
	//Free the buffers
	free(buff_host);
	free(buff_username);
	free(buff_password);
	free(buff_database);

	return mysql;
}

void sql_closeConnectionC(MYSQL* mysql) {
	mysql_close(mysql);

	#ifdef DEBUG
	printf("Connection closed.\n");
	#endif
}

void sql_openCursorC(MYSQL* mysql, CURSOR** cursor_out, MYSQL** mysql_out) {

	//Allocate memory for the cursor
	CURSOR *cursor = malloc(sizeof(CURSOR));

	if(cursor != NULL) {
		//Initialize cursor
		cursor->connection = mysql;
		cursor->result = NULL;

		#ifdef DEBUG
		printf("Cursor opened.\n");
		#endif
	}

	//Return tuple
	*mysql_out = mysql;
	*cursor_out = cursor;
}

void sql_executeC(CleanString statement, CURSOR* cursor, int* error_out, CURSOR** cursor_out) {

	//Check if we need to free a previous resultset first
	if(cursor->result != NULL) {
		mysql_free_result(cursor->result);
	}

	//Execute the statement
	if(mysql_real_query(cursor->connection,CleanStringCharacters(statement),CleanStringLength(statement))) {

		#ifdef DEBUG
		fprintf(stderr, "Query failed.\n");
        fprintf(stderr, "%s\n",mysql_error(cursor->connection));
		#endif
		
		*error_out = 1;
		*cursor_out = cursor;
		return;
	}

	//Save the pointer to the result set
	cursor->result = mysql_store_result(cursor->connection);

	if(cursor->result != NULL) {
		//Fetch the number of fields in the result set
		cursor->result_num_fields = mysql_num_fields(cursor->result);
		//Fetch the pointer to the field descriptions
		cursor->result_fields = mysql_fetch_fields(cursor->result);
	}

	#ifdef DEBUG
	if(cursor->result != NULL) {
		printf("Executed statement. Number of rows returned: %d\n", mysql_num_rows(cursor->result));
	} else {
		printf("Executed statement. No result rows\n");
	}
	#endif

	*error_out = 0;
	*cursor_out = cursor;
}

void sql_escapeC(CleanString from_statement, CleanString to_statement, CURSOR* cursor, int* length_out, CURSOR** cursor_out) {

	*length_out = mysql_real_escape_string (
		cursor->connection,
		CleanStringCharacters(to_statement),
		CleanStringCharacters(from_statement),
		CleanStringLength(from_statement)
		);

	*cursor_out = cursor;
}

void sql_insertIdC(CURSOR* cursor, int* id_out, CURSOR** cursor_out) {

	*id_out = mysql_insert_id(cursor->connection);
	*cursor_out = cursor;
}
void sql_numRowsC(CURSOR* cursor, int* num_out, CURSOR** cursor_out) {

	*num_out = mysql_affected_rows(cursor->connection);
	*cursor_out = cursor;

	#ifdef DEBUG
	printf("Num rows: %d\n", *num_out);
	#endif
}

void sql_numFieldsC(CURSOR* cursor, int* num_out, CURSOR** cursor_out) {

	*num_out = mysql_num_fields(cursor->result);
	*cursor_out = cursor;
}

void sql_fetchC (CURSOR* cursor, int* done_out, CURSOR** cursor_out) {

	//Fetch the row
	cursor->row = mysql_fetch_row(cursor->result);

	if(cursor->row != NULL) {
		//Fetch the lengths for this row.
		cursor->row_data_lengths = mysql_fetch_lengths(cursor->result);
		if(cursor->row_data_lengths != NULL) {
			*done_out = 0;
		} else {
			*done_out = 1;
		}
	} else {
		*done_out = 1;
	}
	*cursor_out = cursor;
}
void sql_fetchC_length (int i, CURSOR* cursor, int* length_out, CURSOR** cursor_out) {
	
	*length_out = cursor->row_data_lengths[i];
	*cursor_out = cursor;
}

void sql_fetchC_data (int i, CleanString data, CURSOR* cursor, int* type_out, CURSOR** cursor_out) {

	if(cursor->row[i] != NULL) {
		strncpy(CleanStringCharacters(data), cursor->row[i], CleanStringLength(data));

		//Convert the MySQL type code to a StdSQL type code
		switch(cursor->result_fields[i].type) {
			case MYSQL_TYPE_TINY:
			case MYSQL_TYPE_SHORT:
			case MYSQL_TYPE_LONG:
			case MYSQL_TYPE_INT24:
			case MYSQL_TYPE_LONGLONG:
				*type_out = STDSQLTYPE_INTEGER;
				break;
			case MYSQL_TYPE_DECIMAL:
			case MYSQL_TYPE_NEWDECIMAL:
				*type_out = STDSQLTYPE_REAL; //Is this correct?
				break;
			case MYSQL_TYPE_FLOAT:
				*type_out = STDSQLTYPE_FLOAT;	
				break;
			case MYSQL_TYPE_DOUBLE:
				*type_out = STDSQLTYPE_DOUBLE;
				break;
			case MYSQL_TYPE_TIMESTAMP:
				*type_out = STDSQLTYPE_TIMESTAMP;
				break;
			case MYSQL_TYPE_DATE:
				*type_out = STDSQLTYPE_DATE;
				break;
			case MYSQL_TYPE_TIME:
				*type_out = STDSQLTYPE_TIME;
				break;
			case MYSQL_TYPE_DATETIME:
				*type_out = STDSQLTYPE_DATETIME;
				break;
			case MYSQL_TYPE_STRING:
				*type_out = STDSQLTYPE_CHAR;
				break;
			case MYSQL_TYPE_VAR_STRING:
				*type_out = STDSQLTYPE_VARCHAR;
				break;
			case MYSQL_TYPE_BLOB:
				*type_out = STDSQLTYPE_TEXT;
				break;
			case MYSQL_TYPE_ENUM:
				*type_out = STDSQLTYPE_ENUM;
				break;
			default:
				*type_out = STDSQLTYPE_UNKNOWN;
				break;
		}
		//Detect if the field was an enum field
		if(cursor->result_fields[i].flags & ENUM_FLAG) {
				*type_out = STDSQLTYPE_ENUM;
		}
	} else {
		*type_out = STDSQLTYPE_NULL;
	}

	*cursor_out = cursor;
}

void sql_closeCursorC(CURSOR* cursor) {

	//We don't need to free anything if we have a non-initialized cursor
	if(cursor == NULL) {
		return;
	}

	//Free the resultset 
	if(cursor->result != NULL) {
		mysql_free_result(cursor->result);
	}	

	//Free the cursor itself
	free(cursor);

	#ifdef DEBUG
	printf("Cursor closed.\n");
	#endif
}
