;; lego-mode.el Major mode for editing lego
;; Copyright (C) 1994, 1995 Thomas Schreiber <lego@dcs.ed.ac.uk>
;; Time-stamp: <12 May 95 tms ~lego/emacs/lego-mode.el>
;; Credits to the authors of sml-mode, where we copied some code and
;;         to David Aspinall <da@lfcs.ed.ac.uk> for his help
;; Requires: compile, shell
;;
;; Plans
;;  1) Refine and intros by clicking
;;  2) Rewrite by clicking
;;  3) short cuts for LEGO commands e.g.,
;;       C-c i for intros
;;       C-c I for Intros
;;       C-c r for Refine
;;  4) complete LEGO name space e.g.,
;;       run Ctxt and strip types

;; CHANGES
;; lego-simple-send will only send messages to the end of the lego-process 

;; 7-11-94
;; fixed some bugs pointed out by Perdita Stevens

;; 9-11-94
;; lego-shells starts in another buffer, not screen

;; 28-12-94
;; lego-send-buffer

;; 29-12-94
;; lego-send-line no longer appends a semicolon

;; January/February 1995
;; implementing lego-send-command

;; changed pull-down to popup-menus for xemacs

;; 30-1-95
;; lego-send-line moves down one line after sending the current line 

;; February
;; integrating Mark Mark Ruys' legogrep

;; 24 April 1995
;; lego-send-buffer now invokes Make instead of Load/Reload

;; 26 April 1995
;; the buffer associated with the LEGO process will only pop up when
;; it is created

;; May 1995
;; lego-send-buffer-until-point

(require 'compile)
(require 'shell)
(load "~/afst/elisp/administratie")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;		 What version of Emacs are we using?		    ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar running-xemacs  nil)
(defvar running-emacs19 nil)

(setq running-xemacs  (string-match "XEmacs\\|Lucid" emacs-version))
(or running-xemacs
    (setq running-emacs19 (string-match "^19\\." emacs-version)))

;; INSTALLATION

;; Insert
;; (setq auto-mode-alist (cons '("\\.l$" . lego-mode) auto-mode-alist))
;; (autoload 'lego--mode "lego-mode"
;;                      "Major mode for editing Lego proof scripts." t)
;; (autoload 'lego-shell "lego-mode" "Inferior shell invoking lego." t)

;; in your .emacs file.

;; CUSTOMISATION

;; see
;; http://www.dcs.ed.ac.uk/packages/lego/html/emacs-customisation.html
;; on the World Wide Web for further details.

(defvar lego-test-all-name "test_all"
  "The name of the LEGO module which inherits all other modules of the
  library.")

(defvar legogrep-command (concat "legogrep -n \"\" " lego-test-all-name)
  "Last legogrep command used in \\{legogrep}; default for next legogrep.")

(defvar lego-www-home-page "http://www.dcs.ed.ac.uk/packages/lego/")

(defvar lego-map nil "The map used in lego-mode and in lego-shell.")

(if lego-map ()
  (setq lego-map (make-sparse-keymap))
  (define-key lego-map "\C-c\C-s" 'legogrep)
  (define-key lego-map "\C-c\C-p" 'lego-prf)
  (define-key lego-map "\C-c\C-k" 'lego-killref)
  (define-key lego-map "\C-c\C-u" 'lego-undo-1)
  (define-key lego-map "\C-c\C-t" 'lego-restart)
  (define-key lego-map "\C-c\C-c" 'lego-interrupt-subjob)
  (define-key lego-map "\C-c;" 'lego-active-semicolon-minor-mode)
  )

(defvar lego-mode-map nil "The mode map used in lego-mode.")
(if lego-mode-map
    ()
  (setq lego-mode-map (copy-keymap lego-map))
  (define-key lego-mode-map ";"        'lego-semicolon-function)
  (define-key lego-mode-map "\C-c\C-b" 'lego-send-buffer)
  (define-key lego-mode-map "\C-c\C-h" 'lego-send-buffer-until-point)
  (define-key lego-mode-map "\C-c\C-j" 'lego-send-line)
  (define-key lego-mode-map "\C-c\C-r" 'lego-send-region))

(defvar lego-path-name "LEGOPATH"
  "The name of the environmental variable to search for modules. This
  is used by \\{legogrep} to find the files corresponding to a
  search.")

(defvar lego-path-seperator ":"
  "A character indicating how the items in the \\{lego-path-name} are
  seperated.") 

(defvar lego-prog-name "lego"
  "*Name of program to run as lego.")

(defvar lego-prog-directory ""
  "*Name of directory of the program to run as lego.")

(defvar lego-prog-name-ask-p nil
  "*If t, you will be asked which program to run when the inferior
LEGO process starts up.")

(defvar lego-save-query t
  "*If non-nil, ask user for permission to save the current buffer before
  processing a module.")

(defvar lego-shell-echo-input t
  "If nil, input to the lego shell will not be echoed")

(defvar lego-shell-map nil 
  "The mode map for lego-shell.")

(defvar lego-load-name "Make" 
  "The name of the lego command to Make or Load a file.")

(if lego-shell-map ()
  (setq lego-shell-map (copy-keymap lego-map))
  
  ;; XEmacs (formerly Lucid) and FSF Emacs 19 do this differently
  (if (not (fboundp 'set-keymap-parents))
      (setq lego-shell-map (append shell-mode-map lego-shell-map))
    (set-keymap-parents lego-shell-map shell-mode-map)))

;; END OF CUSTOMIZATION

;; If you change anything below here, you're on your own

;; changes to the compile package
;; this should no longer be necessary in Emacs 19.29 and XEmacs 19.12


;; Find a buffer for file FILENAME.
;; Search the directories in compilation-search-path.
;; A nil in compilation-search-path means to try the
;; current directory, which is passed in DIR.
;; If FILENAME is not found at all, ask the user where to find it.
;; Pop up the buffer containing MARKER and scroll to MARKER if we ask the user.
(defun compilation-find-file (a b c &rest formats)
  (if running-emacs19 (setq marker a filename b dir c)
                      (setq marker c filename a dir b))
  (or formats
      (setq formats (or (car (nthcdr 4 (car compilation-error-regexp-alist)))
			'("%s"))))
  (let ((dirs compilation-search-path)
	result thisdir fmts name)
    (while (and dirs (null result))
      (setq thisdir (or (car dirs) dir)
	    fmts formats)
      (while (and fmts (null result))
	(setq name (expand-file-name (format (car fmts) filename) thisdir)
	      result (and (file-exists-p name)
			  (find-file-noselect name))
	      fmts (cdr fmts)))
      (setq dirs (cdr dirs)))
    (or result
	;; The file doesn't exist.
	;; Ask the user where to find it.
	;; If he hits C-g, then the next time he does
	;; next-error, he'll skip past it.
	(progn
	  (let* ((pop-up-windows t)
		 (w (display-buffer (marker-buffer marker))))
	    (set-window-point w marker)
	    (set-window-start w marker))
	  (setq name
		(expand-file-name
		 (read-file-name (format "Find this error in: (default %s) "
					 filename)
				 dir filename t)))
	  (if (file-directory-p name)
	      (setq name (concat (file-name-as-directory name) filename)))
	  (if (file-exists-p name)
	      (find-file-noselect name))))))
;; end of changes to the compile package

(defconst lego-comment-start "(* ")
(defconst lego-comment-end   " *)")

(defvar legogrep-regexp-alist
  '(("^\\([^:( \t\n]+\\)[:( \t]+\\([0-9]+\\)[:) \t]" 1 2 nil ("%s.l")))
  "Regexp used to match legogrep hits.  See `compilation-error-regexp-alist'.")

(defconst lego-mode-version-string
  "LEGO-MODE. Version 1.9 (May 1995) Thomas Schreiber <lego@dcs.ed.ac.uk>")

(defvar lego-process-name "lego"
  "*The name of the lego-process")

(defvar lego-shell-buffer-name (concat "*" lego-process-name "*")
  "The name of the Lego shell")

;; courtesy of Mark Ruys <markr@cs.kun.nl>
;; 1-2-1995 to strip multiple prompts
(defvar lego-shell-prompt-pattern "^\\(Lego> *\\)+"
  "*The prompt pattern for the inferion shell running lego.")

(defvar lego-shell-working-dir ""
  "The working directory of the lego shell")

(defun w3-remove-file-name (address)
  "remove the file name in a World Wide Web address"
  (string-match "://[^/]+/" address)
  (concat (substring address 0 (match-end 0))
	  (file-name-directory (substring address (match-end 0)))))

(defvar lego-library-www-page
  (concat (w3-remove-file-name lego-www-home-page)
	  "html/library/newlib.html"))

(defvar lego-www-customisation-page
  (concat (w3-remove-file-name lego-www-home-page)
	  "html/emacs-customisation.html"))

(defun string-to-list (s seperator) 
  "converts strings `s' seperated by the character `seperator' to a
  list of words" 
  (let ((end-of-word-occurence (string-match (concat seperator "+") s)))
    (if (not end-of-word-occurence)
	(if (string= s "") 
	    nil
	  (list s))
      (cons (substring s 0 end-of-word-occurence) 
	    (string-to-list 
	     (substring s
			(string-match (concat "[^" seperator "]")
				      s end-of-word-occurence)) seperator)))))


(defun lego-get-path ()
  (let ((path-name (getenv lego-path-name)))
    (cond ((not path-name)
	   ;; Martin Steffen <mnsteffe@informatik.uni-erlangen.de>
	   ;; has pointed out that calling lego-get-path has to deal
	   ;; with a user who hasn't set the environmental variable
	   ;; LEGOPATH. It is probably best if lego is installed as a
	   ;; shell script which sets a sensible default for LEGOPATH
	   ;; if the user hasn't done so before. See the documentation
	   ;; of the library for further details.
	   (message "Warning: LEGOPATH has not been set!")
	   ;; LEGO's default if LEGOPATH is void
	   (setq path-name ".")))	
    (string-to-list path-name lego-path-seperator)))

(defun legogrep (command-args)
  "Run legogrep, with user-specified args, and collect output in a buffer.
While grep runs asynchronously, you can use the \\[next-error] command
to find the text that legogrep hits refer to.

This command uses a special history list for its arguments, so you can
easily repeat a grep command."
  (interactive
   (list (;;read-shell-command "Run legogrep (like this): "
	  ;;		     legogrep-command 'grep-history)
	  read-from-minibuffer
	  "Run legogrep (like this): "
	  (if (string-match "\"\"" legogrep-command)
	      (cons legogrep-command (match-end 0))
              legogrep-command)
	  (and running-xemacs read-shell-command-map) nil 'grep-history)))
  (compile-internal command-args 
		    "No more legogrep hits"
		    "grep"
		    ;; Give it a simpler regexp to match.
		    nil legogrep-regexp-alist))

(defun lego-shell-buffer () (get-buffer lego-shell-buffer-name))

(defun lego-display-buffer (buffer)
  (let ((tmp-buffer (current-buffer))
        (new-buffer (display-buffer buffer)))
    (display-buffer tmp-buffer)
    new-buffer))
;  (let ((tmp-buffer (current-buffer)))
;    (display-buffer buffer)
;    (display-buffer tmp-buffer)))

(defun lego-simple-send (string &optional semicolon silent)
  (lego-shell)				; if no LEGO process is active
                                        ; at this stage, LEGO needs to
                                        ; be started
  (let ((lego-buf (lego-shell-buffer)))
    (if lego-buf
	(progn
	  (and semicolon
	       (or (string-match " *; *\\\'" string) ;; does string end with a ;
		   (setq string (concat string ";")))) ;; if not, append one
	  (save-excursion
	    (progn
	      (set-buffer lego-buf)
	      (goto-char (point-max))
	      (if (and lego-shell-echo-input (not silent))
		  (progn
		    (princ string lego-buf)
		    (comint-send-input))
		(comint-simple-send lego-process-name string)
		))))
	  (message "No active LEGO process"))))

(defvar lego-active-semicolon-minor-mode nil)
(make-variable-buffer-local 'lego-active-semicolon-minor-mode)
(put 'lego-active-semicolon-minor-mode 'permanent-local t)

(defun lego-semicolon-function ()
  (interactive)
  (if lego-active-semicolon-minor-mode (lego-send-command)
    (self-insert-command 1)))
    
(defun lego-active-semicolon-minor-mode (&optional arg)
  "Toggle LEGO's Active Semicolon minor mode.
With arg, turn on the Active Semicolon minor mode if and only if arg
is positive.

If Active semicolon mode is enabled, a semicolon will process the
current command."

 (interactive "P")
 
;; has this minor mode been registerd as such?
  (or (assq 'lego-active-semicolon-minor-mode minor-mode-alist)
      (setq minor-mode-alist
	    (append minor-mode-alist
		    (list '(lego-active-semicolon-minor-mode " ;")))))

 (setq lego-active-semicolon-minor-mode
	(if (null arg) (not lego-active-semicolon-minor-mode)
	  (> (prefix-numeric-value arg) 0)))
   (force-mode-line-update))
  

(defun lego-shell ()
  "Inferior shell invoking lego.
It is not possible to have more than one shell running lego.

Like the shell mode.


Mode map
========

\\{lego-shell-map}
Variables controlling the mode:

lego-prog-name (default \"lego\")
     The string used to invoke the lego program.

lego-shell-prompt-pattern (default \"^\\\\(Lego> *\\\\)+\")
     The prompt pattern.

Runs lego-shell-hook-before-displaying-buffer and
     lego-shell-hook-after-displaying-buffer."

  (interactive)
  (let ((lego-buf (and lego-process-name (lego-shell-buffer))))
    (if (comint-check-proc lego-buf)
	()
      ;; Desired buffer doesn't exist, start up a new process.
      (message "Starting lego...")
      (save-excursion
	(and lego-prog-name-ask-p
	     (setq lego-prog-name
		   (read-file-name
		    (concat "Lego (default " lego-prog-name "): ")
		    lego-prog-directory
		    lego-prog-name)))
	(setq lego-shell-working-dir default-directory)
	(or lego-process-name
	  (setq lego-process-name
		(concat
		 "Inferior "
		 (substring lego-prog-name
			    (string-match "[^/]*$" lego-prog-name)
			    (string-match "$" lego-prog-name)))))
	(set-buffer
	 (let ((lego-prog-name-list (string-to-list lego-prog-name " ")))
	   (apply 'make-comint 
		  (append (list lego-process-name 
				(car lego-prog-name-list) nil)
			  (cdr lego-prog-name-list)))))
	(erase-buffer)                  ; Erase the buffer in case
					; a previous process died in there
	(use-local-map lego-shell-map)
	(make-local-variable 'comint-prompt-regexp)
	(setq comint-prompt-regexp lego-shell-prompt-pattern)
	(setq comint-scroll-to-bottom-on-output t)
	(setq major-mode 'lego-shell)
	(setq mode-name "Lego-Shell")
      (run-hooks 'lego-shell-hook-before-displaying-buffer)
      (let ((lego-shell-window
             (lego-display-buffer lego-shell-buffer-name)))
        (set-buffer lego-shell-buffer-name)
        (run-hooks 'lego-shell-hook-after-displaying-buffer))
      (setq compilation-search-path (cons nil (lego-get-path))))
      (message "Starting LEGO... done.")
;      (lego-simple-send (concat "Cd \"" 
;				(expand-file-name lego-shell-working-dir) 
;				"\";") nil t)
      )))

  
(defun lego-mode ()
  "Major mode for editing Lego proof scripts.

Mode map
========
\\{lego-mode-map}

Runs lego-mode-hook if non nil"

  (interactive)
  (kill-all-local-variables)
  (use-local-map ref-map)  ;; mverp
  (setq major-mode 'lego-mode)
  (setq mode-name "Lego")
  (setq compilation-search-path (cons nil (lego-get-path)))
  (make-local-variable 'comment-start)
  (setq comment-start lego-comment-start)
  (make-local-variable 'comment-end)
  (setq comment-end lego-comment-end)
  (make-local-variable 'comment-start-skip)
  ;; This matches a start of comment (I sure hope!)
  (setq comment-start-skip "(\\*+[ \t]?")
  (run-hooks 'lego-mode-hook)
  (and running-xemacs (message "Press button 3 for popup-menus")))

(defun lego-simulate-send-region (point1 point2 &optional semicolon)
  "Send the area between point1 and point2 to the inferior shell running lego."
  (lego-simple-send (buffer-substring point1 point2) semicolon))

(defun lego-send-buffer ()
  "Send current buffer to inferior shell running lego taking advantage
  of lego's module system."
  (interactive)

  ;; the following code was nicked from AUCTeX's tex-buffer.el
  (and (buffer-modified-p)
       (or (not lego-save-query)
	   (y-or-n-p (concat "Save file "
			     (buffer-file-name)
			     "? ")))
       (save-buffer))
  
  (let (module-name)
    (setq module-name
	  (substring (file-name-nondirectory (buffer-file-name)) 0 -2))
    (lego-simple-send (concat "Try ForgetMark " module-name "; "
			      lego-load-name " " module-name) t t)))

(defun lego-send-buffer-until-point ()
  "Send current buffer to inferiour shell running LEGO and stop
  processing at point"
  (interactive)
  (let ((current-buffer (current-buffer))
	(temp-buffer (generate-new-buffer "*LEGO TMP*")))
    (save-excursion
      (set-buffer temp-buffer)
      (insert-buffer current-buffer))
    (let ((current-point (point)))
      (insert lego-comment-start)
      (goto-char (point-max))
      (insert lego-comment-end)
      (insert "echo \"Emacs pretends the file ends here ;-)\"")
      (lego-send-buffer)
      (erase-buffer)
      (insert-buffer temp-buffer)
      (goto-char current-point))
    (kill-buffer temp-buffer)))

;; lego-send-command tries to figure out where commands start and end
;; without having to parse LEGO expressions. This is fairly tricky and
;; I discover errors ever so often!

(defun lego-send-command ()
  "Send current command to inferior shell running lego. If point is a
  space and previous non-space character is not a semicolon, then a ;
  is inserted in Active Semicolon minor mode and the command is assumed to end here"  
  (interactive)
  (let (start end trailing-semicolon)
    (save-excursion
	;; search for a semicolon and ignore following spaces/tabs/newlines/comments
	(setq start (re-search-backward ";\\([ \t\n]\\|(\\*.*\\*)\\)*[^ \t\n)]"
					nil t)
	      start (if start 
			(goto-char (- (re-search-forward "[ \t\n;]*[^ \t$;]" nil t) 1))
		      (goto-char (point-min)))
	      trailing-semicolon (re-search-forward ";" nil t)
	      end (or trailing-semicolon (goto-char (point-max)))))
      (cond ((or (not trailing-semicolon)
		 (and (< (point) end)
		      (re-search-forward "[ \t\n]" (+ (point) 1) t)))
	     (and (> (point) (point-min)) trailing-semicolon (backward-char))
	     (and lego-active-semicolon-minor-mode (insert ";"))
	    (setq end (point))))
    (lego-simulate-send-region start end t)))

(defun lego-send-line ()
  "Send current line to inferior shell running lego"
  (interactive)
  (save-excursion
    (let (start end)
      (beginning-of-line 1)
      (setq start (point))
      (end-of-line 1)
      (setq end (point))
      (lego-simulate-send-region start end)))
  (forward-line 1))

(defun lego-send-region ()
  "Sends the current region to the inferior shell running lego and
  appends a ; if neccessary."
  (interactive)
  (let (start end)
    (save-excursion
      (setq end (point))
      (exchange-point-and-mark)
      (setq start (point)))
    (lego-simulate-send-region start end t)))

(defun lego-exit ()
  "Exit the LEGO process

  Runs lego-exit-lego-hook if non nil"

  (interactive)
  (lego-simple-send "\C-q\C-d")
  (and (get-buffer lego-shell-buffer-name)
       (kill-buffer lego-shell-buffer-name))
  ;; it is important that the hooks are run after the buffer has been
  ;; killed. In the reverse order e.g., intall-shell-fonts causes
  ;; problems and it is impossilbe to restart the LEGO shell
  (run-hooks 'lego-exit-lego-hook)
  (message "LEGO process terminated."))

(defun lego-interrupt-subjob ()
  "Send an interrupt signal to the LEGO process."
  (interactive) (lego-simple-send "\C-q\C-c" nil t))

(defun lego-killref ()
  "Kill the current refinement proof."
  (interactive)
  (lego-simple-send "KillRef;"))

(defun lego-help ()
  "Print help message."
  (interactive)
  (lego-simple-send "Help;"))

(defun lego-undo-1 ()
  "Undo the last step in a proof."
  (interactive)
  (lego-simple-send "Undo 1;"))

(defun lego-restart ()
  "Restart the proof."
  (interactive)
  (lego-simple-send "Undo 1000;"))

(defun lego-prf ()
  "List proof state."
  (interactive)
  (lego-simple-send "Prf;"))

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; font lock features ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;
(cond (window-system
       (defvar font-lock-declaration-name-face 
	 (progn 
	   (cond ((x-display-color-p)
		  (copy-face 'bold 'font-lock-declaration-name-face)

		  ;; Emacs 19.28 compiles this down to
		  ;; internal-set-face-1. This is not compatible with XEmacs
		  (dont-compile
		    (set-face-foreground
		     'font-lock-declaration-name-face "chocolate")))
		 (t (copy-face 'bold-italic 'font-lock-declaration-name-face)))
	   (if running-emacs19
	       (setq font-lock-declaration-name-face
		     (face-name 'font-lock-declaration-name-face)))))
    


       (defun lego-decl-defn-regexp (char)
	 (concat "\\[\\([^][" char " ]+\\) *\\(\\[[^]]*\\]\\)* *" char))

       (defconst lego-font-lock-keywords-1
	 (purecopy
	  (list
	   ;; fontify comments
	   '("(\\*.*\\*)" . font-lock-comment-face)
	   ;; fontify commands
	   '("\\<andI\\>\\|\\<Cd\\>\\|\\<Claim\\>\\|\\<Constructors\\>\\|\\<Ctxt\\>\\|\\<Cut\\>\\|\\<Decls\\>\\|\\<Discharge\\>\\|\\<DischargeKeep\\>\\|\\<Double\\>\\|\\<echo\\>\\|\\<Expand\\>\\|\\<ExportState\\>\\|\\<Equiv\\>\\|\\<Fields\\>\\|\\<Forget\\>\\|\\<ForgetMark\\>\\|\\<Freeze\\>\\|\\<From\\>\\|\\<Goal\\>\\|\\<Help\\>\\|\\<Hnf\\>\\|\\<Immed\\>\\|\\<Include\\>\\|\\<Inductive\\>\\|\\<Init\\>\\|\\<intros\\>\\|\\<Intros\\>\\|\\<KillRef\\>\\|\\<Load\\>\\|\\<Make\\>\\|\\<Next\\>\\|\\<NoReductions\\>\\|\\<Normal\\>\\|\\<Parameters\\>\\|\\<Prf\\>\\|\\<Pwd\\>\\|\\<Qrepl\\>\\|\\<Record\\>\\|\\<Refine\\>\\|\\<Relation\\>\\|\\<Reload\\>\\|\\<Save\\>\\|\\<Theorems\\>\\|\\<Undo\\>\\|\\<Unfreeze\\>\\|\\<Zap\\>" .font-lock-keyword-face)
	   ;; Declarations & Definitions
	   '("\\<Module\\>\\|\\<Import\\>" . font-lock-function-name-face)
	   (list (lego-decl-defn-regexp "=") 1 'font-lock-function-name-face)
	   '("\\<Goal\\>[ \t]+\\(\\<[^ \t\n\):]+\\>\\) *:"
	     1 font-lock-function-name-face)
	   '("\\<Save\\>[ \t]+\\([^ \t\n\):]+\\) *;" 1 font-lock-function-name-face)
	   '("\\<defn\\>  \\([^ \t\n\)]+\\) =" 1 font-lock-function-name-face)
	   (list (lego-decl-defn-regexp ":") 1 'font-lock-declaration-name-face)
	   (list (lego-decl-defn-regexp "|") 1 'font-lock-declaration-name-face)
	   '("^  \\([^ \t\n\)]+\\) = ... : " 1 font-lock-function-name-face)
	   '("^  \\([^ \t\n\)]+\\) : " 1 font-lock-declaration-name-face)
	   '("\\<decl\\>  \\([^:]+\\) :" 1 font-lock-declaration-name-face)
	   '("\\<decl\\>  \\(.+\\) |" 1 font-lock-declaration-name-face)
	   ;; Kinds
	   '("\\<Prop\\>\\|\\<Type *\\(([^)]+)\\)?" . font-lock-type-face)
	   
	   ;; strings
	   '("\"\\([\n]*\\)\"" 1 font-lock-string-face)
	   )))

       (cond (running-xemacs
	      (put 'lego-mode 'font-lock-keywords 'lego-font-lock-keywords-1)
	      (put 'lego-shell 'font-lock-keywords
		   'lego-font-lock-keywords-1))
	     (running-emacs19
	      (add-hook 'lego-mode-hook
			'(lambda () (setq font-lock-keywords
					  lego-font-lock-keywords-1)))
	      (add-hook 'lego-shell-hook-after-displaying-buffer
			'(lambda () (setq font-lock-keywords
					  lego-font-lock-keywords-1)))))))

;; Popup and Pulldown Menu

(defvar lego-shared-menu
  (append '(""
  ["Display proof state" lego-prf (get-buffer lego-shell-buffer-name)]
	   ["Kill the current refinement proof"
	    lego-killref (get-buffer lego-shell-buffer-name)]
	   ["Undo last proof step" lego-undo-1
	    (get-buffer lego-shell-buffer-name)]
	   ["Restart the proof" lego-restart
	    (get-buffer lego-shell-buffer-name)]
	   ["Exit LEGO" lego-exit (get-buffer lego-shell-buffer-name)]
	   "----"
	   ("LEGO Grep Mode Commands"
	    ["LEGO Grep..."		legogrep t]
	    ["Kill Grep"	kill-compilation (get-buffer-process (current-buffer))]
	    ["Goto Match" compile-goto-error (default-value 'compilation-error-list)]
	    ["Next Match"	next-error (default-value 'compilation-error-list)]
	    ["Previous Match"	previous-error (default-value 'compilation-error-list)]
    )
	   ("Help"
	    ["Print help message" lego-help t]
	    ["The LEGO library (WWW)"
	     (w3-fetch lego-library-www-page) t]
	    ["The LEGO Proof-assistant (WWW)"
	     (w3-fetch lego-www-home-page) t]
	    ["Help on Emacs LEGO-mode" describe-mode t]
	    ["Customisation" (w3-fetch lego-www-customisation-page) t]
	    ))))

(defvar lego-menu  
	 (append '("LEGO Commands"
	   ["Start LEGO" lego-shell
	    (not (comint-check-proc (and lego-process-name
					 (lego-shell-buffer))))]
	   ["Toggle active ;" lego-active-semicolon-minor-mode t]

	   "----"
	   ["Process buffer"  lego-send-buffer  t]
	   ["Process buffer until point" lego-send-buffer-until-point t]
	   ["Process command" lego-send-command t]
	   ["Process line"    lego-send-line    t]

	   ["Process region"  lego-send-region  t]
	   "----"
	   ["Comment region"  comment-region    t]
	   "----"
	   )
		 lego-shared-menu
		 )
	 "*The menu for LEGO.")

(defvar lego-shell-menu
  (cons
   "LEGO Shell Commands"
   (append
    lego-shared-menu
    (and
     running-xemacs
     (list "----"  
	   ["Goto Next Prompt" comint-next-prompt t]
	   ["Goto Previous Prompt" comint-previous-prompt t]
	   "----"
	   ["Previous Input" comint-previous-matching-input-from-input t]
	   ["Next Input" comint-next-matching-input-from-input t]
	   ["Previous Input matching Regexp..." 'comint-previous-matching-input t]
	   ["Next Input matching Regexp..." 'comint-next-matching-input t]
	   ["List Command History" comint-dynamic-list-input-ring t]
	   "----"
           ["Send INT Signal" lego-interrupt-subjob t]
 )))))

(defun lego-mode-popup-menu (event)
  "Display the lego-mode menu."
  (interactive "@e")
  (popup-menu lego-menu))

(defun lego-shell-popup-menu (event)
  "Display the lego-shell-mode menu."
  (interactive "@e")
  (let ((history (comint-make-history-menu)))
    (popup-menu (if history
		    (append lego-shell-menu
			    (list "---" (cons "Command History" history)))
		  lego-shell-menu))))

;;;;;;;;;;;;;;
;; de menus ;;
;;;;;;;;;;;;;;

(defvar ref-menu
  (list "Refinement Commands"
;;        ["Ga naar verfijning" 'kies-verf (not te-beginnen)]
;;        "-------"
        ["Nieuw bewijs" begin-nieuw-bewijs te-beginnen]
        ["Open bewijs" herstart-oud-bewijs te-beginnen]
        ["Sluit bewijs" sluit-bewijs (not te-beginnen)]
        "-------"
        ["Geef verfijning" verfijn proof-mode]
        ["Voeg verfijning in" insert-verf proof-mode]
        ["Begin bewijzen" begin-bewijs (not (or te-beginnen proof-mode))]
        "-------"
        ["Maak LEGO-file" 'make-lego-file (not te-beginnen)])
  "ref-menu voorlopig")

(defvar adm-menu
  (list "Administration Commands"
        ["Ga naar verfijning" 'kies-verf (not te-beginnen)]
        ["Ga stap terug" 'back-up
          (not (equal (get-req current-proof-adm) "(**)"))]
        ["Verwijder verfijning" 'verwijder-verf 
          (not te-beginnen)]
        "---------"
        ["Laat verfijnboom zien" 'laat-bewijsb-zien (not te-beginnen)])
  "adm-menu")

(defvar reftree-menu
  (list "Refine-tree Commands"
        ["Sluit bewijs-boom" 'sluit-reftree t]
        ["Spring naar verfijning" 'spring-verf t])
  "reftree-menu")

(defun reftree-mode-popup-menu (event)
  "Laat reftree-menu zien"
  (interactive "@e")
  (popup-menu reftree-menu))

(defun adm-mode-popup-menu (event)
  "Laat adm-menu zien"
  (interactive "@e")
  (popup-menu adm-menu))

(defun ref-mode-popup-menu (event)
  "Laat ref-menu zien"
  (interactive "@e")
  (popup-menu ref-menu))

(defvar ref-map nil "The map used in ref-mode")

(defvar adm-map nil "The map used in the adm-window")

(defvar reftree-map nil "The map used in the reftree-mode")

(if reftree-map ()
  (setq reftree-map (make-sparse-keymap))
  (define-key reftree-map "\C-l" (make-sparse-keymap))
    (define-key reftree-map "\C-l\C-j" 'spring-verf)
    (define-key reftree-map "\C-l\C-c" 'sluit-reftree))

(if adm-map ()
  (setq adm-map (make-sparse-keymap))
  (define-key adm-map "\C-l" (make-sparse-keymap))
    (define-key adm-map "\C-l\C-l" 'recenter)
    (define-key adm-map "\C-l\C-v" 'kies-verf)
    (define-key adm-map "\C-l\C-d" 'verwijder-verf)
    (define-key adm-map "\C-l\C-t" 'back-up)
    (define-key adm-map "\C-l\C-z" 'laat-bewijsb-zien)
  )

(if ref-map ()
  (setq ref-map (copy-keymap lego-mode-map))
  (define-key ref-map "\C-l" (make-sparse-keymap))
    (define-key ref-map "\C-l\C-l" 'recenter)
    (define-key ref-map "\C-l\C-n" 'begin-nieuw-bewijs)
    (define-key ref-map "\C-l\C-o" 'herstart-oud-bewijs)
    (define-key ref-map "\C-l\C-b" 'begin-bewijs)
    (define-key ref-map "\C-l\C-r" 'verfijn)
    (define-key ref-map "\C-l\C-s" 'sluit-bewijs)
    (define-key ref-map "\C-l\C-m" 'make-lego-file)
    (define-key ref-map "\C-l\C-i" 'insert-verf)
    (define-key ref-map "\C-l;" 'lego-active-semicolon-minor-mode)
  )

;;;;;;;;;;;;;;;;;;;;;
;; einde mijn code ;;
;;;;;;;;;;;;;;;;;;;;;

(cond (running-xemacs
       (define-key lego-mode-map 'button3  'lego-mode-popup-menu)
       (define-key ref-map 'button3  'ref-mode-popup-menu) ;; test ;;
       (define-key lego-shell-map 'button3 'lego-shell-popup-menu)
       (define-key adm-map 'button-3 'adm-mode-popup-menu)
       (define-key reftree-map 'button-3 'reftree-mode-popup-menu))
      ((string-match "^19" emacs-version)
       (require 'easymenu)
       (easy-menu-define lego-mode-menu
			 ref-map
			 "Menu used in lego mode."
			 (cons "LEGO"
			       (cdr lego-menu)))
       (easy-menu-define ref-mode-menu   ;; test ;;
			 ref-map
			 "Menu used in ref mode."
			 (cons "Refinements"
			       (cdr ref-menu)))
       (easy-menu-define reftree-mode-menu   ;; test ;;
			 reftree-map
			 "Menu used in reftree mode."
			 (cons "Refinement-tree"
			       (cdr reftree-menu)))
       (easy-menu-define adm-mode-menu
			 adm-map
			 "Menu used in adm-window."
			 (cons "Administration"
			       (cdr adm-menu)))
       (easy-menu-define lego-shell-menu
			 lego-shell-map
			 "Menu used in the lego shell."
			 (cons "LEGO" (cdr lego-shell-menu)))))

;;;;;;;;;;;  
;; mverp ;;
;;;;;;;;;;;
(setq ref-point (make-marker)) ;; Geeft waar nieuwe verf. moet komen
(setq old-point (make-marker)) ;; Voor het onderbreken van een bewijs
(defconst saved-once ())

(defun verfijn () "Behandelt het opnemen van een verfijning"
  (interactive)
  (if proof-mode
    (progn
      (onderbreek-bewijs)
      (geef-verfijning)
      (hervat-bewijs (get-naam current-proof-adm)))))

(defun onderbreek-bewijs () "Springt uit bewijs-mode"
  (interactive)
  (set-marker old-point (point))
  (lego-simple-send "KillRef;")
  (lego-simple-send (format "Forget %smarker;" (get-naam current-proof-adm)))
  (goto-char ref-point))

(defun hervat-bewijs (naam) "Springt terug naar bewijs-mode"
  (interactive)
  (set-file-marker naam)
  (beginning-of-buffer)
  (if (string= (buffer-substring 1 8) "Include") (beginning-of-line 2))
  (if (string= (buffer-substring (point) (+ (point) 6)) "Forget") 
    (beginning-of-line 2))
  (lego-simulate-send-region (point) old-point t)
  (goto-char old-point))

;;;;;;;;;;;;;;;;;;
;; Sluit bewijs ;;
;;;;;;;;;;;;;;;;;;

(defun sluit-bewijs () "Sluit een bewijs correct af"
  (interactive)
  (if te-beginnen ()
  (progn 
    (if (y-or-n-p "Wilt u de huidige versie opslaan? ")
      (progn 
        (set-buffer (format "%s.l" (get-naam current-proof-adm)))
        (sla-op (get-naam current-proof-adm))))
    (if (y-or-n-p "Wilt u het bewijs sluiten? ")
      (kill-proof)))))

(defun kill-proof () "Sluit alles af en staat opnieuw starten toe"
  (kill-buffer (format "%s.l" (get-naam current-proof-adm)))
  (kill-buffer "*lego*")
  (kill-buffer "adm-buffer")
  (delete-frame adm-frame)
  (select-frame (next-frame werk-frame nil))
  (delete-frame)
  (delete-process lego-process-name)
  (select-frame werk-frame)
  (defconst te-beginnen t)
  (defconst proof-mode ())
  (defconst current-proof-adm (new-adm)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; werken met adm-window ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun kies-verf () "kiest de verfijning op deze plaats en start bewijs"
  (interactive)
  (if (equal (buffer-name) "adm-buffer")
    (progn
      (let (verf start end)
        (beginning-of-line)
        (setq start (point))
        (end-of-line)
        (setq end (point))
        (setq verf (buffer-substring start end))
        (setq verf (verf-zstatus verf))
        (if (is-lem verf current-proof-adm)
          (start-bewijs-verf verf))))))

(defun verf-zstatus (verf) "Haalt de status van een verf"
  (if (equal (substring verf -4 -1) "TOD")
    (substring verf 0 -8)
    (substring verf 0 -7)))

(defun start-bewijs-verf (verf) "start het bewijs van de verfijning op"
  (select-frame werk-frame)
  (if proof-mode 
    (progn
      (lego-simple-send "KillRef;")
      (defconst proof-mode ())))
  (ask-for-closure (get-naam current-proof-adm))
  (if (file-exists-p (format "%s.l" (naam-uit-verf verf)))
    (stap-vooruit-oud (naam-uit-verf verf))
    (stap-vooruit-nieuw verf (get-naam current-proof-adm))))

(defun ask-for-closure (naam) "Vraagt of er offieel gesloten moet worden"
  (set-buffer (format "%s.l" naam))
  (if (y-or-n-p "Wilt u de huidige versie opslaan? ")
      (sla-op naam))
  (kill-buffer (current-buffer)))

(defun sla-op (naam) "Slaat de huidige file op en sluit hem af"
  (save-buffer)
  (check-status)
  (update-adm-file naam))

(defun check-status () "Zoek welke kleur het bewijs heeft"
  (if (is-geel)
    (if (is-groen current-proof-adm)
      (if (or (equal 'groen (nth 0 (get-proof current-proof-adm)))
              (equal "groen" (nth 0 (get-proof current-proof-adm))))
          ()
          (defconst current-proof-adm 
            (set-proof current-proof-adm 
              (set-kleur (get-stat current-proof-adm) "groen" 
                         (get-proof current-proof-adm)))))
      (if (or (equal 'geel (nth 0 (get-proof current-proof-adm)))
              (equal "geel" (nth 0 (get-proof current-proof-adm))))
          ()
          (defconst current-proof-adm 
            (set-proof current-proof-adm 
              (set-kleur (get-stat current-proof-adm) "geel" 
                         (get-proof current-proof-adm))))))
    (if (or (equal 'rood (nth 0 (get-proof current-proof-adm)))
              (equal "rood" (nth 0 (get-proof current-proof-adm))))
          ()
          (defconst current-proof-adm 
            (set-proof current-proof-adm 
              (set-kleur (get-stat current-proof-adm) "rood" 
                         (get-proof current-proof-adm)))))))

(defun is-geel () "Checkt of een file geel als kleur heeft"
  (print "geel")
  (save-excursion
    (set-buffer "*lego*")
    (let (naam proof gok gos start end)
      (setq naam (get-naam current-proof-adm))
      (setq proof (get-stat current-proof-adm))
      (setq gok ())
      (setq gos ())
      (beginning-of-buffer)
      (while (search-forward (format "Goal %s" proof) nil t)
        (beginning-of-line)
        (setq start (point))
        (end-of-line)
        (setq end (point))
        (delete-region start end)
        (setq gos t))
      (beginning-of-buffer)
      (while (search-forward (format "\"%s\"  saved as" naam) nil t)
        (beginning-of-line)
        (setq start (point))
        (end-of-line)
        (setq end (point))
        (delete-region start end)
        (setq gok t))
      (if (and gok gos)
        (progn 
          (defconst saved-once t)
          gok)
        (if (and saved-once (not (or gok gos)))
          (and saved-once t)
          ())))))

(defun is-groen (adm) "Is het bewijs absoluut?"
  (print "groen")
  (let (groen lems)
    (setq groen t)
    (setq lems (get-lem adm))
    (while (and (not (equal lems ())) groen)
      (if (not (equal (format "%s" (nth 0 (car lems))) "groen"))
        (setq groen ()))
      (setq lems (cdr lems)))
    groen))

(defun update-adm-file (naam) "Zet de kleuren in alle adm-files goed"
   (write-region (format "%S" current-proof-adm) nil 
                 (format "%s.adm" naam) nil nil)
   (if (equal (get-req current-proof-adm) "(**)")
     ()
     (behandel-parent (get-req current-proof-adm)
                      (get-stat current-proof-adm)
                      (nth 0 (get-proof current-proof-adm)))))

(defun behandel-parent (parent child color) "Zet de kleuren in adm-files"
  (let (name tmp-adm to-go kind kleur change)
    (setq name parent)
    (setq to-go t)
    (setq kind child)
    (setq kleur color)
    (setq change ())
    (while to-go
      (setq tmp-adm (read (find-file-noselect (format "%s.adm" name))))
      (kill-buffer (format "%s.adm" name))
      (setq tmp-adm (make-adm (get-req tmp-adm)
                              (get-proof tmp-adm)
                              (set-color kind kleur (get-lem tmp-adm))))
      (if (equal (format "%s" (nth 0 (get-proof tmp-adm))) "groen")
          (if (is-groen tmp-adm) ()
              (setq tmp-adm (make-adm
                              (get-req tmp-adm)
                              (set-kleur 
                                (get-stat tmp-adm)
                                "geel"
                                (get-proof tmp-adm))
                              (get-lem tmp-adm)))
              (setq change t)))
      (if (equal (format "%s" (nth 0 (get-proof tmp-adm))) "geel")
          (if (is-groen tmp-adm)
              (setq tmp-adm (make-adm
                              (get-req tmp-adm)
                              (set-kleur 
                                (get-stat tmp-adm)
                                "groen"
                                (get-proof tmp-adm))
                              (get-lem tmp-adm)))
              (setq change t)))
      (write-region (format "%S" tmp-adm) nil (format "%s.adm" name) nil nil)
      (if (equal "(**)" (get-req tmp-adm)) (setq to-go ())
        (progn 
          (setq kind (get-stat tmp-adm))
          (setq name (get-req tmp-adm))
          (setq kleur (nth 0 (get-proof tmp-adm))))))))
;;;;;;;;;;;;;;;;;;
;; nieuw-bewijs ;;
;;;;;;;;;;;;;;;;;;

(defun stap-vooruit-nieuw (stment parent) "Gaat een dieper in reftree met nieuw lemma"
  (let (naam)
    (defconst saved-once ())
    (setq naam (naam-uit-verf stment))
    (find-file (format "%s.l" naam))
    (lego-simple-send (format "Forget %s;" naam))
    (set-file-marker naam)
    (make-new-adm naam)
    (insert (format "Include %s;" parent))
    (newline)
    (insert (format "Forget %s;" naam))
    (newline)))

(defun set-file-marker (naam) "Zet een marker voor terugstappen"
  (lego-simple-send (format "[%smarker:Prop];" naam)))

(defun make-new-adm (naam) "Reset het adm-window naar nieuwe file"
  (save-excursion
    (set-buffer "adm-buffer")
    (erase-buffer)
    (open-bewijs-met-naam naam))) 

(defun back-up () "Stap terug interactive"
  (interactive)
  (stap-terug (get-req current-proof-adm)))

(defun stap-terug (naam) "Zet een stap terug in de reftree"
  (lego-simple-send (format "Forget %smarker;" naam))
  (select-frame werk-frame)
  (ask-for-closure (get-naam current-proof-adm))
  (defconst saved-once ())
  (set-file-marker naam)
  (herlees-file naam)
  (reset-cpa-adm naam)
  (plaats-ref-point (get-stat current-proof-adm))
  (set-proof-mode (get-stat current-proof-adm)))

(defun plaats-ref-point (proof) "Plaats ref-point in een buffer"
  (beginning-of-buffer)
  (if (search-forward (format "Goal %s;" proof) nil t) 
    (progn (beginning-of-line)
           (set-marker ref-point (point))))
  (end-of-buffer))

(defun set-proof-mode (proof) "Kijkt of men in proofmode is"
  (beginning-of-buffer)
  (if (search-forward (format "Goal %s;" proof) nil t)
    (defconst proof-mode t))
  (beginning-of-buffer)
  (if (search-forward (format "Save %s;" proof) nil t)
    (defconst proof-mode ()))
  (end-of-buffer))

(defun herlees-file (naam) "Leest de file van naam, zonder de includes en forget"
  (find-file (format "%s.l" naam))
  (if (string= (buffer-substring 1 8) "Include") (beginning-of-line 2))
  (if (string= (buffer-substring 1 7) "Forget") (beginning-of-line 2))
  (end-of-buffer)
  (lego-send-region))

(defun reset-cpa-adm (naam) "Zet de cpa en adm op hun oude waarden terug"
  (save-excursion
    (set-buffer "adm-buffer")
    (erase-buffer)
    (open-bewijs-met-naam naam)))

(defun clear-adm () "Wist adm-window"
  (setq temp-buf (current-buffer))
  (set-buffer "adm-buffer")
  (erase-buffer)
  (set-buffer temp-buf))

(defun sprong-nieuw (naam) "Springt naar een nieuw bewijs"
  (defconst saved-once ())
  (lego-simple-send "Forget begin;")
  (lego-simple-send "[begin:Prop];")
  (clear-adm)
  (open-bewijs-met-naam naam)
  (importeer-file (get-req current-proof-adm))
  (lego-simple-send (format "Forget %s;" naam))
  (set-file-marker naam)
  (insert (format "Include %s;" (get-req current-proof-adm)))
  (newline)
  (insert (format "Forget %s;" naam))
  (newline))

(defun importeer-file (naam) "Lees file in en check op nodige KillRef"
  (let (bufje)  
    (setq bufje (find-file-noselect (format "%s.l" naam)))
    (save-excursion
      (set-buffer bufje)
      (zend-en-killref))
    (kill-buffer bufje)))

(defun zend-en-killref () "zend en check"
  (beginning-of-buffer)
  (end-of-buffer)
  (lego-send-region)
  (to-killref))

(defun to-killref () "Killref of niet?"
 (let (save-f goal-f)
   (if (search-backward "Save" nil t) (setq save-f (point)) (setq save-f 0))
   (end-of-buffer)
   (if (search-backward "Goal" nil t) (setq goal-f (point)) (setq goal-f 0))
   (if (< save-f goal-f) (lego-simple-send "KillRef;"))))

;;;;;;;;;;;;;;;;
;; oud bewijs ;;
;;;;;;;;;;;;;;;;

(defun sprong-oud (naam) "Springt naar een reeds bewerkt bewijs"
  (ask-for-closure (get-naam current-proof-adm))
  (defconst saved-once ())
  (lego-simple-send "Forget begin;")
  (lego-simple-send "[begin:Prop];")
  (clear-adm)
  (select-frame werk-frame)
  (save-excursion
    (set-buffer "adm-buffer")
    (open-bewijs-met-naam naam))
  (import-and-mark naam)
  (switch-to-buffer (format "%s.l" naam)))

(defun import-and-mark (naam) "importeer de file en mark haar"
  (if (file-exists-p (format "%s.l" (get-req current-proof-adm)))
    (importeer-file (get-req current-proof-adm)))
  (lego-simple-send (format "Forget %s;" naam))
  (set-file-marker naam)
  (defconst proof-mode ())
  (if (file-exists-p (format "%s.l" naam))
    (herlees-file naam)
    (progn 
      (find-file (format "%s.l" naam))
      (if (string= (get-req current-proof-adm) "(**)") ()
        (progn
          (insert (format "Include %s;" (get-req current-proof-adm)))
          (newline)))
      (insert (format "Forget %s;" naam))
      (newline)))
  (plaats-ref-point (get-stat current-proof-adm))
  (set-proof-mode (get-stat current-proof-adm)))

(defun stap-vooruit-oud (naam) "Zet een stap naar onderen in de reftree"
  (defconst saved-once ())
  (lego-simple-send (format "Forget %s;" naam))
  (set-file-marker naam)
  (clear-adm)
  (save-excursion
    (set-buffer "adm-buffer")
    (open-bewijs-met-naam naam))
  (find-file (format "%s.l" naam))
  (herlees-file naam))

;;;;;;;;;;;;;;;;;;
;; begin bewijs ;;
;;;;;;;;;;;;;;;;;;

(defconst proof-mode () "is men aan het bewijzen?")

(defun begin-bewijs () "begint bewijs met Goal"
  (interactive)
  (if (or te-beginnen proof-mode) ()
    (let (proof)
      (setq proof (nth 1 (get-proof current-proof-adm)))
      (defconst proof-mode t)
      (set-marker ref-point (point))
      (insert (format "Goal %s;" proof))
      (lego-simple-send (format "Goal %s;" proof))
      (newline))))

;;;;;;;;;;;;;;;;;;;;;
;; geef verfijning ;;
;;;;;;;;;;;;;;;;;;;;;

(defun geef-verfijning () "Neemt verfijning op"
  (let (refnaam refstat lem)
    (setq refnaam (read-from-minibuffer "Refinement name> "))
    (if (file-exists-p (format "%s.adm" refnaam))
      (print "Refinement name exsists")
      (setq refstat (read-from-minibuffer "Refinement statement> "))
      (setq lem (list "rood" (concat refnaam ":" refstat)))
      (behandel-verfijning lem)
      (write-adm (format "%s.adm" refnaam) 
        (make-adm (get-naam current-proof-adm) lem ()))
      (maak-werk-file (get-naam current-proof-adm) refnaam)
      (defconst current-proof-adm (add-lem-adm current-proof-adm lem))
      (refresh-adm))))

(defun maak-werk-file (parent verf) "maakt een begin .l file aan"
  (find-file-noselect (format "%s.l" verf))
  (save-excursion
    (set-buffer (format "%s.l" verf))
    (beginning-of-buffer)
    (insert (format "Include %s;" parent))
    (newline)
    (insert (format "Forget %s;" verf))
    (newline)
    (save-buffer))
  (kill-buffer (format "%s.l" verf)))

(defun behandel-verfijning (verf) "Zet verf neer in text"
  (let (stat)
    (if (equal (get-lem current-proof-adm) ())
      (progn 
        (beginning-of-line)
        (insert "(** Refinement **)")
        (newline)
        (setq stat (nth 1 verf))
        (insert (concat "[" stat "];"))
        (newline 2)
        (set-marker ref-point (point)))
      (progn
        (beginning-of-line 0)
        (insert "(** Refinement **)")
        (newline)
        (setq stat (nth 1 verf))
        (insert (concat "[" stat "];"))
        (newline)
        (beginning-of-line 2)
        (set-marker ref-point (point))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;
;; verwijder verfijning ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun verwijder-verf () "Verwijder een verfijning"
  (interactive)
  (if (equal (buffer-name) "adm-buffer")
    (progn
      (let (verf start end)
        (beginning-of-line)
        (setq start (point))
        (end-of-line)
        (setq end (point))
        (setq verf (buffer-substring start end))
        (setq verf (verf-zstatus verf))
        (if (is-lem verf current-proof-adm)
          (progn
            (delete-ref verf start end)))))))

(defun delete-ref (verf start end) "Doe het delete werk"
  (if (not (equal (overlays-at (+ start 1)) ()))
    (delete-overlay (car (overlays-at (+ start 1)))))
  (delete-region start (+ end 1))
  (defconst current-proof-adm (delete-lem  verf current-proof-adm))
  (set-buffer (format "%s.l" (get-naam current-proof-adm)))
  (beginning-of-buffer)
  (search-forward (format "%s" verf) nil t)
  (let (verf begin eind)
       (beginning-of-line 0)
       (setq begin (point))
       (end-of-line 2)
       (setq eind (point))
       (delete-region begin eind))
  (end-of-buffer))

;;;;;;;;;;;;;;;;;;;;;;;
;; insert verfijning ;;
;;;;;;;;;;;;;;;;;;;;;;;

(defun insert-verf () "Insert een verfijning in het huidige bewijs"
  (interactive)
  (if proof-mode
    (let (refnaam input-naam)
      (setq refnaam (read-from-minibuffer "The refinement to be inserted> "))
      (if (not (file-exists-p (format "%s.l" refnaam)))
        (print "File does not exist.")
        (setq input-naam 
          (read-from-minibuffer "The refinement it will be placed for> "))
        (if (not (is-lem-by-name input-naam))
          (print "This is not a refinement.")
          (lego-simple-send (format "Forget %smarker;" 
            (get-naam current-proof-adm)))
          (insert-ref refnaam input-naam)
          (beginning-of-buffer)
          (end-of-buffer)
          (lego-send-region))))))

(defun insert-ref (refnaam input-naam) "De daadwerkelijke toevoeging"
  (let (inserted start end curverf refin tmp-adm)
    (setq tmp-adm (read (find-file-noselect (format "%s.adm" refnaam))))
    (kill-buffer (format "%s.adm" refnaam))
    (setq refin (get-proof tmp-adm))
    (defconst current-proof-adm 
      (insert-lems refin input-naam current-proof-adm))
    (setq refin (get-stat tmp-adm))
    (refresh-adm)
    (beginning-of-buffer)
    (setq inserted ())
    (while (and (not inserted) 
                (not (eobp))
                (search-forward "(** Refinement **)" nil t))
      (beginning-of-line 2)
      (setq start (point))
      (end-of-line)
      (setq end (point))
      (setq curverf (buffer-substring (+ start 1) end))
      (setq curverf (naam-uit-verf curverf))
      (if (equal curverf input-naam)
        (progn
          (beginning-of-line 0)
;;        (beginning-of-line 0)
          (insert "(** Refinement **)")
          (newline)
          (insert (format "[%s];" refin))
          (newline)
          (setq inserted t)))))
  (update-parent refnaam (get-naam current-proof-adm)))

(defun update-parent (refnaam parent) "Updates de ref.l en ref.adm file"
  (let (tmp-adm)
    (setq tmp-adm (read (find-file-noselect (format "%s.adm" refnaam))))
    (kill-buffer (format "%s.adm" refnaam))
    (setq tmp-adm (set-req tmp-adm parent))
    (write-adm (format "%s.adm" refnaam) tmp-adm))
  (let (tmp-buffer start end)
    (setq tmp-buffer (find-file-noselect (format "%s.l" refnaam)))
    (save-excursion
      (set-buffer tmp-buffer)
      (beginning-of-buffer)
      (setq start (point))
      (end-of-line)
      (setq end (point))
      (delete-region start end)
      (beginning-of-line)
      (insert (format "Include %s;" parent))
      (save-buffer))
    (kill-buffer tmp-buffer)))

;;;;;;;;;;;;;;;;;;;;;;;;;;
;; begin voor interface ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;
(defconst te-beginnen t "Moet er nog iets worden gestart")

(defun begin-nieuw-bewijs () "Voor een verse start"
  (interactive)
  (if te-beginnen
    (progn
      (defconst saved-once ())
      (let (proofnaam proofstat prooftot)
        (setq proofnaam (read-from-minibuffer "The name of the proof> "))
        (if (file-exists-p (format "%s.l" proofnaam))
          (print "File exists")
          (progn
            (setq proofstat (read-from-minibuffer "The statement> "))
            (setq prooftot (list "rood" (concat proofnaam ":" proofstat)))
            (defconst current-proof-adm (set-proof current-proof-adm prooftot))
            (defconst werk-frame (selected-frame))
            (start-adm)
            (make-adm-file (format "%s.adm" proofnaam))
            (lego-shell)
            (defconst te-beginnen nil)
            (force-mode-line-update)
            (lego-simple-send "[begin:Prop]")
            (set-file-marker proofnaam)
            (find-file (format "%s.l" proofnaam))
            (insert " ")
            (erase-buffer)
            (save-buffer)))))))

(defun start-adm () "Opent het adm window" 
  (open-adm-window)
  (save-excursion
    (set-buffer "adm-buffer")
    (print-proof (get-proof current-proof-adm))))

(defun make-adm-file (naam) "Maakt eerste adm file aan" 
  (write-adm naam current-proof-adm))

(defun herstart-oud-bewijs () "Laad een oud bewijs"
  (interactive)
  (if te-beginnen
    (progn
      (let (filenaam)
        (setq filenaam (read-from-minibuffer "The proofname> "))
        (open-adm-window)
        (save-excursion
          (set-buffer "adm-buffer")
          (open-bewijs-met-naam filenaam)))
      (defconst werk-frame (selected-frame))
      (defconst saved-once ())
      (lego-shell)
      (defconst te-beginnen nil)
      (force-mode-line-update)
      (lego-simple-send "[begin:Prop];")
      (import-and-mark (get-naam current-proof-adm)))))

(defun open-bewijs-met-naam (naam) "Opent-bewijs-adm maar dan met naam"
  (read-adm (format "%s.adm" naam))
  (print-proof (get-proof current-proof-adm))
  (print-lemmas (get-lem current-proof-adm))
  (kill-buffer (format "%s.adm" naam)))

;;;;;;;;;;;;;;;;;;;;;;;;;;
;; laat bewijsboom zien ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun laat-bewijsb-zien () "Laat de bewijsboom zien"
  (interactive)
  (if te-beginnen ()
  (progn 
    (select-frame adm-frame)
    (get-buffer-create "bewijs-buffer")
    (switch-to-buffer "bewijs-buffer")
    (if (equal (vraag-top-hier) "top")
      (totale-boom)
      (laat-boom-zien (get-naam current-proof-adm)))
    (reftree-mode))))

(defun reftree-mode () "Mode voor het reftree-window. //{reftree-mode-map}"
  (interactive)
  (use-local-map reftree-map)
  (setq mode-name "Refinement-tree")
  (setq major-mode 'reftree-mode))

(defun vraag-top-hier () "Vraagt of de top moet of de huidige plaats"
  (read-from-minibuffer "From here or from the top (here or top)?> "))

(defun totale-boom () "laat de hele bewijsboom zien"
  (let (cur-req tmp-adm)
    (setq tmp-adm current-proof-adm)
    (setq cur-req (get-req current-proof-adm))
    (while (not (equal cur-req "(**)"))
      (setq tmp-adm (read (find-file-noselect (format "%s.adm" cur-req))))
      (kill-buffer (format "%s.adm" cur-req))
      (setq cur-req (get-req tmp-adm)))
    (laat-boom-zien (get-naam tmp-adm))))

(defun laat-boom-zien (naam) "Print de boom"
  (let (tmp-adm tmp-lems depth)
    (setq tmp-adm (read (find-file-noselect (format "%s.adm" naam))))
    (kill-buffer (format "%s.adm" naam))
    (setq tmp-lems (list (get-lem tmp-adm)))
    (print-lemma (get-proof tmp-adm) 0)
    (setq depth 2)
    (while (not (equal tmp-lems ()))
      (if (equal (car tmp-lems) ())
        (progn
          (setq tmp-lems (cdr tmp-lems))
          (setq depth (- depth 2)))
        (progn
          (print-lemma (car (car tmp-lems)) depth)
          (setq tmp-adm (read (find-file-noselect 
                          (format "%s.adm" 
                            (naam-uit-verf 
                              (format "%s" (nth 1 (car (car tmp-lems)))))))))
          (kill-buffer (format "%s.adm" 
                         (naam-uit-verf 
                           (format "%s" (nth 1 (car (car tmp-lems)))))))
          (setq tmp-lems (cons (cdr (car tmp-lems)) (cdr tmp-lems)))
          (if (equal (get-lem tmp-adm) ()) ()
            (progn
              (setq depth (+ depth 2))
              (setq tmp-lems (cons (get-lem tmp-adm) tmp-lems)))))))))

(defun sluit-reftree () "Sluit het reftree-window"
  (interactive)
  (if (equal (selected-frame) adm-frame)
    (progn
      (switch-to-buffer "adm-buffer")
      (kill-buffer "bewijs-buffer"))
    (let (tmp-frm)
      (setq tmp-frm (selected-frame))
      (select-frame adm-frame)
      (switch-to-buffer "adm-buffer")
      (kill-buffer "bewijs-buffer")
      (select-frame tmp-frm))))

(defun spring-verf () "Springt naar verf"
  (interactive)
  (if (equal (buffer-name) "bewijs-buffer")
    (progn
      (let (verf start end name)
        (beginning-of-line)
        (setq start (point))
        (end-of-line)
        (setq end (point))
        (setq verf (buffer-substring start end))
        (setq verf (verf-zstatus verf))
        (setq verf (haal-spat-van-verf verf))
        (setq name (naam-uit-verf verf))
        (sprong-oud name))
        (sluit-reftree))))

(defun haal-spat-van-verf (ref) "Haalt spaties uit verf"
  (let (refje)
    (setq refje ref)
    (while (equal (substring refje 0 1) " ")
      (setq refje (substring refje 1)))
    refje))

;;;;;;;;;;;;;;;;
;; adm-window ;;
;;;;;;;;;;;;;;;;

(defun open-adm-window () "Opent het adm-window"
  (defconst adm-frame (make-frame '((name."administration"))))
  (let (old-frame)
    (setq old-frame (selected-frame))
    (select-frame adm-frame)
    (switch-to-buffer "adm-buffer")
    (adm-mode)
    (select-frame old-frame)))

(defun adm-mode () "Mode voor het adm-window. //{adm-mode-map}"
  (interactive)
  (use-local-map adm-map)
  (setq mode-name "Administration")
  (setq major-mode 'adm-mode))

(defun refresh-adm () "Herleest de cpa voor het adm-window"
  (clear-adm)
  (save-excursion
    (set-buffer "adm-buffer")
    (print-proof (get-proof current-proof-adm))
    (print-lemmas (get-lem current-proof-adm))))

;;;;;;;;;;;;;;;;;;;;
;; maak LEGO-file ;;
;;;;;;;;;;;;;;;;;;;;

(defun make-lego-file () "Maakt een LEGO files van de LEGO-r files"
  (interactive)
  (let (top tmp-adm)
    (setq top (zoek-top))
    (setq tmp-adm (read (find-file-noselect (format "%s.adm" top))))
    (defconst current-proof-adm tmp-adm)
    (kill-buffer (format "%s.adm" top))
    (if (is-groen tmp-adm)
      (put-includes-rec top))
    (final-test top)))

(defun zoek-top () "Zoekt de file van de top of reftree"
  (let (temp-adm tmp-req)
    (setq temp-adm current-proof-adm)
    (setq tmp-req (get-req temp-adm))
    (while (not (equal tmp-req "(**)"))
      (setq temp-adm (read (find-file-noselect (format "%s.adm" tmp-req))))
      (kill-buffer (format "%s.adm" tmp-req))
      (setq tmp-req (get-req temp-adm)))
    (get-naam temp-adm)))

(defun put-includes-rec (naam) "Regelt de iteratie van put-includes"
  (let (filetjes adm start end verf cur)
    (setq filetjes (list naam))
    (while (not (equal filetjes ()))
       (setq cur (car filetjes))
       (find-file-noselect (format "%s.l" (eval 'cur)))
       (setq filetjes (cdr filetjes))
       (save-excursion
         (set-buffer (format "%s.l" (eval 'cur)))
         (beginning-of-buffer)
         (if (equal "Include" (buffer-substring 1 8))
           (progn 
             (end-of-line 2)
             (delete-region 1 (+ (point) 1))))
         (beginning-of-buffer)
         (while (search-forward "(** Refinement **)" nil t)
           (end-of-line 2)
           (setq end (point))
           (beginning-of-line)
           (setq start (point))
           (setq verf (buffer-substring start end))
           (setq filetjes (cons (substring (naam-uit-verf verf) 1) filetjes))
           (delete-region start end)
           (insert (format "Include %sf;" (substring (naam-uit-verf verf) 1))))
         (if (file-exists-p (directory-file-name "final/")) ()
             (make-directory "final"))
         (write-file (format "final/%sf.l" (eval 'cur))))
       (kill-buffer (format "%sf.l" (eval 'cur))))))

(defun final-test (naam) "De laatste Lego-test"
  (defconst te-beginnen t)
  (defconst proof-mode ())
  (kill-buffer "adm-buffer")
  (delete-frame adm-frame)
  (save-excursion
    (set-buffer "*lego*")
    (erase-buffer))
  (find-file-noselect (format "final/%sf.l" naam))
  (lego-simple-send (format "Cd %S;" "final"))
  (save-excursion
    (set-buffer (format "%sf.l" naam))
    (lego-simple-send "Forget begin;")
    (beginning-of-buffer)
    (end-of-buffer)
    (lego-send-region))
  (kill-buffer (format "%sf.l" naam))
  (check-result naam)
  (select-frame (next-frame werk-frame nil))
  (delete-frame)
  (delete-process lego-process-name)
  (kill-buffer "*lego*")
  (defconst current-proof-adm (new-adm))
  (select-frame werk-frame))

(defun check-result (naam) "Kijkt of file klopt"
  (while (not (or (is-OK naam) (error-saved)))(sit-for 1))
  (if (is-OK naam)
    (signal-ok)
    (signal-error)))

(defun is-OK (naam) "Kijkt of er misschien iets goed is."
    (save-excursion
      (set-buffer "*lego*")
      (beginning-of-buffer)
      (search-forward (format "\"%s\"  saved as" naam) nil t)))

(defun error-saved () "Kijkt of er misschien een fout zit"
  (save-excursion
    (set-buffer "*lego*")
    (beginning-of-buffer)
    (search-forward "cannot Save" nil t)))

(defun signal-ok () "Geeft OK boodschap"
  (x-popup-dialog (selected-frame) 
    (list "The proof is absolutely correct." (list "OK" t))))

(defun signal-error () "Geeft error message"
  (x-popup-dialog (selected-frame) 
    (list "There seems to be something wrong. This proof is NOT absolutely correct." (list "OK" t))))

;;;;;;;;;;;;;;;;;;;;;
;; einde mijn code ;;
;;;;;;;;;;;;;;;;;;;;;

(provide 'lego-mode)