implementation module graphics


import	StdInt, StdBool, StdReal, StdChar, StdList, StdFunc, StdEnum, StdArray, StdTuple, StdMisc, StdOrdList
import	deltaEventIO, deltaPicture, deltaFont, deltaDialog
import	board, language, systemsettings


::	Size			:==	(!Int,!Int)

grey				:==	RGB 0.5 0.5 0.5
darkgrey			:==	RGB 0.31 0.31 0.31
rbBoardGrey			:== RGB 0.75  0.75  0.75
rbLighterGrey		:== RGB 0.878 0.878 0.878
rbBoardRed3			:== RGB 1.0   0.5   0.5
rbBoardRed2			:== RGB 0.75  0.625 0.625
rbBoardBlue3		:== RGB 0.5   0.5   1.0
rbBoardBlue2		:== RGB 0.625 0.625 0.75
rbSquare			:== RGB 1.0   1.0   0.75 
rbDarkYellow		:== RGB 0.5   0.5   0.0

displaywidth		:==	250
displayheight		:==	130
boardwidth			:==	391
boardheight			:==	391
squarewidth			::	Int
squarewidth			=:	boardwidth/15
squareheight		::	Int
squareheight		=:	boardheight/15

alphabet			:==	"abcdefghijklmnopqrstuvwxyz"


/*	Mapping 'Amanda-space' to 'Scrabble-space' to 'Pixel-space':
	Amanda-space	: ((-1.0,1.0),(1.0,-1.0))
	Scrabble-space	: ((0.0,0.0), (14.0,14.0))
	Pixel-space		: ((0,0), (width,height))
*/

abs2rel :: !(!Int,!Int) -> (!Int,!Int)
abs2rel (x,y) = (x/squarewidth,y/squareheight)

instance toString ControlState
where
	toString :: !ControlState -> {#Char}
	toString (StringCS s)	= s
	toString _				= abort "toString not applied to (StringCS _).\n"

toStringCS :: String -> ControlState
toStringCS s = StringCS s


/*	The drawing operations.	*/

boardlook :: !Board !Size !SelectState ControlState -> [DrawFunction]
boardlook (hor,_) size=:(w,h) select cstate
	= [	SetPenColour	rbBoardGrey 
	  ,	FillRectangle	((0,0),size)
	  ,	SetPenColour	WhiteColour
	  ]
	  ++ 
	  [	DrawVectorAt (squarewidth*i+1,0) (0,h) \\ i<-is	]
	  ++
	  [	DrawVectorAt (0,squareheight*i+1) (w,0) \\ i<-is	]
	  ++
	  [	SetPenColour darkgrey : [ DrawVectorAt (squarewidth*i,1) (0,h-1) \\ i<-is ] ]
	  ++
	  [	DrawVectorAt (1,squareheight*i) (w-1,0) \\ i<-is ]
	  ++	map (drawsquare rbBoardBlue2) doubleletterpositions
	  ++	map (drawsquare rbBoardBlue3) tripleletterpositions
	  ++	map (drawsquare rbBoardRed2)  doublewordpositions
	  ++	map (drawsquare rbBoardRed3)  triplewordpositions
	  ++	drawcenter
	  ++
	  [	drawletter l (i,j) \\ i<-[0..14], j<-[0..14], l<-[(hor!!j)!!i] ]
	  ++
	  (	if (isAble select) (drawfocus True cstate) []	)
where
	is			= [0..15]
	
	drawcenter :: [DrawFunction]
	drawcenter
		= [drawsquare rbBoardGrey (7,7),SetPenColour grey,FillPolygon (absposition (7.5,7.5),shape)]
	where
		h		= (squarewidth-1)/2
		v		= (squareheight-1)/2
		shape	= [(0,0-v),(h,v),(0-h,v),(0-h,0-v),(h,0-v)]
	//	absposition maps a position in 'Scrabble-space' to a position in 'Pixel-space'.
		absposition :: !(!Real,!Real) -> (!Int,!Int)
		absposition (col,row)
			= (toInt (col*toReal squarewidth),toInt (row*toReal squareheight))
	
	drawsquare :: !Colour !(!Int,!Int) !Picture -> Picture
	drawsquare colour (col,row) picture
		# picture	= SetPenColour	colour			picture
		  picture	= FillRectangle	((l,t),(r,b))	picture
		= picture
	where
		l = col*squarewidth+2
		t = row*squareheight+2
		r = (col+1)*squarewidth
		b = (row+1)*squareheight
	
	isAble :: SelectState -> Bool
	isAble Able = True
	isAble _	= False

drawfocus :: !Bool !ControlState -> [DrawFunction]
drawfocus noterase (PairCS (IntCS x) (IntCS y))
	= [	SetPenColour	lefttopcolour
	  ,	MovePenTo		(l,b)
	  ,	LinePen			(0,0-(squareheight-1))
	  ,	LinePen			(squarewidth-1,0)
	  ,	SetPenColour    rightbotcolour
	  ,	LinePen			(0,squareheight-1)
	  ,	LinePen			(0-(squarewidth-1),0)
	  ]
where
	(col,row)						= abs2rel (x,y)
	l								= col*squarewidth+1
	b								= (row+1)*squareheight
	(lefttopcolour,rightbotcolour)	= if noterase (darkgrey, WhiteColour)
												  (WhiteColour, darkgrey)
drawfocus _ _
	= abort "drawfocus not applied to (PairCS (IntCS _) (IntCS _)).\n"


drawletter :: !Char !(!Int,!Int) !Picture -> Picture
drawletter l (i,j) picture
	| l==' '
	= picture
	# picture	= SetPenColour	rbSquare									picture
	  picture	= FillRectangle	((x+2,y+2),(x+squarewidth,y+squareheight))	picture
	  picture	= MovePenTo		(x+2,y+squareheight-1)						picture
	  picture	= SetPenColour	WhiteColour									picture
	  picture	= LinePenTo		(x+2,y+2)									picture
	  picture	= LinePenTo		(x+squarewidth-1,y+2)						picture
	  picture	= SetPenColour	YellowColour								picture
	  picture	= LinePenTo		(x+squarewidth-1,y+squareheight-1)			picture
	  picture	= LinePenTo		(x+2,y+squareheight-1)						picture
	  picture	= MovePenTo		(x+squarewidth/4,y+h-h/3)					picture
	  picture	= SetFont		letterfont									picture
	  picture	= SetPenColour	BlackColour									picture
	  picture	= DrawChar		(toUpper l)									picture
	  picture	= SetFont		smallfont									picture
	  picture	= SetPenColour	rbDarkYellow								picture
	  picture	= DrawStringAt	(x+squarewidth-2-plen,y+h-3) scoretext		picture
	| otherwise	= picture
where
	x			= i*squarewidth
	y			= j*squareheight
	h			= squareheight
	scoretext	= toString (lettervalue l)
	plen		= FontStringWidth scoretext smallfont


redrawboard :: !Board !(IOState t) -> IOState t
redrawboard board iostate
	= ChangeDialog scrabbleId [ChangeControlLook 100 (boardlook board (boardwidth,boardheight))] iostate

letterboxlook :: ![Char] SelectState ControlState -> [DrawFunction]
letterboxlook letters _ _
	= [	SetPenColour	rbBackground 
	  ,	FillRectangle	((0,0),(squarewidth*4,squareheight*15))
	  ]
	  ++
	  [	drawletter c (0,j) \\ (c,j)<-zip2 leftchars  js	]
	  ++
	  [	drawletter c (2,j) \\ (c,j)<-zip2 rightchars js	]
	  ++
	  [	SetFont			letterfont
	  ,	SetPenColour	BlackColour
	  ]
	  ++
	  [	drawcount c (1,j) \\ (c,j)<-zip2 leftcounts  js	]
	  ++
	  [	drawcount c (3,j) \\ (c,j)<-zip2 rightcounts js	]
where
	js							= [0..14]
	counts						= countletters alphabet (sort letters)
	(left,right)				= splitAt 15 counts
	(leftchars, leftcounts)		= unzip left
	(rightchars,rightcounts)	= unzip right
	
	drawcount :: !Int !(!Int,!Int) !Picture -> Picture
	drawcount count (i,j) picture
		= DrawStringAt (x+squarewidth/4,y+h-h/3) (toString count) picture
	where
		x = i*squarewidth
		y = j*squareheight
		h = squareheight
	
	countletters :: !String ![Char] -> [(Char,Int)]
	countletters chars letters
		| chars==""
		= []
		# c						= chars.[0]
		  (count,letters)			= countletter c letters
		= [(c,count):countletters (chars%(1,size chars-1)) letters]
	where
		countletter :: !Char ![Char] -> (Int,![Char])
		countletter c all_letters=:[letter:letters]
			| c<>letter
			= (0,all_letters)
			# (count,letters)	= countletter c letters
			| otherwise
			= (count+1,letters)
		countletter _ _
			= (0,[])

drawletterbox :: ![Char] !(IOState t) -> IOState t
drawletterbox letters iostate
	= ChangeDialog scrabbleId [ChangeControlLook 111 (letterboxlook letters)] iostate

drawplayer1letters :: ![Char] !(IOState t) -> IOState t
drawplayer1letters letters iostate
	= ChangeDialog scrabbleId [ChangeControlState 102 (StringCS (toString letters))] iostate

drawplayer2letters :: ![Char] !(IOState t) -> IOState t
drawplayer2letters letters iostate
	= ChangeDialog scrabbleId [ChangeControlState 104 (StringCS (toString letters))] iostate

playerletterslook :: !Size SelectState !ControlState -> [DrawFunction]
playerletterslook dim _ (StringCS ws)
	= [	SetPenColour	rbBackground
	  ,	FillRectangle	((0,0),dim)
	  :
	  [	drawletter ws.[i] (i,0) \\ i<-[0..size ws-1] ]
	  ]
playerletterslook _ _ _
	= abort "playerletterslook not applied to (StringCS _).\n"

drawplayer1score :: !Int !(IOState t) -> IOState t
drawplayer1score s iostate
	= ChangeDialog scrabbleId [ChangeDynamicText 106 (toString s)] iostate

drawplayer2score :: !Int !(IOState t) -> IOState t
drawplayer2score s iostate
	= ChangeDialog scrabbleId [ChangeDynamicText 108 (toString s)] iostate

drawcommunication :: ![String] !(IOState s) -> IOState s
drawcommunication text iostate
	= ChangeDialog scrabbleId [	ChangeControlState 110 (ListCS (map toStringCS text))
							  ,	ChangeControlLook  110 (displaylook (displaywidth,displayheight))
							  ] iostate

displaylook :: !Size SelectState !ControlState -> [DrawFunction]
displaylook size _ (ListCS text)
	= [drawtext (map toString text) size]
where
	drawtext :: ![String] !Size !Picture -> Picture
	drawtext text size=:(w,h) picture
		# picture = drawdisplay		size		picture
		  picture = SetFont			(font 12)	picture
		  picture = SetPenColour	RedColour	picture
		  picture = seq [ DrawStringAt (w/20,h*y/10) l \\ (y,l)<-zip2 [2,4..] text ] picture
		= picture

drawprogress :: !Player !Progress !Placing !(IOState t) -> IOState t
drawprogress player progress placing iostate
	= ChangeDialog scrabbleId [ChangeControlLook 110 (progresslook player progress placing (displaywidth,displayheight))] iostate
where
	progresslook :: !Player !Progress !Placing !Size SelectState ControlState -> [DrawFunction]
	progresslook player progress placing size _ _
		= [progresslook` player progress placing size]
	where
		progresslook` :: !Player !Progress !Placing !Size !Picture -> Picture
		progresslook` player (Letter letter _) placing size=:(w,h) picture
			# picture		= drawdisplay	size							picture
			  picture		= SetFont		thefont							picture
			  picture		= SetPenColour	grey							picture
			  picture		= DrawStringAt	letterspos alphabet				picture
			  picture		= SetPenColour	GreenColour						picture
			  picture		= DrawStringAt	letterspos alphabet_l_incl		picture
			  picture		= SetPenColour	RedColour						picture
			  picture		= DrawStringAt	letterspos alphabet_l_excl		picture
			  picture		= SetPenColour	GreenColour						picture
			  picture		= DrawStringAt	(tekstindent,toInt (0.15*h`)) (toString player+++determines_new_word) picture
			  picture		= DrawStringAt	(foundpos,toInt (0.60*h`)) found_upto_now picture
			  picture		= MovePen		(10,0)							picture
			  picture		= DrawString	placing.word					picture
			  picture		= DrawStringAt	(atpos,toInt (0.75*h`)) at_pos	picture
			  picture		= MovePen		(10,0)							picture
			  picture		= DrawString	placingtext						picture
			  picture		= DrawStringAt	(scorepos,toInt (0.90*h`)) score_upto_now picture
			  picture		= MovePen		(10,0)							picture
			  picture		= DrawString	(toString placing.score)		picture
			= picture
		where
			(x,y)			= placing.pos
            thefont			= font 12
			foundlength		= FontStringWidth found_upto_now thefont
            rtabstop		= tekstindent+foundlength
			foundpos		= tekstindent
			atpos			= rtabstop - FontStringWidth at_pos thefont
            scorepos		= rtabstop - FontStringWidth score_upto_now thefont
			
			w`				= toReal w
			h`				= toReal h
			letterspos		= (toInt (0.05*w`),toInt (0.35*h`))
			tekstindent		=  toInt (0.05*w`)
			
			alphabet_l_excl	= if (letter=='a') "" (alphabet%(0,l_index-1))
			alphabet_l_incl	= alphabet%(0,l_index)
			l_index			= toInt letter-a_index
			a_index			= toInt 'a'
			
			placingtext		= toString (x,y)+++" "+++toString placing.dir
		progresslook` player (Finish _) _ size=:(w,h) picture
			# picture		= SetPenColour	grey			picture
			  picture		= FillRectangle	((0,0),size)	picture
			  picture		= SetFont		(font 12)		picture
			  picture		= SetPenColour	RedColour		picture
			  picture		= DrawStringAt	(toInt (0.05*w`),toInt (0.95*h`)) (toString player+++determined_new_word) picture
			= picture
		where
			w`				= toReal w
			h`				= toReal h

DrawStringAt :: !Point !String !Picture -> Picture
DrawStringAt pos text picture
	# picture	= MovePenTo pos picture
	  picture	= DrawString text picture
	= picture

DrawVectorAt :: !Point !Vector !Picture -> Picture
DrawVectorAt pos v picture
	# picture	= MovePenTo pos picture
	  picture	= LinePen v picture
	= picture

drawdisplay :: !Size !Picture -> Picture
drawdisplay size=:(w,h) picture
	# picture	= SetBackColour	BlackColour		picture
	  picture	= EraseRectangle ((0,0),size)	picture
	  picture	= SetPenColour	grey			picture
	  picture	= MovePenTo		(-1,h-1)		picture
	  picture	= LinePenTo		(-1,-1)			picture
	  picture	= LinePenTo		(w,-1)			picture
	  picture	= MovePenTo		(-2,h)			picture
	  picture	= LinePenTo		(-2,-2)			picture
	  picture	= LinePenTo		(w+1,-2)		picture
	  picture	= SetPenColour	WhiteColour		picture
	  picture	= MovePenTo		(-1,h)			picture
	  picture	= LinePenTo		(w,h)			picture
	  picture	= LinePenTo		(w,-2)			picture
	  picture	= SetPenColour	rbLighterGrey	picture
	  picture	= MovePenTo		(-2,h+1)		picture
	  picture	= LinePenTo		(w+1,h+1)		picture
	  picture	= LinePenTo		(w+1,-3)		picture
	= picture
