
Module semantics Import nFunc syntax;

(* This module defines single-sorted models and structures. *)

(* --------------------------------------------------------------------------------
   Given a signature, define a structure as
    -  a carrier set A
    -  a valuation function intFunc which assigns to every function symbol a function
    -  a valuation function intPred which assigns to every predicate symbol a predicate

   So intFunc and intPred gives us a interpretation of all symbols of the signature.

   Given Axioms being a predicate over structures, define a model as
    - a structure s
    - a proof of (Axioms s)

   We use sigma-types because inductive types would be slower to check.
*)

[     Structure [sig:Signature] : TYPE
          = <A:Set> ({c:FuncSymb sig} nFunc A (FuncArity c)) #
                    ({p:PredSymb sig} nPred A (PredArity p))
]
[     Axioms [sig:Signature] : TYPE
          = (Structure sig) -> Prop
]
[     Model [sig|Signature] [ax:Axioms sig] : TYPE
          = <str:Structure sig> ax str
];

(* --------------------------------------------------------------------------------
   Let M be a model over a signature and some axioms. Define terms to
   extract the various components of M.
*)

[sig | Signature] [ax | Axioms sig] [M : Model ax];

[     structure : Structure sig
          = M.1
]
[     axioms : ax structure
          = M.2
]
[     car : Set
          = structure.1
]
[     obj : SET
          = el car
];

[     intFunc : {c:FuncSymb sig} nFunc car (FuncArity c)
          = structure.2.1
]
[     intCons : {c:FuncSymb sig} arrow obj obj (FuncArity c)
          = [c:FuncSymb sig] (intFunc c).apn
]
[     intPred : {p:PredSymb sig} nPred car (PredArity p)
          = structure.2.2
];

(* --------------------------------------------------------------------------------
   So we have
    el  : Set -> SET
    car : (Model ax) -> Set
    obj : (Model ax) -> SET
  It is convenient to have abbreviations of the notion of functions and
  predicates over the carrier of a model.
*)

[     UFunMdl : SET
          = Fun car car
]
[     BFunMdl : SET
          = Fun2 car car car
]
[     PredMdl : SET
          = Pred car
]
[     BRelMdl : SET
          = Rel car car
];

(* --------------------------------------------------------------------------------
   Define some terms to construct a structure and model.
*)

[     Structure_intro [A|Set] [intFunc:{c:FuncSymb sig} nFunc A (FuncArity c)]
                              [intPred:{p:PredSymb sig} nPred A (PredArity p)]
          : Structure sig
          = (A, intFunc, intPred : Structure sig)
]
[     Axioms_intro [z : {A|Set}({c:FuncSymb sig} nFunc A (FuncArity c)) -> 
                               ({p:PredSymb sig} nPred A (PredArity p)) -> Prop]
          : Axioms sig
          = [str:Structure sig] z str.2.1 str.2.2
]
[     Model_intro [A|Set] [intCons:{c:FuncSymb sig} nFunc A (FuncArity c)]
                          [intPred:{p:PredSymb sig} nPred A (PredArity p)]
                          [axioms : ax (Structure_intro intCons intPred)]
          : Model ax
          = (Structure_intro intCons intPred, axioms : Model ax)
];

(* ================================================================================
   An assignment is a valuation of some variables into the carrier of the model.
   We implement assignments by non-empty lists.
*)

[     Assignment : Set
          = neList car
];

(* --------------------------------------------------------------------------------
   Define the interpretation of a term with respect to an
   assignment. First deal with the case of n-tuples of terms.
*)

Goal int_n : {n|nat} Assignment.el -> (Terms sig n).el -> (Product car n).el;
  intros m rho;
  Refine terms_elim ? [n:nat](Terms sig n).el\(Product car n).el;
  intros n; Refine ne_nth rho n;
  intros c t ih; Refine c.intFunc.app ih;
  Refine star;
  Refine nat_elim [n|nat](terms sig OneN)->(terms sig n)->(el (Product car OneN))->
       (el (Product car n))->el (Product car (succ n));
  intros t1 t0 ih1 ih0; Refine ih1;
  intros n IH t1 tn ih1 ihn; Refine tuple ih1 ihn;
Save;

Goal {n|nat} extensional2 (int_n|n);
  Intros m rho rho' _ v w _;
  Qrepl H1;
  Refine terms_elim sig [n:nat][w:terms sig n] Eq (int_n rho w) (int_n rho' w);
  intros x; Refine ne_nth_exten H ?.Eq_refl;
  intros c t ih;
    Equiv Eq (c.intFunc.app (int_n rho t)) (c.intFunc.app (int_n rho' t));
    Refine extp ? ih;
  Refine Eq_refl;
  Refine nat_elim [n|nat]{x1:terms sig OneN}{x2:terms sig n}
        (Eq (int_n rho x1) (int_n rho' x1))->
        (Eq (int_n rho x2) (int_n rho' x2))->
        Eq (int_n rho (tcons sig x1 x2)) (int_n rho' (tcons sig x1 x2));
  intros; Refine H2;
  intros; Refine pair; Immed;
Save int_n_exten;

[     Int_n [n:nat] : Fun2 (Assignment) (Terms sig n) (Product car n)
          = Fun2_intro (int_n|n) (int_n_exten|n)
];

(* Next, define the interpretation of a term with respect to an assignment. *)

[     int : Assignment.el -> (Term sig).el -> obj
          = [rho:el Assignment] [t:el (Term sig)] (int_n rho t)
];

Goal {t,t'|(Term sig).el}{rho,rho'|Assignment.el}
     (Eq (int rho t) (int rho' t')) -> Eq (int_n rho t) (int_n rho' t');
  intros;
  Refine H;
Save int_int_n;

Goal {t,t'|(Term sig).el}{rho,rho'|Assignment.el}
     (Eq (int_n rho t) (int_n rho' t')) -> Eq (int rho t) (int rho' t');
  intros; Immed;
Save int_n_int;

Goal extensional2 int;
  Intros ______;
  Refine int_n_int;
  Refine int_n_exten H H1;
Save int_exten;

[     Int : Fun2 Assignment (Term sig) car
          = Fun2_intro int int_exten
];

(* ================================================================================
   Prove the Compatibility lemma:

     rho_1 = rho_2  ->  [t]_rho_1 = [t]_rho_2
*)

Goal {rho,rho':el Assignment} (Eq rho rho') ->
     {t:el (Term sig)} Eq (int rho t) (int rho' t);
  intros;
  Refine int_exten H ?.Eq_refl;
Save CompAss;

(* -------------------------------------------------------------------------
   Define the collapsing procedure.
*)

(* Prove:    x = y  ->  x[y:=t] = t *)

Goal {x,y|el Nat} (Eq x y) -> {t:el (Term sig)} Eq (Subst_n (TFV sig x) y t) t;
  intros;
  Refine if_true H;
Save Subst_eq;

(* Prove:    x != y  ->  x[y:=t] = x *)

Goal {x,y|el Nat} ~(Eq x y) -> {t:el sig.Term} Eq (Subst_n (TFV sig x) y t) (TFV sig x);
  intros;
  Refine if_false H;
Save Subst_neq;

(* Prove:    [VAR x] = [u]  -> [t] = [t[x:=u]] *)

Goal {x:el Nat}{t,u:el sig.Term}{rho:el Assignment}
     (Eq (int rho (sig.TFV x)) (int rho u)) ->
     (Eq (int rho t) (int rho (Subst t x u)));
  intros;
  Refine int_n_int;
  Refine terms_elim ?
     ([n:nat][t:terms sig n] Eq (int_n rho t) (int_n rho (Subst_n t x u)));
  intros z;
    Refine int_int_n;
    orE Nat_discr z x;
    intros;
      Qrepl Subst_eq H1 u; Qrepl H1; Refine H;
    intros;
      Qrepl Subst_neq H1 u; Refine Eq_refl;
  intros c t' ih; Refine extp ? ih;
  Refine Eq_refl;
  Refine nat_elim [n|nat]{x1:terms sig OneN}{x2:terms sig n}
        (Eq (int_n rho x1) (int_n rho (Subst_n x1 x u)))->
        (Eq (int_n rho x2) (int_n rho (Subst_n x2 x u)))->
        Eq (int_n rho (tcons sig x1 x2))
           (int_n rho (Subst_n (tcons sig x1 x2) x u));
  intros __ H1 H2; Immed;
  intros ___ H1 H2; Refine pair; Immed;
Save SubstitutionLemma;

(* Prove:    [VAR x] = [u]  ->  [t[x:=u]] = [t'[x:=u]]  -> [t] = [t'] *)

Goal {t,t'|el sig.Term}{rho|el Assignment}
     {x:el Nat}{u:el sig.Term}
     (Eq (int rho (TFV sig x)) (int rho u)) ->
     (Eq (int rho (Subst t x u)) (int rho (Subst t' x u))) ->
     (Eq (int rho t) (int rho t'));
  intros;
  Refine Eq_trans (int rho (Subst t x u));
    Refine SubstitutionLemma ???? H;
  Refine Eq_trans (int rho (Subst t' x u));
    Refine H1;
  Refine Eq_sym;
  Refine SubstitutionLemma ???? H;
Save Unify;

Discharge sig;
